# -*- coding: utf-8 -*-
import base64
import binascii
from collections import OrderedDict
import json
import random
import re
import string
from typing import (
    Optional,
    Union,
)
import zlib

from cryptography.hazmat.backends import default_backend
from cryptography.hazmat.primitives.ciphers import (
    algorithms,
    Cipher,
    modes,
)
from django.conf import settings
from django.utils.encoding import smart_bytes
from passport.backend.core.builders.blackbox import (
    BaseBlackboxError,
    BLACKBOX_OAUTH_VALID_STATUS,
)
from passport.backend.oauth.core.common.blackbox import get_blackbox
from passport.backend.oauth.core.common.utils import (
    bytes_to_int,
    from_base64_url,
    mask_string,
)
from passport.backend.oauth.core.db.client import Client
from passport.backend.oauth.core.db.eav import EntityNotFoundError
from passport.backend.oauth.core.db.eav.attributes import attr_by_name
from passport.backend.oauth.core.db.eav.types import DB_NULL
from passport.backend.oauth.core.db.token import (
    get_access_token_from_refresh_token,
    Token,
)
from passport.backend.utils.time import unixtime_to_datetime


ATTRS_TO_MASK = {
    'token_attributes': [
        attr_by_name('token', attr_name)[0]
        for attr_name in ('access_token', 'alias')
    ],
    'client_attributes': [
        attr_by_name('client', attr_name)[0]
        for attr_name in ('secret', 'old_secret')
    ],
}


def get_environment_by_id(env_id):
    mapping = {
        0: 'localhost:production',
        1: 'intranet:production',
        2: 'localhost:testing',
        3: 'intranet:testing',
        4: 'stress:stress',
    }
    try:
        return mapping[env_id]
    except KeyError:
        return 'Unknown environment'


def get_resolve_blackbox():
    """Возвращает билдер ЧЯ для работы с аккаунтами из текущей БД"""
    return get_blackbox(
        url=settings.RESOLVE_UID_BLACKBOX_URL,
        tvm_dst_alias=settings.RESOLVE_UID_BLACKBOX_ALIAS,
    )


def _get_client_data(client_id):
    """На входе - именно id, а не display_id"""
    try:
        client = Client.by_id(client_id, allow_deleted=True)
    except EntityNotFoundError:
        return
    else:
        return {
            'id': client_id,
            'display_id': client.display_id,
            'default_title': client.default_title,
            'is_deleted': client.is_deleted,
            'is_yandex': client.is_yandex,
        }


def get_offline_info_for_refresh_token(refresh_token: string) -> tuple[bool, dict, str]:
    access_token = get_access_token_from_refresh_token(refresh_token)
    if not access_token:
        is_valid = False
        data = None
    else:
        is_valid, data = get_offline_info_for_token(access_token)
        if is_valid and data:
            data.update(access_token=mask_string(access_token, len(access_token) // 2))

    return is_valid, data, access_token


def get_environment_for_token(access_token: string) -> tuple[bool, dict]:
    """
    Проверяем что окружение в котором сформирован токен соответствует текущему окружению
    """
    is_valid = True
    try:
        # Токен приходят из формы т.е прийти может что-угодно
        env = int(access_token[1])
        if not bool(re.search('y[0-4]_[-_A-Za-z\\d]{50}', access_token)):
            raise ValueError('Token does not satisfy pattern')
    except (ValueError, IndexError):
        is_valid = False
        data = OrderedDict([
            ('token_environment', 'Invalid token'),
            ('current_environment', get_environment_by_id(settings.ENV_ID)),
        ])
    else:
        if settings.ENV_ID != env:
            is_valid = False
        data = OrderedDict([
            ('token_environment', get_environment_by_id(env)),
            ('current_environment', get_environment_by_id(settings.ENV_ID)),
        ])
    return is_valid, data


def get_offline_info_for_token(access_token: string) -> tuple[bool, dict]:
    is_valid = True
    token_length = len(access_token)
    data = OrderedDict([
        ('length', token_length),
        ('version', None),
        ('shard', None),
        ('user', {}),
        ('client', {}),
        ('crc', None),
        ('environment', {}),
        ('environment_matches', None)
    ])
    regular_token_length = settings.REGULAR_TOKEN_LENGTH
    if not settings.TOKEN_41_BYTE_FORMAT:
        regular_token_length = settings.DEPRECATED_TOKEN_LENGTH['39']
    # В логах токены маскируются звёздочками. Но звёздочка - запрещённый для base64 символ и
    # обрежется при декодировании. Поэтому заменим её допустимым символом.
    access_token = access_token.replace('*', 'A')

    if token_length == settings.DEPRECATED_TOKEN_LENGTH['32']:
        data.update(
            version='1 (16 bytes in hex)',
            shard=1,
        )
    elif token_length == settings.DEPRECATED_TOKEN_LENGTH['34']:
        try:
            bytes_ = from_base64_url(smart_bytes(access_token))
        except binascii.Error:
            data.update(
                version='invalid',
            )
            is_valid = False
        else:
            data.update(
                version='2 (25 bytes in base64_url)',
                shard=bytes_to_int(bytes_[:1]),
                user={
                    'uid': bytes_to_int(bytes_[1:5]),
                },
                client={
                    'id': bytes_to_int(bytes_[5:9]),
                },
            )
    elif token_length == settings.DEPRECATED_TOKEN_LENGTH['39']:
        try:
            bytes_ = from_base64_url(smart_bytes(access_token))
        except binascii.Error:
            data.update(
                version='invalid',
            )
            is_valid = False
        else:
            data.update(
                version='3 (29 bytes in base64_url)',
                shard=bytes_to_int(bytes_[:1]),
                user={
                    'uid': bytes_to_int(bytes_[1:9]),
                },
                client={
                    'id': bytes_to_int(bytes_[9:13]),
                },
            )
    # todo and not access_token.startswith('1.') удалить перед продом нужно для обратной совместимости
    # todo считается что в прод версии токены > 39 stateless
    elif token_length == settings.REGULAR_TOKEN_LENGTH and not access_token.startswith('1.'):
        try:
            # token example: y3_AQAAAAAAAAABAAAAAQAAAAAAAAABbEPKzPj7RUyCQcNHXpnSeg_SFs4
            # y - префикс: 3 - id окружения: _ - разделитель: остальное base64
            # base64 [3:]
            bytes_ = from_base64_url(smart_bytes(access_token[3:]))
        except binascii.Error:
            data.update(
                version='invalid',
            )
            is_valid = False
        else:
            token_crc = bytes_to_int(bytes_[-4:])
            body_crc = zlib.crc32(bytes_[:37])
            environment_matches, environment_data = get_environment_for_token(access_token)
            environment_data.environment_matches = environment_matches
            data.update(
                version='5 (41 bytes in base64_url with prefix)',
                shard=bytes_to_int(bytes_[:1]),
                user={
                    'uid': bytes_to_int(bytes_[1:9]),
                },
                client={
                    'id': bytes_to_int(bytes_[9:13]),
                },
                crc='matches' if token_crc == body_crc else 'not matches',
                environment=environment_data,
            )
    elif token_length > regular_token_length and access_token.startswith('1.'):
        _, uid, client_id, expire_ts, create_ts, _ = access_token.split('.', 5)
        data.update(
            version='4 (stateless v1)',
            token_id=int(create_ts + uid),
            created=str(unixtime_to_datetime(int(create_ts) / 1000)),
            expires=str(unixtime_to_datetime(int(expire_ts))),
            user={
                'uid': int(uid),
            },
            client={
                'id': int(client_id),
            },
        )
    else:
        data.update(
            version='invalid',
        )
        is_valid = False

    client_id = data.get('client', {}).get('id')
    if client_id:
        data['client'] = _get_client_data(client_id)

    return is_valid, data


def get_offline_info_for_alias(uid, token_alias):
    data = None
    is_valid = True
    alias_length = len(token_alias.replace(' ', ''))
    if alias_length != settings.TOKEN_ALIAS_LENGTH:
        data = 'Invalid length (%s instead of %s)' % (alias_length, settings.TOKEN_ALIAS_LENGTH)
        is_valid = False

    return is_valid, data


def get_db_info(token: Optional[Token]) -> tuple[bool, Union[OrderedDict, dict]]:
    if token is None:
        return False, {'error': 'Token not found'}
    data = OrderedDict([
        ('id', token.id),
        ('device_id', token.device_id if token.device_id != DB_NULL else None),
        ('device_name', token.device_name),
        ('scopes', ', '.join(s.keyword for s in token.scopes)),
        ('x_token_id', token.x_token_id),
        ('created', str(token.created)),
        ('issued', str(token.issued)),
        ('expires', str(token.expires) if token.expires else None),
        ('is_ttl_refreshable', str(token.is_refreshable) if token.expires else None),
        ('token', token.masked_body),
        ('type', 'app_password' if token.is_app_password else 'token'),
    ])
    data['user'] = {'uid': token.uid} if token.uid else None
    data['client'] = _get_client_data(token.client_id)

    is_valid = bool(data.get('client') and not data['client'].get('is_deleted'))
    return is_valid, data


def get_token_info_from_blackbox(access_token: str) -> tuple[bool, Union[str, dict]]:
    try:
        blackbox_raw_info = get_resolve_blackbox().oauth(
            oauth_token=access_token,
            ip='127.0.0.1',
            dbfields=settings.BLACKBOX_DBFIELDS,
            attributes=settings.BLACKBOX_ATTRIBUTES,
            need_token_attributes=True,
            need_client_attributes=True,
            need_display_name=False,
            get_login_id=True,
        )
    except BaseBlackboxError as e:
        return False, str(e)

    is_valid = blackbox_raw_info['status'] == BLACKBOX_OAUTH_VALID_STATUS
    if is_valid:
        for field, attrs_to_mask in ATTRS_TO_MASK.items():
            for attr in blackbox_raw_info['oauth'][field]:
                if int(attr) in attrs_to_mask:
                    blackbox_raw_info['oauth'][field][attr] = '***'

    blackbox_raw_info = json.dumps(blackbox_raw_info, indent=2)
    return is_valid, blackbox_raw_info


class DecryptFailedError(Exception):
    pass


class AmCredentialsManager(object):
    def __init__(self, b64_secret=None):
        super(AmCredentialsManager, self).__init__()
        self._secret = base64.b64decode(
            (b64_secret or settings.AM_CREDS_ENCRYPTION_SECRET).encode(),
        )
        self._iv = b'\0' * 16

    def _make_random_text(self, length):
        return ''.join(
            random.choice(string.ascii_uppercase + string.digits)
            for _ in range(length)
        )

    def _align(self, text):
        suffix_len = 15 - len(text) % 16
        return '%s^%s' % (text, self._make_random_text(suffix_len))

    def encrypt(self, text):
        encryptor = Cipher(
            algorithms.AES(self._secret),
            modes.CFB(self._iv),
            backend=default_backend(),
        ).encryptor()
        return base64.b64encode(
            encryptor.update(
                self._align(text).encode(),
            ) + encryptor.finalize(),
        ).decode()

    def decrypt(self, text):
        decryptor = Cipher(
            algorithms.AES(self._secret),
            modes.CFB(self._iv),
            backend=default_backend(),
        ).decryptor()
        try:
            decrypted = (
                decryptor.update(
                    base64.b64decode(text.encode()),
                ) + decryptor.finalize()
            ).decode()
            text, padding = decrypted.rsplit('^', 1)
            return text, padding
        except (binascii.Error, UnicodeDecodeError) as e:
            raise DecryptFailedError(e) from e
