# -*- coding: utf-8 -*-
from datetime import timedelta
from functools import wraps
import json
from math import ceil

from django.conf import settings
from django.core.paginator import Paginator
from django.http import (
    HttpResponse,
    HttpResponseBadRequest,
    HttpResponseForbidden,
    HttpResponseNotFound,
    HttpResponseRedirect,
)
from django.shortcuts import render
from django.urls import reverse
from django_yauth.decorators import yalogin_required
import humanize
from passport.backend.core.builders.blackbox.utils import get_attribute
from passport.backend.core.builders.passport import PassportNoSubscriptionError
from passport.backend.oauth.admin.admin.forms import (
    ClientForm,
    EditClientForm,
    EditUserForm,
    EncryptAmCredsForm,
    FilterForm,
    FindUserForm,
    ScopeForm,
    SearchForm,
    VerifyTokenForm,
)
from passport.backend.oauth.admin.admin.utils import (
    AmCredentialsManager,
    DecryptFailedError,
    get_db_info,
    get_offline_info_for_alias,
    get_offline_info_for_refresh_token,
    get_offline_info_for_token,
    get_resolve_blackbox,
    get_token_info_from_blackbox,
)
from passport.backend.oauth.core.common.blackbox import (
    get_revoke_time_from_bb_response,
    REVOKER_APP_PASSWORDS,
    REVOKER_TOKENS,
    REVOKER_WEB_SESSIONS,
)
from passport.backend.oauth.core.common.blackbox_helpers import (
    get_login_or_uid_by_uid,
    try_convert_uids_to_logins,
)
from passport.backend.oauth.core.common.passport import get_passport
from passport.backend.oauth.core.common.utils import (
    int_or_default,
    mask_string,
)
from passport.backend.oauth.core.db.acl import (
    is_action_enabled,
    services_to_admin,
)
from passport.backend.oauth.core.db.client import (
    APPROVAL_STATUS_NAMES,
    ApprovalStatus,
    Client,
)
from passport.backend.oauth.core.db.eav import (
    EntityNotFoundError,
    UPDATE,
)
from passport.backend.oauth.core.db.scope import Scope
from passport.backend.oauth.core.db.token import (
    check_if_is_refreshable,
    get_ttl_by_scopes,
    list_tokens_with_clients_by_user,
    Token,
)
from passport.backend.oauth.core.logs.historydb import to_event_log
from passport.backend.oauth.core.logs.statbox import to_statbox
from passport.backend.utils.time import unixtime_to_datetime


DT_NEVER_REVOKED = unixtime_to_datetime(1)

GRANT_ENTER_ADMIN = 'enter_admin'
GRANT_VIEW_CLIENTS = 'view_clients'
GRANT_EDIT_CLIENTS = 'edit_clients'
GRANT_POSTMODERATE_CLIENTS = 'postmoderate_clients'
GRANT_VIEW_USERS = 'view_users'
GRANT_EDIT_USERS = 'edit_users'
GRANT_CREATE_SCOPES = 'create_scopes'
GRANT_VIEW_TOKENS = 'view_tokens'
GRANT_ENCRYPT_AM_CREDS = 'encrypt_am_creds'


def services_from_client(client):
    return set(s.service_name for s in client.scopes)


def services_from_user(uid, requested_service=None):
    user_services = services_to_admin(uid)
    if requested_service and requested_service in user_services:
        # пользователь сам захотел увидеть только приложения данного сервиса
        user_services = {requested_service}
    return user_services


def can_admin(uid, client):
    client_services = services_from_client(client)
    user_services = services_from_user(uid)

    if not client_services:
        return True  # приложение без скоупов - надо же его кому-то показать
    elif user_services.intersection(client_services):
        return True  # приложение, имеющее модерируемый пользователем скоуп
    else:
        return False


def grants_required(grants):
    def decorator(function):
        @wraps(function)
        def decorated(request, *args, **kwargs):
            admin_uid = request.yauser.uid
            for grant in grants:
                if not is_action_enabled(admin_uid, grant):
                    return HttpResponseForbidden(
                        'У вас (uid=%d) нет гранта %s для доступа к данному разделу' % (admin_uid, grant),
                    )
            return function(request, *args, **kwargs)
        return decorated
    return decorator


def _client_pager(request, page, search_data, filter_data, **base_index_kwargs):
    # Загружает из БД только тех клиентов, которые нужны для указанной страницы пейджера
    search_type = search_data.get('search_type')
    search = search_data.get('search')
    service = search_data.get('service')
    requires_approval = search_data.get('requires_approval')
    yandex_only = search_data.get('yandex_only')

    index_kwargs = base_index_kwargs.copy()
    index_kwargs.update(
        services=services_from_user(uid=request.yauser.uid, requested_service=service),
    )
    if requires_approval:
        index_kwargs.update(
            approval_status=ApprovalStatus.Pending,
        )
    if yandex_only:
        index_kwargs.update(
            is_yandex=True,
        )
    if search and search_type == 'creator_uid':
        index_kwargs.update(uid=int(search))

    offset = limit = 0

    if search and search_type == 'client_id':
        # Здесь нет фильтрации по остальным полям, но для одного приложения это некритично
        client = Client.by_display_id(search, allow_deleted=True)
        selected_clients = [client] if client else []
        client_count = len(selected_clients)
    elif search and search_type == 'client_id_internal':
        # Здесь нет фильтрации по остальным полям, но для одного приложения это некритично
        try:
            client = Client.by_id(int(search), allow_deleted=True)
            selected_clients = [client]
        except EntityNotFoundError:
            selected_clients = []
        client_count = len(selected_clients)
    else:
        client_count = Client.count_by_index('params', **index_kwargs)
        if not client_count:
            selected_clients = []
        else:
            max_pages = int(ceil(float(client_count) / settings.CLIENTS_PER_PAGE))
            page = min(max(int_or_default(page, 1), 1), max_pages)

            # Сначала выгружаем из БД приложения для текущей страницы пейджера
            offset = (page - 1) * settings.CLIENTS_PER_PAGE
            limit = settings.CLIENTS_PER_PAGE

            selected_clients = Client.by_index(
                'params',
                allow_deleted=True,
                limit=limit,
                offset=offset,
                **index_kwargs
            )

    # Затем отфильтровываем приложения в рамках текущей страницы пейджера
    # TODO: когда-нибудь унести эту фильтрацию на клиент-сайд
    filtered_clients = []
    for client in selected_clients:
        is_ok = True
        if filter_data['scope'] and not any(filter_data['scope'] == s.keyword for s in client.scopes):
            is_ok = False
        elif filter_data['title'] not in client.default_title:
            is_ok = False
        elif filter_data['only_with_nonpublic_granttypes'] and not client.allow_nonpublic_granttypes:
            is_ok = False

        filtered_clients.append(client if is_ok else None)

    left = client_count - offset - limit
    all_clients = [None] * offset + filtered_clients + [None] * left

    paginator = Paginator(all_clients, settings.CLIENTS_PER_PAGE)
    return paginator.page(page)


def log_admin_action(admin_uid, action, **kwargs):
    to_statbox(
        mode='admin',
        action=action,
        admin_uid=admin_uid,
        **kwargs
    )


@yalogin_required
@grants_required([GRANT_ENTER_ADMIN])
def index(request, env_name):
    return render(request, 'index.html', {'env_name': env_name})


@yalogin_required
@grants_required([GRANT_ENTER_ADMIN, GRANT_VIEW_CLIENTS])
def view_clients(request, env_name):
    admin_uid = request.yauser.uid
    if request.method == 'GET':
        search_form = SearchForm(services=services_to_admin(admin_uid), data=request.GET)
        if not search_form.is_valid():
            return HttpResponseBadRequest('Invalid search request')

        filter_form = FilterForm(data=request.GET, scopes=Scope.list(show_hidden=True))
        if not filter_form.is_valid():
            return HttpResponseBadRequest('Invalid search request')

        if 'nosearch' in request.GET:
            search_skipped = True
            clients, forms = [], []
        else:
            search_skipped = False
            clients = _client_pager(
                request=request,
                page=request.GET.get('page', 1),
                search_data=search_form.cleaned_data,
                filter_data=filter_form.cleaned_data,
            )
            forms = [
                ClientForm(initial={'client_id': client_.display_id}) if client_ else None
                for client_ in clients
            ]
        return render(
            request,
            'view_clients.html',
            {
                'env_name': env_name,
                'page': clients,
                'clients': zip(clients, forms),
                'search_form': search_form,
                'filter_form': filter_form,
                'search_skipped': search_skipped,
            },
        )
    else:
        form = ClientForm(request.POST)
        if form.is_valid():
            client_id = form.cleaned_data['client_id']
            client = Client.by_display_id(client_id)
            if not client:
                return HttpResponseNotFound('Client %s not found' % client_id)
            elif not can_admin(admin_uid, client):
                return HttpResponseForbidden('Forbidden')
            elif not (
                is_action_enabled(admin_uid, GRANT_POSTMODERATE_CLIENTS) or
                is_action_enabled(admin_uid, GRANT_EDIT_CLIENTS)
            ):
                return HttpResponseForbidden(
                    'У вас (uid=%d) нет гранта %s для выполнения данной операции' % (admin_uid, GRANT_POSTMODERATE_CLIENTS),
                )

            with UPDATE(client):
                if 'block' in request.POST:
                    client.block()
                    to_event_log(
                        action='block',
                        target='client',
                        client_id=client_id,
                        scopes=','.join(s.keyword for s in client.scopes),
                    )
                    log_admin_action(admin_uid, action='block_client', client_id=client_id)
                elif 'unblock' in request.POST:
                    client.unblock()
                    log_admin_action(admin_uid, action='unblock_client', client_id=client_id)
                elif 'approve' in request.POST:
                    client.approve()
                    log_admin_action(admin_uid, action='approve_client', client_id=client_id)
                elif 'reject' in request.POST:
                    client.reject_approval()
                    log_admin_action(admin_uid, action='reject_client', client_id=client_id)

                return HttpResponseRedirect(
                    '%s?search_type=client_id&search=%s' % (
                        reverse('view_clients', kwargs={'env_name': env_name}),
                        client_id,
                    ),
                )

        return HttpResponseBadRequest('Something bad has happened')


@yalogin_required
@grants_required([GRANT_ENTER_ADMIN, GRANT_VIEW_CLIENTS])
def edit_client(request, env_name, client_id):
    admin_uid = request.yauser.uid
    client = Client.by_display_id(client_id, allow_deleted=True)
    if not client:
        return HttpResponseNotFound('Client not found')
    if not can_admin(admin_uid, client):
        return HttpResponseForbidden('You have no permissions to edit this client')
    if request.method == 'POST':
        if not is_action_enabled(admin_uid, GRANT_EDIT_CLIENTS):
            return HttpResponseForbidden(
                'У вас (uid=%d) нет гранта %s для выполнения данной операции' % (admin_uid, GRANT_EDIT_CLIENTS),
            )

        form = EditClientForm(request.POST)
        if form.is_valid():
            with UPDATE(client):
                client.change_creator(uid=int(form.cleaned_data['creator']))
                client.is_yandex = form.cleaned_data['is_yandex']
                for field_name in (
                    'title_ru',
                    'title_uk',
                    'title_en',
                    'title_tr',
                    'description_ru',
                    'description_uk',
                    'description_en',
                    'description_tr',
                    'telegram_bot_name',
                    'ios_default_app_id',
                    'ios_extra_app_ids',
                    'ios_appstore_url',
                    'android_default_package_name',
                    'android_extra_package_names',
                    'android_cert_fingerprints',
                    'android_appstore_url',
                    'turboapp_base_url',
                    'allow_nonpublic_granttypes',
                    'owner_uids',
                    'owner_groups',
                ):
                    setattr(client, field_name, form.cleaned_data[field_name])

            log_admin_action(admin_uid, action='edit_client', client_id=client_id)  # TODO: add details

            return HttpResponseRedirect(
                '%s?search_type=client_id&search=%s' % (
                    reverse('view_clients', kwargs={'env_name': env_name}),
                    client_id,
                ),
            )
    else:
        bb = get_resolve_blackbox()
        creator_login = get_login_or_uid_by_uid(
            uid=client.uid,
            ip=request.META['REMOTE_ADDR'],
            blackbox=bb,
        )
        owner_logins = sorted(
            try_convert_uids_to_logins(
                uids=client.owner_uids,
                ip=request.META['REMOTE_ADDR'],
                blackbox=bb,
            ).values(),
        )
        owner_group_logins = sorted(
            try_convert_uids_to_logins(
                uids=client._owner_group_uids,
                ip=request.META['REMOTE_ADDR'],
                blackbox=bb,
            ).values(),
        )

        ttl = get_ttl_by_scopes(client.scopes)
        if ttl:
            humanize.i18n.activate('ru_RU')
            token_ttl = '{} {}'.format(
                'Не менее, чем' if check_if_is_refreshable(client.scopes) else 'Ровно',
                humanize.precisedelta(timedelta(seconds=ttl)),
            )
        else:
            token_ttl = 'Не ограничен'

        form = EditClientForm(initial={
            'id': client.id,
            'secret': mask_string(client.secret, show_first_n=6),
            'default_title': client.default_title,
            'title_ru': client.title_ru,
            'title_uk': client.title_uk,
            'title_en': client.title_en,
            'title_tr': client.title_tr,
            'default_description': client.default_description,
            'description_ru': client.description_ru,
            'description_uk': client.description_uk,
            'description_en': client.description_en,
            'description_tr': client.description_tr,
            'redirect_uris': '\n'.join(client.redirect_uris),
            'token_ttl': token_ttl,
            'scopes': ', '.join(s.keyword for s in client.scopes),
            'deleted_scopes': ', '.join(s.keyword for s in client.deleted_scopes),
            'extra_visible_scopes': ', '.join(s.keyword for s in client.extra_visible_scopes),
            'homepage': client.homepage,
            'approval_status': APPROVAL_STATUS_NAMES[client.approval_status],
            'created': client.created,
            'modified': client.modified,
            'glogouted': client.glogouted or 'Never',
            'deleted': client.deleted or 'Never',
            'creator': creator_login,
            'owner_uids': '\n'.join(owner_logins),
            'owner_groups': '\n'.join(client.owner_groups),
            'owner_group_logins': '\n'.join(owner_group_logins),
            'is_yandex': client.is_yandex,
            'telegram_bot_name': client.telegram_bot_name,
            'ios_default_app_id': client.ios_default_app_id,
            'ios_extra_app_ids': '\n'.join(client.ios_extra_app_ids),
            'ios_appstore_url': client.ios_appstore_url,
            'android_default_package_name': client.android_default_package_name,
            'android_extra_package_names': '\n'.join(client.android_extra_package_names),
            'android_cert_fingerprints': '\n'.join(client.android_cert_fingerprints),
            'android_appstore_url': client.android_appstore_url,
            'turboapp_base_url': client.turboapp_base_url,
            'allow_nonpublic_granttypes': client.allow_nonpublic_granttypes,
        })
    return render(
        request,
        'edit_client.html',
        {
            'env_name': env_name,
            'form': form,
            'client': client,
            'oauth_host': settings.OAUTH_HOST,
        },
    )


@yalogin_required
@grants_required([GRANT_ENTER_ADMIN, GRANT_VIEW_USERS])
def view_users(request, env_name):
    if request.method == 'POST':
        form = FindUserForm(request.POST)
        if form.is_valid():
            return HttpResponseRedirect(reverse('edit_user', kwargs={'env_name': env_name, 'uid': form.cleaned_data['uid']}))
    else:
        form = FindUserForm()

    return render(
        request,
        'view_users.html',
        {
            'env_name': env_name,
            'form': form,
        },
    )


@yalogin_required
@grants_required([GRANT_ENTER_ADMIN, GRANT_VIEW_USERS])
def edit_user(request, env_name, uid):
    admin_uid = request.yauser.uid
    bb_response = get_resolve_blackbox().userinfo(
        uid=uid,
        ip='127.0.0.1',
        dbfields=settings.BLACKBOX_DBFIELDS,
        attributes=settings.BLACKBOX_ATTRIBUTES,
        need_public_name=False,
    )
    if not bb_response.get('uid'):
        return HttpResponseNotFound('Пользователь с uid=%s не найден' % uid)
    login = bb_response['display_login'] or bb_response['login']
    is_corporate = get_attribute(bb_response, settings.BB_ATTR_ACCOUNT_IS_CORPORATE)
    avatar_id = bb_response['display_name']['avatar']['default']
    visible_hidden_scopes = [
        scope.keyword
        for scope in Scope.list(
            show_hidden=False,
            uid=uid,
        )
        if scope.is_hidden
    ]
    revokers = []
    for name, revoker in (
        ('Токены', REVOKER_TOKENS),
        ('ПП', REVOKER_APP_PASSWORDS),
        ('Куки', REVOKER_WEB_SESSIONS),
    ):
        revoke_time = get_revoke_time_from_bb_response(bb_response, revoker)
        if revoke_time == DT_NEVER_REVOKED:
            revoke_time = None
        revokers.append((name, revoke_time))

    passport_admin_url = settings.PASSPORT_ADMIN_URL_TEMPLATE % {'uid': uid}

    if request.method == 'POST':
        if not is_action_enabled(admin_uid, GRANT_EDIT_USERS):
            return HttpResponseForbidden(
                'У вас (uid=%d) нет гранта %s для выполнения данной операции' % (admin_uid, GRANT_EDIT_USERS),
            )

        form = EditUserForm(request.REQUEST)
        if form.is_valid():
            if form.cleaned_data['is_corporate'] and not is_corporate:
                get_passport().subscribe(uid=uid, sid='yacorp')
                log_admin_action(admin_uid, action='edit_user', uid=uid, is_corporate=True)
            elif not form.cleaned_data['is_corporate'] and is_corporate:
                try:
                    get_passport().unsubscribe(uid=uid, sid='yacorp')
                    log_admin_action(admin_uid, action='edit_user', uid=uid, is_corporate=False)
                except PassportNoSubscriptionError:
                    pass  # всё уже отписано до нас

            return HttpResponseRedirect(reverse('edit_user', kwargs={'env_name': env_name, 'uid': uid}))
    else:
        form = EditUserForm(initial={
            'is_corporate': is_corporate,
        })

    return render(
        request,
        'edit_user.html',
        {
            'env_name': env_name,
            'form': form,
            'login': login,
            'uid': uid,
            'passport_admin_url': passport_admin_url,
            'avatars_url': settings.AVATARS_READ_URL,
            'avatar_id': avatar_id,
            'visible_hidden_scopes': visible_hidden_scopes,
            'revokers': revokers,
        },
    )


@yalogin_required
@grants_required([GRANT_ENTER_ADMIN, GRANT_CREATE_SCOPES])
def new_scope(request, env_name):
    if request.POST:
        form_scope = ScopeForm(request.POST)
        if form_scope.is_valid():
            keyword = form_scope.cleaned_data['keyword']
            scope = {
                'keyword': keyword,
                'default_title': form_scope.cleaned_data['default_title'],
                'ttl': form_scope.cleaned_data['ttl'] or None,
                'is_ttl_refreshable': form_scope.cleaned_data['is_ttl_refreshable'],
                'requires_approval': form_scope.cleaned_data['requires_approval'],
                'is_hidden': form_scope.cleaned_data['is_hidden'],
                'visible_for_uids': list(form_scope.cleaned_data['visible_for_uids']) or [],
                'has_xtoken_grant': form_scope.cleaned_data['has_xtoken_grant'],
            }
            response = HttpResponse(json.dumps(scope, indent=4, ensure_ascii=False))
            response['Content-Disposition'] = 'attachment; filename=%s.%s.json' % (
                keyword,
                settings.ENV_TYPE if settings.ENV_NAME == 'localhost' else '%s.%s' % (
                    settings.ENV_NAME,
                    settings.ENV_TYPE,
                ),
            )
            return response
    else:
        form_scope = ScopeForm(initial={
            'keyword': 'test:permission',
            'default_title': 'Делать так, чтобы отпустило',
            'ttl': settings.DEFAULT_TOKEN_TTL,
            'is_ttl_refreshable': False,
            'requires_approval': False,
            'is_hidden': True,
            'visible_for_uids': '',
            'has_xtoken_grant': False,
        })
    return render(
        request,
        'create_scope.html',
        {
            'env_name': env_name,
            'form_scope': form_scope,
            'default_token_ttl': settings.DEFAULT_TOKEN_TTL,
        },
    )


@yalogin_required
@grants_required([GRANT_ENTER_ADMIN, GRANT_VIEW_TOKENS])
def view_tokens(request, env_name, uid):
    bb_response = get_resolve_blackbox().userinfo(
        uid=uid,
        ip='127.0.0.1',
        dbfields=settings.BLACKBOX_DBFIELDS,
        attributes=settings.BLACKBOX_ATTRIBUTES,
        need_display_name=False,
    )
    if not bb_response['uid']:
        return HttpResponseNotFound('User %s not found' % uid)

    login = bb_response['login']
    tokens_revoked_at = get_revoke_time_from_bb_response(
        bb_response,
        REVOKER_TOKENS,
    )
    app_passwords_revoked_at = get_revoke_time_from_bb_response(
        bb_response,
        REVOKER_APP_PASSWORDS,
    )

    tokens_info = list_tokens_with_clients_by_user(
        uid=uid,
        tokens_revoked_at=tokens_revoked_at,
        app_passwords_revoked_at=app_passwords_revoked_at,
        omit_deleted_clients=False,
        limit=settings.TOKENS_PER_PAGE,  # на самом деле будет больше - лимит действует для каждого шарда в отдельности
    )
    limit_reached = len(tokens_info) >= settings.TOKENS_PER_PAGE

    return render(
        request,
        'view_tokens.html',
        {
            'env_name': env_name,
            'uid': uid,
            'login': login,
            'tokens_info': tokens_info,
            'limit_reached': limit_reached,
        },
    )


@yalogin_required
@grants_required([GRANT_ENTER_ADMIN, GRANT_VIEW_TOKENS])
def verify_token(request, env_name):
    form = VerifyTokenForm(request.REQUEST)
    show_result = form.is_valid()

    offline_info_valid = db_info_valid = blackbox_info_valid = True
    offline_info = db_info = blackbox_raw_info = None
    token = access_token = token_alias = None

    if show_result:
        if form.cleaned_data['token_id']:
            token_id = form.cleaned_data['token_id']
            try:
                token = Token.by_id(token_id, allow_deleted=True)
                access_token = token.access_token
                token_alias = token.alias
            except (EntityNotFoundError, KeyError):  # KeyError - от шардера
                pass

        elif form.cleaned_data['refresh_token']:
            refresh_token = form.cleaned_data['refresh_token']
            offline_info_valid, offline_info, access_token = get_offline_info_for_refresh_token(refresh_token)
            token = Token.by_access_token(access_token)

        elif form.cleaned_data['access_token']:
            access_token = form.cleaned_data['access_token']
            offline_info_valid, offline_info = get_offline_info_for_token(access_token)
            token = Token.by_access_token(access_token)

        elif form.cleaned_data['uid'] and form.cleaned_data['token_alias']:
            uid = form.cleaned_data['uid']
            token_alias = form.cleaned_data['token_alias']
            offline_info_valid, offline_info = get_offline_info_for_alias(uid, token_alias)
            token = Token.by_uid_and_alias(uid=uid, alias=token_alias)

        db_info_valid, db_info = get_db_info(token)

        if access_token:
            blackbox_info_valid, blackbox_raw_info = get_token_info_from_blackbox(access_token)
        elif token_alias:
            blackbox_raw_info = '-'  # TODO: тоже как-то сходить в ЧЯ
            blackbox_info_valid = True
        else:
            blackbox_raw_info = 'No data to query blackbox'
            blackbox_info_valid = False

    return render(
        request,
        'verify_token.html',
        {
            'env_name': env_name,
            'form': form,
            'show_result': show_result,
            'offline_info': offline_info,
            'offline_info_valid': offline_info_valid,
            'db_info': db_info,
            'db_info_valid': db_info_valid,
            'blackbox_raw_info': blackbox_raw_info,
            'blackbox_info_valid': blackbox_info_valid,
        },
    )


@yalogin_required
@grants_required([GRANT_ENTER_ADMIN, GRANT_ENCRYPT_AM_CREDS])
def encrypt_am_creds(request, env_name):
    def _process_lines(text):
        for i, line in enumerate(text.split('\n'), start=1):
            line = line.strip()
            if line:
                yield i, line.split()

    error_message = None
    if request.POST:
        form = EncryptAmCredsForm(request.REQUEST)
        if form.is_valid():
            form_data = form.cleaned_data
            am_creds_manager = AmCredentialsManager()
            if 'encrypt' in request.POST:
                encrypted_lines = []
                for i, parts in _process_lines(form_data['decrypted']):
                    if len(parts) > 2:
                        encrypted_lines = []
                        error_message = 'Не удалось зашифровать значение №%d: слишком много частей' % i
                        break

                    encrypted = am_creds_manager.encrypt(parts[-1])
                    if len(parts) == 2:
                        encrypted = ' '.join([parts[0], encrypted])
                    encrypted_lines.append(encrypted)

                form_data['encrypted'] = '\n'.join(encrypted_lines)

            elif 'decrypt' in request.POST:
                decrypted_lines = []
                for i, parts in _process_lines(form_data['encrypted']):
                    if len(parts) > 2:
                        decrypted_lines = []
                        error_message = 'Не удалось расшифровать значение №%d: слишком много частей' % i
                        break

                    try:
                        decrypted, _ = am_creds_manager.decrypt(parts[-1])
                        if len(parts) == 2:
                            decrypted = ' '.join([parts[0], decrypted])
                        decrypted_lines.append(decrypted)
                    except DecryptFailedError:
                        decrypted_lines = []
                        error_message = 'Не удалось расшифровать значение №%d: некорректное значение' % i
                        break

                form_data['decrypted'] = '\n'.join(decrypted_lines)

            form = EncryptAmCredsForm(form_data)

    else:
        form = EncryptAmCredsForm()

    return render(
        request,
        'am_creds.html',
        {
            'env_name': env_name,
            'form': form,
            'error_message': error_message,
        },
    )


def ping(request, env_name):
    return HttpResponse('Pong\n')
