# -*- coding: utf-8 -*-
from urllib.parse import urlparse

from django import forms
from django.conf import settings
from passport.backend.oauth.api.api.forms import (
    ClientIdMixin,
    ClientSecretMixin,
    CodeChallengeMixin,
    DeviceIdField,
    DeviceInfoOptionalMixin,
    LimitedFileField,
    RedirectUriField,
    ScopeField,
    ScopesField,
    TokenIdMixin,
)
from passport.backend.oauth.core.api.forms import (
    BooleanField,
    CollectionField,
    error_messages,
    MappingField,
    StrippedCharField,
    TimestampField,
    UidMixin,
    UidOptionalMixin,
)
from passport.backend.oauth.core.db.request import CodeStrength


CODE_STRENGTH_BELOW_MEDIUM = 'below_medium'
CODE_STRENGTH_BELOW_MEDIUM_WITH_CRC = 'below_medium_with_crc'
CODE_STRENGTH_MEDIUM = 'medium'
CODE_STRENGTH_MEDIUM_WITH_CRC = 'medium_with_crc'
CODE_STRENGTH_LONG = 'long'


class RevokeTokensForm(UidMixin, ClientIdMixin, ClientSecretMixin):
    """Форма для отзыва токенов для пары "пользователь-приложение" """


class RevokeTokenForm(TokenIdMixin):
    """Форма для отзыва токена с заданным id"""


class RevokeTokenByAccessTokenForm(forms.Form):
    """Форма для отзыва токена с заданным access_token"""
    access_token = StrippedCharField(required=True, error_messages=error_messages)


class IssueAuthorizationCodeForm(UidOptionalMixin, ClientIdMixin, ClientSecretMixin,
                                 DeviceInfoOptionalMixin, CodeChallengeMixin):
    """Форма для выдачи кода подтверждения в обмен на uid, куку или х-токен"""
    ttl = forms.fields.IntegerField(
        required=False,
        min_value=10,
        max_value=24 * 3600,
        error_messages=error_messages,
    )
    code_strength = MappingField(
        mapping={
            CODE_STRENGTH_MEDIUM: CodeStrength.Medium,
            CODE_STRENGTH_LONG: CodeStrength.Long,
        },
        error_messages=error_messages,
    )
    require_activation = BooleanField(
        default=True,
        error_messages=error_messages,
    )
    by_uid = BooleanField(
        default=False,
        error_messages=error_messages,
    )


class ActivateAuthorizationCodeForm(UidOptionalMixin, ClientIdMixin, ClientSecretMixin):
    """Форма для "активации" кода подтверждения"""
    code = forms.fields.CharField(
        required=True,
        error_messages=error_messages,
    )

    def __init__(self, *args, **kwargs):
        super(ActivateAuthorizationCodeForm, self).__init__(*args, **kwargs)
        self.fields['code'].min_length = settings.REQUEST_CODE_LOW_LENGTH
        self.fields['code'].max_length = settings.REQUEST_CODE_HIGH_LENGTH


class IssueDeviceCodeForm(ClientIdMixin, DeviceInfoOptionalMixin):
    """Форма для выдачи пары кодов"""
    code_strength = MappingField(
        mapping={
            CODE_STRENGTH_BELOW_MEDIUM: CodeStrength.BelowMedium,
            CODE_STRENGTH_BELOW_MEDIUM_WITH_CRC: CodeStrength.BelowMediumWithCRC,
            CODE_STRENGTH_MEDIUM: CodeStrength.Medium,
            CODE_STRENGTH_MEDIUM_WITH_CRC: CodeStrength.MediumWithCRC,
        },
        error_messages=error_messages,
    )
    client_bound = BooleanField(
        default=True,
        error_messages=error_messages,
    )


class CheckDeviceCodeForm(forms.Form):
    """Форма для проверки device_code"""
    device_code = StrippedCharField(required=True, max_length=32, min_length=32, error_messages=error_messages)


class RefreshTokenForm(TokenIdMixin):
    """Форма для подновления токена с заданным id"""


class DeviceStatusForm(UidMixin, DeviceInfoOptionalMixin):
    """Форма для уида и информации о девайсе от АМ"""


class RevokeDeviceForm(UidMixin):
    """Форма для удаления токенов устройства"""
    device_id = DeviceIdField(required=True)


class ManageScopeVisibilityForm(ClientIdMixin):
    """Форма для показа/скрытия скоупов для приложения"""
    scope = ScopeField(required=True)


class CreateClientForTurboappForm(UidOptionalMixin):
    """Форма для создания приложения для использования турбоаппом"""
    title = StrippedCharField(required=True, error_messages=error_messages)
    description = StrippedCharField(required=True, max_length=250, error_messages=error_messages)
    icon_url = RedirectUriField(required=True, error_messages=error_messages)
    turboapp_base_url = RedirectUriField(required=True, error_messages=error_messages)

    def clean(self):
        cleaned_data = super(CreateClientForTurboappForm, self).clean()

        icon_url = cleaned_data.get('icon_url')
        if icon_url:
            scheme, host, path, params, query, fragment = urlparse(icon_url)
            allowed_scheme, allowed_host, _, _, _, _ = urlparse(settings.AVATARS_READ_URL)
            if not (
                scheme == allowed_scheme and
                host == allowed_host and
                not params and
                not query and
                not fragment
            ):
                self.add_error('icon_url', 'invalid')

        return cleaned_data


class TakeoutUserinfoForm(UidMixin):
    """Форма для уида и unixtime"""
    unixtime = TimestampField(required=True, error_messages=error_messages)


class AnonymizedUserInfoForm(ClientIdMixin):
    redirect_uri = RedirectUriField(required=True, error_messages=error_messages)


class ManageClientExternalForm(forms.Form):
    """Базовая форма для создания и редактирования OAuth приложения"""
    title = StrippedCharField(required=True, error_messages=error_messages)
    scopes = ScopesField(required=True, error_messages=error_messages)
    redirect_uris = CollectionField(
        required=True,
        validator=RedirectUriField(error_messages=error_messages),
        max_count=20,
        error_messages=error_messages,
    )
    description = StrippedCharField(required=False, max_length=250, error_messages=error_messages)
    homepage = forms.URLField(required=False, max_length=1024, error_messages=error_messages)
    icon_file = LimitedFileField(required=False, error_messages=error_messages)

    def __init__(self, visible_scopes=None, *args, **kwargs):
        super(ManageClientExternalForm, self).__init__(*args, **kwargs)
        self.fields['scopes'].choices = (visible_scopes or {}).items()
        self.fields['icon_file'].max_upload_size = settings.MAX_ICON_FILE_SIZE


class CreateClientExternalForm(ManageClientExternalForm):
    """Форма для создания OAuth приложения"""
    owner_login = StrippedCharField(required=True, error_messages=error_messages)


class EditClientExternalForm(ManageClientExternalForm, ClientIdMixin):
    """Форма для редактирования OAuth приложения"""
    pass


__all__ = (
    'ActivateAuthorizationCodeForm',
    'AnonymizedUserInfoForm',
    'CheckDeviceCodeForm',
    'CreateClientExternalForm',
    'CreateClientForTurboappForm',
    'DeviceStatusForm',
    'EditClientExternalForm',
    'IssueAuthorizationCodeForm',
    'IssueDeviceCodeForm',
    'ManageScopeVisibilityForm',
    'RefreshTokenForm',
    'RevokeDeviceForm',
    'RevokeTokenByAccessTokenForm',
    'RevokeTokenForm',
    'RevokeTokensForm',
    'TakeoutUserinfoForm',
)
