# -*- coding: utf-8 -*-
from collections import OrderedDict
from datetime import datetime
import json
import logging
from time import time

from django.conf import settings
from django.utils.encoding import (
    smart_bytes,
    smart_text,
)
from passport.backend.core.builders.avatars_mds_api import (
    AvatarsMdsApiBadImageFormatError,
    AvatarsMdsApiImageNotFoundError,
)
from passport.backend.core.builders.blackbox import (
    BLACKBOX_OAUTH_VALID_STATUS,
    get_alias,
)
from passport.backend.core.ydb.exceptions import YdbTemporaryError
from passport.backend.oauth.api.api.external.errors import (
    ActionNotRequiredError,
    ClientIdNotAllowedError,
    ClientSecretNotMatchedError,
    CodeNotAcceptedError,
    CodeNotFoundError,
    CodeOwnerRequiredError,
    DomainInvalidError,
    IconBadFormatError,
    IconNotFoundError,
    TokenInvalidTypeError,
    ScopeNotAllowedError,
)
from passport.backend.oauth.api.api.external.forms import (
    ActivateAuthorizationCodeForm,
    AnonymizedUserInfoForm,
    CheckDeviceCodeForm,
    CreateClientExternalForm,
    CreateClientForTurboappForm,
    DeviceStatusForm,
    EditClientExternalForm,
    IssueAuthorizationCodeForm,
    IssueDeviceCodeForm,
    RefreshTokenForm,
    RevokeDeviceForm,
    RevokeTokenByAccessTokenForm,
    RevokeTokenForm,
    RevokeTokensForm,
    TakeoutUserinfoForm,
)
from passport.backend.oauth.api.api.iface.errors import RedirectUriNotMatchedError
from passport.backend.oauth.core.api.base import BaseApiView
from passport.backend.oauth.core.api.errors import (
    BackendFailedError,
    ClientNotFoundError,
    OAuthTokenValidationError,
    UserNotFoundError,
)
from passport.backend.oauth.core.common.avatars import get_avatars_mds_api
from passport.backend.oauth.core.common.blackbox import (
    get_blackbox,
    get_revoke_time_from_bb_response,
    REVOKER_APP_PASSWORDS,
    REVOKER_TOKENS,
    REVOKER_WEB_SESSIONS,
)
from passport.backend.oauth.core.common.constants import GRANT_TYPE_DEVICE_CODE
from passport.backend.oauth.core.common.jwt import make_jwt
from passport.backend.oauth.core.common.utils import update_url_params
from passport.backend.oauth.core.db.client import (
    Client,
    is_redirect_uri_allowed,
    list_clients_by_creator,
    make_icon_id,
    make_icon_key,
    PLATFORM_ANDROID,
    PLATFORM_IOS,
    PLATFORM_WEB,
)
from passport.backend.oauth.core.db.eav import (
    CREATE,
    DBTemporaryError,
    EntityNotFoundError,
    UPDATE,
)
from passport.backend.oauth.core.db.eav.types import DB_NULL
from passport.backend.oauth.core.db.limits import (
    assert_is_allowed,
    check_grant,
    has_grant,
)
from passport.backend.oauth.core.db.psuid import make_psuid
from passport.backend.oauth.core.db.request import (
    ActivationStatus,
    code_challenge_method_to_name,
    code_strength_to_name,
    CodeType,
    create_request,
    Request,
    VerificationCodeCollisionError,
)
from passport.backend.oauth.core.db.scope import Scope
from passport.backend.oauth.core.db.token import (
    invalidate_single_token,
    invalidate_tokens,
    list_tokens_by_uid,
    list_tokens_with_clients_by_user,
    parse_token,
    StatelessToken,
    Token,
)
from passport.backend.oauth.core.db.token.normal_token import (
    delete_single_token,
    list_tokens_by_user_and_device,
    split_tokens_to_devices,
)
from passport.backend.oauth.core.db.turboapp import (
    is_psuid_allowed_for_redirect_uri,
    is_redirect_uri_turboapp,
)
from passport.backend.oauth.core.logs.statbox import to_statbox
from passport.backend.utils.io import LazyFileLikeObject
from passport.backend.utils.time import datetime_to_integer_unixtime


log = logging.getLogger('api')


REVOKE_TOKEN_BASE_GRANT = 'api.revoke_token.base'
REVOKE_TOKEN_ANY_GRANT = 'api.revoke_token.any'
REVOKE_TOKEN_MAILISH_GRANT = 'api.revoke_token.mailish'

REVOKE_TOKEN_BY_ACCESS_TOKEN_GRANT = 'api.revoke_token.by_access_token'
REVOKE_TOKENS_GRANT = 'api.revoke_tokens'
REFRESH_TOKEN_GRANT = 'api.refresh_token'
ISSUE_AUTHORIZATION_CODE_GRANT = 'api.issue_authorization_code'
ISSUE_AUTHORIZATION_CODE_BY_UID_GRANT = 'api.issue_authorization_code_by_uid'
ISSUE_DEVICE_CODE_GRANT = 'api.issue_device_code'
CHECK_DEVICE_CODE_GRANT = 'api.check_device_code'
DEVICE_STATUS_GRANT = 'api.device_status'
REVOKE_DEVICE_GRANT = 'api.revoke_device'
MANAGE_SCOPE_VISIBILITY_GRANT = 'api.manage_scope_visibility'
CREATE_CLIENT_FOR_TURBOAPP_GRANT = 'api.create_client_for_turboapp'
TAKEOUT_USER_INFO_GRANT = 'api.takeout_user_info'
ANONYMIZED_USER_INFO_GRANT = 'api.anonymized_user_info'


class NonseekableLazyFileLikeObject(LazyFileLikeObject):
    # Не очень корректная имплементация: желательно от неё отказаться
    def seek(self, pos, whence=0):
        pass


class BaseExternalApiView(BaseApiView):
    grants_for_account_type = None
    temporary_exceptions_list = BaseApiView.temporary_exceptions_list + [YdbTemporaryError]

    def get_client(self, check_secret=True):
        self.client = Client.by_display_id(self.form_values['client_id'])
        if self.client is None:
            raise ClientNotFoundError()
        # У старых приложений секрета нет. Поэтому проверяем секрет только при его наличии.
        if check_secret and self.client.secret and self.client.secret != self.form_values['client_secret']:
            raise ClientSecretNotMatchedError()

    def check_grants_for_account_type(self, bb_response):
        alias_types = (
            'portal',
            'pdd',
            'lite',
            'mailish',
            'social',
            'phonish',
            'kinopoisk',
            'uber',
            'yambot',
            'kolonkish',
        )

        if self.grants_for_account_type is None:
            raise ValueError('Account specific grants have not been specified!')  # pragma: no cover

        anything_grant = self.grants_for_account_type.get('any')
        if not anything_grant:
            raise ValueError('Grant account type "any" has not been specified!')  # pragma: no cover

        # Если у пользователя есть "зонтичный" грант на любые аккаунты, то
        # в дальнейшей проверке более специфичного гранта нет смысла.
        if has_grant(
            grant=anything_grant,
            consumer=self.consumer,
            ip=self.request.env.consumer_ip,
            service_ticket=self.request.env.service_ticket,
        ):
            return

        for alias_type in alias_types:
            grant = self.grants_for_account_type.get(alias_type)
            if grant and get_alias(bb_response, alias_type):
                check_grant(
                    grant=grant,
                    consumer=self.consumer,
                    ip=self.request.env.consumer_ip,
                    service_ticket=self.request.env.service_ticket,
                )
                return

        # Если ни один грант не подошёл, то выводим ошибку с "зонтичным" грантом.
        check_grant(
            grant=anything_grant,
            consumer=self.consumer,
            ip=self.request.env.consumer_ip,
            service_ticket=self.request.env.service_ticket,
        )

    def process_form_with_scopes_arg(self, form_class, data=None, files=None):
        scopes = Scope.list()
        visible_scopes = dict((s.keyword, s.default_title) for s in scopes)

        self.form_values.update(
            self.process_form(form_class, form_args={'visible_scopes': visible_scopes}, data=data, files=files),
        )


class RevokeTokens(BaseExternalApiView):
    """
    Отзывает токены для пары (пользователь, приложение)
    """
    allowed_methods = ['POST']
    required_grants = [REVOKE_TOKENS_GRANT]
    base_form = RevokeTokensForm

    def process_request(self, request):
        self.get_client(check_secret=True)
        tokens = [
            token
            for token in list_tokens_by_uid(self.uid)
            if token.client_id == self.client.id
        ]
        invalidate_tokens(tokens, self.client, request.env, reason='api_revoke_tokens')


class RevokeTokenById(BaseExternalApiView):
    """
    Отзывает токен с указанным id
    """
    allowed_methods = ['POST']
    required_grants = [REVOKE_TOKEN_BASE_GRANT]
    grants_for_account_type = {
        'any': REVOKE_TOKEN_ANY_GRANT,
        'mailish': REVOKE_TOKEN_MAILISH_GRANT,
    }
    base_form = RevokeTokenForm

    def process_request(self, request):
        try:
            token = Token.by_id(self.form_values['token_id'])
            client = token.get_client()
            user_info = self.get_user_by_uid(uid=token.uid)
        except EntityNotFoundError:
            return  # токена или приложения нет - токен и так невалиден
        except UserNotFoundError:
            return  # юзер удалён - токен и так невалиден

        self.check_grants_for_account_type(user_info['bb_response'])

        invalidate_single_token(token, client, request.env, reason='api_revoke_token')


class RevokeTokenByAccessToken(BaseExternalApiView):
    """
    Отзывает токен с указанным access_token'ом.
    В отличие от аналогичной публичной ручки, не требует передачи client_id и client_secret.
    """
    allowed_methods = ['POST']
    required_grants = [REVOKE_TOKEN_BY_ACCESS_TOKEN_GRANT]
    base_form = RevokeTokenByAccessTokenForm

    def process_request(self, request):
        bb_response = get_blackbox().oauth(
            oauth_token=self.form_values['access_token'],
            ip=request.env.user_ip,
            need_token_attributes=True,
            need_client_attributes=True,
            need_display_name=False,
        )
        if bb_response['status'] != BLACKBOX_OAUTH_VALID_STATUS:
            return  # токен уже невалиден - всё хорошо

        token = parse_token(bb_response)
        if isinstance(token, StatelessToken):
            raise TokenInvalidTypeError()

        client = Client.parse(bb_response)
        invalidate_single_token(token, client, request.env, reason='api_revoke_token')


class IssueAuthorizationCode(BaseExternalApiView):
    """
    Выдаёт код подтверждения в обмен на uid, куку или х-токен
    """
    allowed_methods = ['POST']
    required_grants = [ISSUE_AUTHORIZATION_CODE_GRANT]
    base_form = IssueAuthorizationCodeForm

    def process_request(self, request):
        self.get_client(check_secret=True)
        uid = self.form_values['uid']
        if self.form_values['by_uid']:
            check_grant(
                ISSUE_AUTHORIZATION_CODE_BY_UID_GRANT,
                consumer=self.consumer,
                ip=self.request.env.consumer_ip,
                service_ticket=self.request.env.service_ticket,
            )
            user_info = self.get_user_by_uid(uid=uid)
        elif self.request.env.user_ticket:
            user_info = self.get_user_from_user_ticket()
        elif self.request.env.authorization:
            user_info = self.get_user_from_token()
        else:
            user_info = self.get_user_from_session(uid=uid)

        try:
            token_request = create_request(
                uid=user_info['uid'],
                client=self.client,
                scopes=self.client.scopes,
                device_id=self.form_values['device_id'],
                device_name=self.form_values['device_name'],
                code_strength=self.form_values['code_strength'],
                code_challenge=self.form_values['code_challenge'],
                code_challenge_method=self.form_values['code_challenge_method'],
                require_activation=self.form_values['require_activation'],
                is_accepted=True,
                make_code=True,
                ttl=self.form_values['ttl'],
                login_id=user_info['login_id'],
            )
        except VerificationCodeCollisionError:
            raise BackendFailedError()

        self.response_values.update(
            code=token_request.code,
            expires_in=token_request.ttl,
        )
        to_statbox(
            mode='issue_code',
            status='ok',
            token_request_id=token_request.display_id,
            code_strength=code_strength_to_name(token_request.code_strength),
            code_challenge_method=code_challenge_method_to_name(token_request.code_challenge_method),
            uid=token_request.uid,
            client_id=self.client.display_id,
            scopes=','.join(map(str, token_request.scopes)),
            require_activation=self.form_values['require_activation'],
        )


class ActivateAuthorizationCode(BaseExternalApiView):
    """
    "Активирует" ранее выданный код подтверждения, если он и переданная кука
    принадлежат одному и тому же пользователю
    """
    allowed_methods = ['POST']
    required_grants = [ISSUE_AUTHORIZATION_CODE_GRANT]
    base_form = ActivateAuthorizationCodeForm

    def process_request(self, request):
        self.get_client(check_secret=True)
        user_info = self.get_user_from_session(uid=self.form_values['uid'])
        token_request = Request.by_verification_code(self.client.id, smart_bytes(self.form_values['code']))

        if not token_request or token_request.is_expired:
            raise CodeNotFoundError()
        if token_request.uid != user_info['uid']:
            raise CodeOwnerRequiredError()
        if not token_request.needs_activation:
            raise ActionNotRequiredError()

        with UPDATE(token_request):
            token_request.activation_status = ActivationStatus.Activated

        to_statbox(
            mode='activate_code',
            status='ok',
            token_request_id=token_request.display_id,
            uid=token_request.uid,
            client_id=self.client.display_id,
        )


class IssueDeviceCode(BaseExternalApiView):
    """
    Выдаёт пару кодов подтверждения
    """
    allowed_methods = ['POST']
    required_grants = [ISSUE_DEVICE_CODE_GRANT]
    base_form = IssueDeviceCodeForm

    def make_verification_url(self, tld, client_id=None):
        verification_url = settings.CUSTOM_DEVICE_CODE_VERIFICATION_URLS_BY_TLD.get(
            tld,
            settings.DEFAULT_DEVICE_CODE_VERIFICATION_URL_TEMPLATE % {'tld': tld},
        )
        if client_id is not None:
            verification_url = update_url_params(verification_url, client_id=client_id)
        return verification_url

    def process_request(self, request):
        self.get_client(check_secret=False)

        assert_is_allowed(
            scope_list=self.client.scopes,
            grant_type=GRANT_TYPE_DEVICE_CODE,
            client_id=self.client.display_id,
            ip=request.env.consumer_ip,
            uid=None,
        )

        device_id = self.form_values['device_id'] or None
        device_name = self.form_values['device_name'] or None

        is_code_client_bound = self.form_values['client_bound']
        try:
            token_request = create_request(
                client=self.client,
                scopes=self.client.scopes,
                is_token_response=False,
                ttl=settings.DEVICE_CODE_TTL,
                device_id=device_id,
                device_name=device_name,
                make_code=True,
                code_strength=self.form_values['code_strength'],
                code_type=CodeType.ClientBound if is_code_client_bound else CodeType.Unique,
            )
        except VerificationCodeCollisionError:
            raise BackendFailedError()

        self.response_values.update(
            device_code=token_request.display_id,
            user_code=token_request.code,
            expires_in=token_request.ttl,
            verification_url=self.make_verification_url(
                tld=request.TLD,
                client_id=self.client.display_id if is_code_client_bound else None,
            ),
        )
        to_statbox(
            mode='issue_device_code',
            status='ok',
            client_id=self.client.display_id,
            token_request_id=token_request.display_id,
            code_strength=code_strength_to_name(token_request.code_strength),
            device_id=device_id,
            device_name=device_name,
        )


class CheckDeviceCode(BaseExternalApiView):
    """
    Проверяет, был ли device_code подтверждён
    """
    allowed_methods = ['GET']
    required_grants = [CHECK_DEVICE_CODE_GRANT]
    base_form = CheckDeviceCodeForm

    def process_request(self, request):
        token_request = Request.by_display_id(smart_bytes(self.form_values['device_code']))
        if not token_request or token_request.is_expired:
            raise CodeNotFoundError()

        if not token_request.is_accepted or not token_request.uid:
            raise CodeNotAcceptedError()

        bb_response = get_blackbox().userinfo(
            uid=token_request.uid,
            ip=request.env.user_ip,
            dbfields=settings.BLACKBOX_DBFIELDS,
            attributes=settings.BLACKBOX_ATTRIBUTES,
            need_display_name=False,
        )
        if not bb_response['uid']:
            # юзер заакцептил реквест и сразу удалился. Считаем реквест протухшим.
            raise CodeNotFoundError()
        elif token_request.is_invalidated_by_user_logout(
            revoke_time=get_revoke_time_from_bb_response(bb_response, REVOKER_WEB_SESSIONS),
        ):
            # юзер отозвал все веб-сессии. Считаем реквест протухшим.
            raise CodeNotFoundError()

        self.response_values.update(
            uid=token_request.uid,
            scopes=[scope.keyword for scope in token_request.scopes],
        )


class RefreshToken(BaseExternalApiView):
    """
    Подновляет токен с указанным id
    """
    allowed_methods = ['POST']
    required_grants = [REFRESH_TOKEN_GRANT]
    base_form = RefreshTokenForm

    def process_request(self, request):
        # ручку будет дёргать Ксива, поэтому выдаём минимум ошибок
        # TODO: потюнить тайминги тут и в ЧЯ (рефрешить токены чаще)
        try:
            token = Token.by_id(self.form_values['token_id'])
        except EntityNotFoundError:
            return  # токена нет - подновлять нечего

        if not token.can_be_refreshed:
            return  # подновлять ещё рано

        try:
            with UPDATE(token):
                token.reset_ttl()
        except DBTemporaryError:
            return  # подновить не получилось - попробуем в следующий раз

        to_statbox(
            mode='refresh',
            target='token',
            status='ok',
            token_id=token.id,
            uid=token.uid,
        )


class DeviceStatus(BaseExternalApiView):
    """
    Сообщает, есть ли у пользователя живые токены для указанного device_id.
    """
    allowed_methods = ['GET']
    required_grants = [DEVICE_STATUS_GRANT]
    base_form = DeviceStatusForm

    def has_auth_on_device(self, uid, device_id, tokens_revoked_at):
        if not device_id:
            return False
        else:
            all_tokens_with_clients = list_tokens_with_clients_by_user(
                uid,
                tokens_revoked_at=tokens_revoked_at,
                app_passwords_revoked_at=datetime.now(),  # сэкономим: не будем получать ПП, они нам не нужны
            )
            if any(
                token.device_id == device_id
                for token, _, is_valid in all_tokens_with_clients
                if is_valid
            ):
                return True
            else:
                return False

    def process_request(self, request):
        user_info = self.get_user_by_uid(uid=self.form_values['uid'])
        device_id = self.form_values['device_id'] or None
        device_name = self.form_values['device_name'] or None

        self.response_values.update(
            device_id=device_id,
            device_name=device_name,
            has_auth_on_device=self.has_auth_on_device(
                uid=user_info['uid'],
                device_id=device_id,
                tokens_revoked_at=get_revoke_time_from_bb_response(user_info['bb_response'], REVOKER_TOKENS),
            )
        )


class RevokeDevice(BaseExternalApiView):
    required_grants = [REVOKE_DEVICE_GRANT]
    allowed_methods = ['POST']
    base_form = RevokeDeviceForm

    def process_request(self, request):
        user_info = self.get_user_by_uid(uid=self.form_values['uid'])
        datetime_now = datetime.now()
        device_tokens_with_clients = list_tokens_by_user_and_device(
            self.uid,
            self.form_values['device_id'],
            revoked_at=get_revoke_time_from_bb_response(
                user_info['bb_response'],
                REVOKER_TOKENS,
            ),
        )
        # TODO: удаление одной транзакцией
        for token, client in device_tokens_with_clients:
            delete_single_token(
                token,
                client,
                request.env,
                reason='api_revoke_device_tokens',
                dt=datetime_now,
            )


class CreateClientForTurboapp(BaseExternalApiView):
    required_grants = [CREATE_CLIENT_FOR_TURBOAPP_GRANT]
    allowed_methods = ['POST']
    base_form = CreateClientForTurboappForm

    def upload_icon(self, icon_url):
        try:
            icon_file_object = get_avatars_mds_api().download(icon_url)
        except AvatarsMdsApiImageNotFoundError:
            raise IconNotFoundError()

        key = make_icon_key(client_id=self.client.display_id)
        try:
            # Тут ретраиться нельзя: при ретрае заливаемый файл может корраптиться из-за некорректной реализации seek
            group_id = get_avatars_mds_api(retries=1).upload_from_file(
                key=key,
                file_=NonseekableLazyFileLikeObject(icon_file_object),
            )
        except AvatarsMdsApiBadImageFormatError:
            raise IconBadFormatError()
        return make_icon_id(group_id, key)

    def process_request(self, request):
        user_info = self.get_user_from_session(uid=self.form_values['uid'])

        scopes = [Scope.by_keyword(keyword) for keyword in settings.DEFAULT_TURBOAPP_SCOPES]
        with CREATE(Client.create(
            uid=user_info['uid'],
            redirect_uris=[],
            scopes=scopes,
            default_title=self.form_values['title'],
            default_description=self.form_values['description'],
            homepage=self.form_values['turboapp_base_url'],
        )) as self.client:
            self.client.turboapp_base_url = self.form_values['turboapp_base_url']
            self.client.icon_id = self.upload_icon(self.form_values['icon_url'])

        self.response_values.update(client_id=self.client.display_id)

        to_statbox(
            mode='create_client',
            type='turboapp',
            status='ok',
            client_id=self.client.display_id,
            creator_uid=self.client.uid,
            client_title=self.client.default_title,
            client_description=self.client.default_description,
            client_scopes=','.join(s.keyword for s in self.client.scopes),
            client_redirect_uris=', '.join(self.client.redirect_uris),
        )


class TakeoutUserInfo(BaseExternalApiView):
    """
    Отдаёт хранимые данные пользователя для выдачи пользователю в виде архива
    """
    allowed_methods = ['GET', 'POST']
    required_grants = [TAKEOUT_USER_INFO_GRANT]
    base_form = TakeoutUserinfoForm

    def respond_success(self):
        # Не отдаём status='ok' по дефолту - возможны и другие успешные статусы
        return self.make_json_response(self.response_values)

    def respond_error(self, e):
        # Ошибки отдаём в нестандартном формате: таковы требования сервиса Takeout
        return self.make_json_response(dict(self.response_values, status='error', error=', '.join(sorted(e.errors))))

    @staticmethod
    def to_json(data):
        return json.dumps(data, indent=2, ensure_ascii=False)

    @staticmethod
    def make_ordered_dict(keys_and_values):
        data = []
        for key, value in keys_and_values:
            if not value:
                continue
            elif isinstance(value, datetime):
                value = datetime_to_integer_unixtime(value)
            elif not isinstance(value, (OrderedDict, dict, list, tuple, int, float, bool)):
                value = smart_text(value)
            data.append((key, value))

        return OrderedDict(data)

    @classmethod
    def token_to_response(cls, token, client):
        return cls.make_ordered_dict([
            ('client_id', client.display_id),
            ('client_title', client.get_title()),
            ('scope', sorted([scope.keyword for scope in token.scopes])),
            ('issued', token.issued),
        ])

    @classmethod
    def client_to_response(cls, client):
        platforms = {}
        if PLATFORM_IOS in client.platforms:
            platforms[PLATFORM_IOS] = cls.make_ordered_dict([
                ('app_ids', client.ios_app_ids),
                ('appstore_url', client.ios_appstore_url),
            ])
        if PLATFORM_ANDROID in client.platforms:
            platforms[PLATFORM_ANDROID] = cls.make_ordered_dict([
                ('package_names', client.android_package_names),
                ('cert_fingerprints', client.android_cert_fingerprints),
                ('appstore_url', client.android_appstore_url),
            ])
        if PLATFORM_WEB in client.platforms:
            platforms[PLATFORM_WEB] = cls.make_ordered_dict([
                ('redirect_uris', client.redirect_uris),
            ])

        result = cls.make_ordered_dict([
            ('id', client.display_id),
            ('title', client.get_title()),
            ('description', client.get_description()),
            ('homepage', client.homepage),
            ('icon_url', client.get_icon_url()),
            ('platforms', platforms),
            ('created', client.created),
        ])
        if client.is_yandex:
            # Добавляем только для яндексовых, чтобы не смущать внешних пользователей
            result.update(is_yandex=True)
        return result

    def get_app_passwords_info(self, valid_tokens_with_clients):
        return [
            self.make_ordered_dict([
                ('name', token.device_name),  # у ПП имя есть всегда
                ('scope', sorted([scope.keyword for scope in token.scopes])),
                ('issued', token.issued),
            ])
            for token, _ in valid_tokens_with_clients
            if token.is_app_password
        ]

    def get_tokens_info(self, valid_tokens_with_clients):
        tokens_with_device = []
        tokens_without_device = []

        for token, client in valid_tokens_with_clients:
            if token.is_app_password:
                continue  # ПП обрабатываем в соседнем методе
            elif token.device_id != DB_NULL:
                tokens_with_device.append((token, client))
            else:
                tokens_without_device.append((token, client))

        device_infos = split_tokens_to_devices(tokens_with_device)
        return self.make_ordered_dict([
            ('devices', [
                self.make_ordered_dict([
                    ('device_id', device_info['device_id']),
                    ('device_name', device_info['device_name']),
                    ('app_platform', device_info['app_platform']),
                    ('tokens', [
                        self.token_to_response(token, client)
                        for token, client in device_info['tokens_with_clients']
                    ]),
                ])
                for device_info in device_infos
            ]),
            ('other', [
                self.token_to_response(token, client)
                for token, client in tokens_without_device
            ]),
        ])

    def get_clients_info(self, uid):
        clients = list_clients_by_creator(uid)
        return [
            self.client_to_response(client)
            for client in clients
        ]

    def process_request(self, request):
        try:
            user_info = self.get_user_by_uid(uid=self.form_values['uid'])
        except UserNotFoundError:
            self.response_values.update(status='no_data')
            return

        uid = user_info['uid']

        valid_tokens_with_clients = [
            (token, client)
            for token, client, is_valid in list_tokens_with_clients_by_user(
                uid=uid,
                tokens_revoked_at=get_revoke_time_from_bb_response(
                    user_info['bb_response'],
                    REVOKER_TOKENS,
                ),
                app_passwords_revoked_at=get_revoke_time_from_bb_response(
                    user_info['bb_response'],
                    REVOKER_APP_PASSWORDS,
                ),
            )
            if is_valid
        ]

        app_passwords_info = self.get_app_passwords_info(valid_tokens_with_clients=valid_tokens_with_clients)
        tokens_info = self.get_tokens_info(valid_tokens_with_clients=valid_tokens_with_clients)
        clients_info = self.get_clients_info(uid=uid)

        if not any([tokens_info, app_passwords_info, clients_info]):
            self.response_values.update(status='no_data')
        else:
            self.response_values.update(status='ok', data={})
            if tokens_info:
                self.response_values['data'].update({
                    'authorized_devices.json': self.to_json(tokens_info),
                })
            if clients_info:
                self.response_values['data'].update({
                    'registered_clients.json': self.to_json(clients_info),
                })
            if app_passwords_info:
                self.response_values['data'].update({
                    'app_passwords.json': self.to_json(app_passwords_info),
                })


class AnonymizedUserInfo(BaseExternalApiView):
    """
    Выдаёт JWT-токен, содержащий PSUID
    """
    allowed_methods = ['GET']
    required_grants = [ANONYMIZED_USER_INFO_GRANT]
    base_form = AnonymizedUserInfoForm

    def assert_redirect_uri_is_allowed(self, redirect_uri):
        if not (
            is_redirect_uri_turboapp(redirect_uri) and
            is_redirect_uri_allowed(
                redirect_uri,
                allowed_redirect_uris=(
                    self.client.redirect_uris +
                    self.client.platform_specific_redirect_uris
                ),
            )
        ):
            raise RedirectUriNotMatchedError()

        if not is_psuid_allowed_for_redirect_uri(
            client=self.client,
            redirect_uri=redirect_uri,
        ):
            raise RedirectUriNotMatchedError()

    def process_request(self, request):
        self.get_client(check_secret=False)
        self.assert_redirect_uri_is_allowed(self.form_values['redirect_uri'])

        user_info = self.get_user_from_token()

        psuid = make_psuid(
            uid=user_info['uid'],
            client=self.client,
        )
        jwt = make_jwt(
            secret=self.client.secret,
            expires=int(time() + settings.PSUID_JWT_TTL),
            issuer=request.get_host(),
            custom_fields=dict(psuid=psuid),
        )

        self.response_values.update(
            jwt=jwt,
            expires_in=settings.PSUID_JWT_TTL,
        )


class BaseManageClientExternal(BaseExternalApiView):
    required_grants = []
    allowed_methods = ['POST']

    def process_request(self, request):
        self.token_info = get_blackbox().oauth(
            oauth_token=self.oauth_token,
            ip=request.env.user_ip,
            need_display_name=False,
            need_public_name=False,
        )
        if (
            self.token_info['status'] != BLACKBOX_OAUTH_VALID_STATUS or
            settings.OAUTH_MANAGE_CLIENTS_SCOPE_KEYWORD not in self.token_info['oauth']['scope'] or
            self.token_info['uid'] is not None
        ):
            raise OAuthTokenValidationError()

        allowed_scopes = settings.EXTERNAL_MANAGE_CLIENT_SCOPES.copy()
        if self.token_info['oauth']['client_id'] in settings.EXTERNAL_MANAGE_DEFAULT_PHONES_ALLOWED_CLIENT_ID:
            allowed_scopes.append(settings.DEFAULT_PHONE_SCOPE_KEYWORD)
        for scope in self.form_values['scopes']:
            if scope.keyword not in allowed_scopes:
                raise ScopeNotAllowedError('Scope not allowed')

    def upload_icon(self):
        key = make_icon_key(client_id=self.client.display_id)
        try:
            group_id = get_avatars_mds_api().upload_from_file(key=key, file_=self.request.FILES['icon_file'])
        except AvatarsMdsApiBadImageFormatError:
            raise IconBadFormatError()
        return make_icon_id(group_id, key)

    def log_ok_to_statbox(self, mode):
        to_statbox(
            mode=mode,
            type='external_web',
            status='ok',
            client_id=self.client.display_id,
            creator_uid=self.client.uid,
            client_title=self.client.default_title,
            client_description=self.client.default_description,
            client_scopes=','.join(s.keyword for s in self.client.scopes),
            client_redirect_uris=', '.join(self.client.redirect_uris),
        )

    def process_root_form(self, data):
        pass


class CreateClientExternal(BaseManageClientExternal):
    required_grants = []
    allowed_methods = ['POST']

    def process_request(self, request):
        self.process_form_with_scopes_arg(CreateClientExternalForm, data=request.REQUEST, files=request.FILES)
        super(CreateClientExternal, self).process_request(request)

        user_info = self.get_user_by_login(self.form_values['owner_login'])
        user_domain = user_info['bb_response']['domain']
        if user_domain != settings.CLIENT_ID_TO_DOMAIN.get(self.token_info['oauth']['client_id']):
            raise DomainInvalidError()

        with CREATE(
            Client.create(
                uid=user_info['uid'],
                default_title=self.form_values['title'],
                default_description=self.form_values.get('description'),
                homepage=self.form_values.get('homepage'),
                scopes=[Scope.by_keyword(scope.keyword) for scope in self.form_values['scopes']],
            )
        ) as self.client:
            self.client._redirect_uris = self.form_values['redirect_uris']
            self.client._callback = None
            self.client.icon_id = self.upload_icon() if 'icon_file' in request.FILES else None

        self.response_values.update(client_id=self.client.display_id, client_secret=self.client.secret)

        self.log_ok_to_statbox('create_client')


class EditClientExternal(BaseManageClientExternal):
    required_grants = []
    allowed_methods = ['POST']

    def process_request(self, request):
        self.process_form_with_scopes_arg(EditClientExternalForm, data=request.REQUEST, files=request.FILES)
        super(EditClientExternal, self).process_request(request)

        self.client = Client.by_display_id(self.form_values['client_id'])
        if self.client is None:
            raise ClientNotFoundError()

        user_domain = self.get_user_by_uid(self.client.uid)['bb_response']['domain']

        if user_domain != settings.CLIENT_ID_TO_DOMAIN.get(self.token_info['oauth']['client_id']):
            raise ClientIdNotAllowedError()

        scopes_removed = not self.client.scopes.issubset(self.form_values['scopes'])

        with UPDATE(self.client) as client:
            if scopes_removed:
                client.glogout()

            client.modified = datetime.now()
            client.default_title = self.form_values['title']
            client.default_description = self.form_values.get('description')
            client.homepage = self.form_values.get('homepage')
            client._redirect_uris = self.form_values['redirect_uris']
            client._callback = None
            client.icon_id = self.upload_icon() if 'icon_file' in request.FILES else None
            client.set_scopes(self.form_values['scopes'])

        self.log_ok_to_statbox('edit_client')
