# -*- coding: utf-8 -*-
import re
from urllib.parse import urlparse

from django import forms
from django.conf import settings
from django.core import validators
from django.core.exceptions import ValidationError
from django.utils.datastructures import MultiValueDict
from passport.backend.oauth.core.api.forms import (
    CollectionField,
    MappingField,
    StrippedCharField,
)
from passport.backend.oauth.core.common.utils import normalize_url
from passport.backend.oauth.core.db.device_info import get_device_info
from passport.backend.oauth.core.db.request import CodeChallengeMethod
from passport.backend.oauth.core.db.scope import Scope


DEVICE_ID_REGEX = re.compile(r'^[ -~]+$')  # все ASCII-символы с кодами от 32 до 126


CODE_CHALLENGE_METHOD_PLAIN = 'plain'
CODE_CHALLENGE_METHOD_S256 = 'S256'
CODE_CHALLENGE_MIN_LENGTH = 43
CODE_CHALLENGE_MAX_LENGTH = 200

# маппинг из ключей ошибок Django в ошибки АПИ
error_messages = {
    'required': 'missing',
    'invalid': 'invalid',
    'max_value': 'invalid',
    'min_value': 'invalid',
    'max_length': 'too_long',
    'min_length': 'too_short',
    'invalid_link': 'invalid',
    'invalid_choice': 'invalid',
    'invalid_list': 'invalid',
}


class LimitedFileField(forms.fields.FileField):
    def __init__(self, max_upload_size=None, **kwargs):
        self.max_upload_size = max_upload_size
        super(LimitedFileField, self).__init__(**kwargs)

    def clean(self, data, initial=None):
        file_ = super(LimitedFileField, self).clean(data, initial)
        if (
            file_ is not None and
            self.max_upload_size is not None and
            file_.size > self.max_upload_size
        ):
            raise ValidationError('too_large')

        return data


class ScopeField(StrippedCharField):
    def clean(self, value):
        raw_value = super(ScopeField, self).clean(value)
        try:
            return Scope.by_keyword(raw_value)
        except ValueError:
            raise ValidationError('invalid')


class ScopesField(forms.fields.MultipleChoiceField):
    def clean(self, value):
        raw_value = super(ScopesField, self).clean(value)
        return set([Scope.by_keyword(keyword) for keyword in raw_value])


class RedirectUriField(StrippedCharField):
    def __init__(self, max_length=1024, **kwargs):
        super(RedirectUriField, self).__init__(max_length=max_length, **kwargs)

    def clean(self, value):
        value = super(RedirectUriField, self).clean(value)
        if not value:
            return ''
        elif '|' in value:
            raise forms.ValidationError('invalid')

        try:
            scheme, netloc, path, params, query, fragment = urlparse(value)
        except ValueError:
            raise forms.ValidationError('invalid')

        if not scheme:
            raise forms.ValidationError('scheme_missing')
        elif scheme.lower() in ['http', 'https'] and not netloc:
            raise forms.ValidationError('not_absolute')
        elif scheme.lower() in settings.CLIENT_CALLBACK_FORBIDDEN_SCHEMES:
            raise forms.ValidationError('scheme_forbidden')

        try:
            normalize_url(value)
        except (ValueError, UnicodeError):
            raise forms.ValidationError('invalid')

        return value


class DepartmentGroupField(StrippedCharField):
    def __init__(self, required=True, **kwargs):
        super(DepartmentGroupField, self).__init__(required=required, **kwargs)

    def clean(self, value):
        value = super(DepartmentGroupField, self).clean(value)
        if '|' in value or ':' not in value:
            raise forms.ValidationError('invalid')
        prefix = value.split(':', 1)[0]
        if prefix not in settings.DEPARTMENT_GROUP_PREFIXES:
            raise forms.ValidationError('prefix_invalid')
        return value


class LanguageMixin(forms.Form):
    language = forms.CharField(required=True, error_messages=error_messages)

    def clean_language(self):
        language = self.data['language'].lower()
        if language in settings.LANGUAGES:
            return language
        return settings.LANGUAGE_FALLBACK_MAPPING.get(
            language,
            settings.DEFAULT_LANGUAGE,
        )


class ClientIdMixin(forms.Form):
    client_id = StrippedCharField(required=True, max_length=32, min_length=32, error_messages=error_messages)


class ClientIdOptionalMixin(forms.Form):
    client_id = StrippedCharField(required=False, max_length=32, min_length=32, error_messages=error_messages)


class ClientIdsMixin(forms.Form):
    client_id = StrippedCharField(required=True, max_length=10000, error_messages=error_messages)

    def clean_client_id(self):
        return self.data.getlist('client_id')


class ClientSecretMixin(forms.Form):
    client_secret = StrippedCharField(required=True, max_length=32, min_length=32, error_messages=error_messages)


class RequestIdMixin(forms.Form):
    request_id = StrippedCharField(required=True, max_length=32, min_length=32, error_messages=error_messages)


class TokenIdMixin(forms.Form):
    token_id = forms.fields.IntegerField(required=True, min_value=1, error_messages=error_messages)


class TokenIdsMixin(forms.Form):
    token_id = forms.fields.CharField(required=True, error_messages=error_messages)

    def clean_token_id(self):
        raw_data = self.data.getlist('token_id')

        try:
            token_ids = [int(token_id) for token_id in raw_data]
        except (ValueError, TypeError):
            raise forms.ValidationError('invalid')

        if any(token_id <= 0 for token_id in token_ids):
            raise forms.ValidationError('invalid')

        return token_ids


class RedirectUriOptionalMixin(forms.Form):
    redirect_uri = RedirectUriField(required=False, error_messages=error_messages)


class RedirectUrisOptionalMixin(forms.Form):
    redirect_uri = CollectionField(
        validator=RedirectUriField(error_messages=error_messages),
        max_count=20,
        error_messages=error_messages,
    )


class DeviceIdField(StrippedCharField):
    def __init__(self, required=False, **kwargs):
        super(DeviceIdField, self).__init__(
            required=required,
            min_length=6,
            max_length=50,
            validators=[
                validators.RegexValidator(
                    regex=DEVICE_ID_REGEX,
                ),
            ],
            error_messages=error_messages,
            **kwargs
        )


class DeviceInfoMixin(forms.Form):
    """Умеет собирать device_id и device_name из многочисленных параметров, передаваемых АМом"""

    device_id = DeviceIdField(required=True)
    device_name = StrippedCharField(
        required=True,
        min_length=1,
        error_messages=error_messages,
    )
    uuid = StrippedCharField(required=False, error_messages=error_messages)
    app_id = StrippedCharField(required=False, error_messages=error_messages)
    app_platform = StrippedCharField(required=False, error_messages=error_messages)
    manufacturer = StrippedCharField(required=False, error_messages=error_messages)
    model = StrippedCharField(required=False, error_messages=error_messages)
    app_version = StrippedCharField(required=False, error_messages=error_messages)
    am_version = StrippedCharField(required=False, error_messages=error_messages)

    def __init__(self, data, **kwargs):
        data = MultiValueDict({key: data.getlist(key) for key in data})  # в data может прийти и MergeDict
        device_info = get_device_info(data)
        for key, value in device_info.items():
            data.setlist(key, [value])
        data['device_name'] = data.get('device_name') or data.get('model_name')
        super(DeviceInfoMixin, self).__init__(data, **kwargs)


class DeviceInfoOptionalMixin(DeviceInfoMixin):
    def __init__(self, *args, **kwargs):
        super(DeviceInfoOptionalMixin, self).__init__(*args, **kwargs)
        for field_name in ['device_id', 'device_name']:
            self.fields[field_name].required = False


class CodeChallengeMixin(forms.Form):
    code_challenge = forms.fields.CharField(
        required=False,
        min_length=CODE_CHALLENGE_MIN_LENGTH,
        max_length=CODE_CHALLENGE_MAX_LENGTH,
        error_messages=error_messages,
    )
    code_challenge_method = MappingField(
        required=False,
        mapping={
            CODE_CHALLENGE_METHOD_PLAIN: CodeChallengeMethod.Plain,
            CODE_CHALLENGE_METHOD_S256: CodeChallengeMethod.S256,
        },
        default=CODE_CHALLENGE_METHOD_PLAIN,
        error_messages=error_messages,
    )

    def clean(self):
        cleaned_data = super(CodeChallengeMixin, self).clean()
        if not cleaned_data.get('code_challenge'):
            cleaned_data['code_challenge_method'] = CodeChallengeMethod.NotRequired
        return cleaned_data
