# -*- coding: utf-8 -*-
from django.conf import settings
from passport.backend.core.builders.blackbox.blackbox import BlackboxTemporaryError
from passport.backend.oauth.api.api.iface.errors import (
    BaseApiError,
    ClientApprovalPendingError,
    ClientApprovalRejectedError,
    ClientBlockedError,
    ClientCreatorRequiredError,
    ClientNotEditableError,
    PasswordRequiredError,
    RedirectUriNotMatchedError,
    RedirectUriNotRegisteredError,
    YandexClientDestructiveActionError,
)
from passport.backend.oauth.core.api.base import BaseApiView
from passport.backend.oauth.core.api.errors import ClientNotFoundError
from passport.backend.oauth.core.db.client import (
    ApprovalStatus,
    Client,
    is_redirect_uri_allowed,
)
from passport.backend.oauth.core.db.scope import Scope


class BaseIfaceApiView(BaseApiView):
    check_password_verification_age = False
    check_verification_age_if_have_password = False

    def assert_redirect_uri_is_allowed(self, redirect_uri):
        if not is_redirect_uri_allowed(
            redirect_uri,
            allowed_redirect_uris=(
                self.client.redirect_uris +
                self.client.platform_specific_redirect_uris
            ),
        ):
            raise RedirectUriNotMatchedError()

    def get_client_by_display_id(self, client_id=None, require_creator=False):
        self.client = Client.by_display_id(client_id or self.form_values['client_id'])
        if self.client is None:
            raise ClientNotFoundError()
        if (
            require_creator and
            not self.client.is_created_by(self.uid) and
            not self.client.is_owned_by(self.uid)
        ):
            raise ClientCreatorRequiredError()

    def assert_client_is_allowed(self, redirect_uri=None, ignore_redirect_uri=False):
        """
        Может ли приложение получать токены в вебе. Проверяет:
         - наличие redirect_uri и его допустимость
         - незаблокированность приложения
         - промодерированность, если требуется
        Сохраняет redirect_uri для ответа.
        """
        if not ignore_redirect_uri:
            if not self.client.default_redirect_uri:
                raise RedirectUriNotRegisteredError()

            if redirect_uri:
                self.assert_redirect_uri_is_allowed(redirect_uri)

            self.response_values['redirect_uri'] = redirect_uri or self.client.default_redirect_uri

        if self.client.is_blocked:
            raise ClientBlockedError()

        if self.client.approval_status != ApprovalStatus.NotRequired:
            if self.client.approval_status == ApprovalStatus.Pending:
                raise ClientApprovalPendingError()
            elif self.client.approval_status == ApprovalStatus.Rejected:
                raise ClientApprovalRejectedError()

    def assert_client_is_editable(self):
        """
        Может ли данный пользователь отредактировать это приложение
        """
        if not self.client.can_be_edited(uid=self.uid, user_ip=self.request.env.user_ip):
            raise ClientNotEditableError()

    def process_form_with_scope_arg(self, form_class, show_hidden=False, data=None, files=None, client=None):
        scopes = set(Scope.list(
            show_hidden=show_hidden,
            uid=self.uid,
        ))
        if client is not None:
            scopes.update(client.extra_visible_scopes)
        visible_scope = dict((s.keyword, s.default_title) for s in scopes)

        self.form_values.update(self.process_form(
            form_class,
            form_args={'visible_scopes': visible_scope},
            data=data,
            files=files,
        ))

    def protect_yandex_client_from_destruction(self):
        if self.client.is_yandex and settings.PROTECT_YANDEX_CLIENTS_FROM_DESTRUCTION:
            raise YandexClientDestructiveActionError()

    def get_user(self, optional=False):
        try:
            if self.request.env.authorization:
                self.user_info = self.get_user_from_token()
            else:
                self.user_info = self.get_user_from_session(uid=self.form_values['uid'])

            self._uid = self.user_info['uid']
            if (
                self.check_password_verification_age or
                self.check_verification_age_if_have_password and self.user_info['have_password']
            ):
                password_age = self.user_info['password_verification_age']
                if password_age < 0 or password_age > settings.PASSWORD_VERIFICATION_MAX_AGE:
                    raise PasswordRequiredError()
        except (BaseApiError, BlackboxTemporaryError):
            if optional:
                self._uid = None
            else:
                raise

    @property
    def uid(self):
        return self._uid
