# -*- coding: utf-8 -*-
import re

from django import forms
from django.conf import settings
from django.core import validators
from django.core.exceptions import ValidationError
from passport.backend.oauth.api.api.forms import (
    ClientIdMixin,
    ClientIdOptionalMixin,
    ClientIdsMixin,
    CodeChallengeMixin,
    DepartmentGroupField,
    DeviceIdField,
    DeviceInfoOptionalMixin,
    LanguageMixin,
    LimitedFileField,
    RedirectUriField,
    RedirectUriOptionalMixin,
    RedirectUrisOptionalMixin,
    RequestIdMixin,
    ScopesField,
    TokenIdMixin,
    TokenIdsMixin,
)
from passport.backend.oauth.core.api.forms import (
    BooleanField,
    CollectionField,
    error_messages,
    ExtendedValidatorsMixin,
    StrippedCharField,
    UidField,
    UidMixin,
    UidOptionalMixin,
)
from passport.backend.oauth.core.db.client import (
    PLATFORM_ANDROID,
    PLATFORM_IOS,
    PLATFORM_TURBOAPP,
    PLATFORM_WEB,
)


RESPONSE_TYPE_TOKEN = 'token'
RESPONSE_TYPE_CODE = 'code'

IOS_APP_ID_REGEX = re.compile(r'^[A-Z0-9]{10}\.[A-Za-z0-9.-]+$')
ANDROID_PACKAGE_NAME_REGEX = re.compile(r'^[A-Za-z0-9_.]+$')
ANDROID_FINGERPRINT_REGEX = re.compile(r'^[A-Fa-f0-9:]{95}$')


class UidForm(UidMixin):
    """Форма для uid"""


class BaseApiForm(UidOptionalMixin):
    """Форма для опционального uid"""


class BaseApiFormWithOptionalClient(BaseApiForm, ClientIdOptionalMixin):
    """Форма для опциональных uid и client_id"""


class BaseApiFormWithClient(BaseApiForm, ClientIdMixin):
    """Форма для опционального uid и обязательного client_id"""


class ListClientsForm(BaseApiForm, LanguageMixin):
    """Форма для опционального uid и language"""


class UidLanguageForm(UidMixin, LanguageMixin):
    """Форма для uid и language"""


class LanguageForm(LanguageMixin):
    """Форма для language"""


class ListTokensForm(UidOptionalMixin, LanguageMixin):
    """Форма для uid, language"""


class ClientInfoForm(ClientIdMixin, LanguageMixin, UidOptionalMixin):
    """Форма для client_id, language и опционального uid"""
    force_no_slugs = BooleanField()


class UidClientForm(UidMixin, ClientIdMixin):
    """Форма для uid и client_id"""


class ClientsInfoForm(ClientIdsMixin, LanguageMixin):
    """Форма для нескольких client_id и language"""


class TokenForm(UidMixin, TokenIdMixin):
    """Форма для uid и token_id"""


class TokensForm(BaseApiForm, TokenIdsMixin):
    """Форма для опционального uid и одного или нескольких token_id"""


class UserInfoForm(BaseApiForm, LanguageMixin):
    """Форма для опционального uid и language"""
    force_no_slugs = BooleanField()


class EditClientForm(BaseApiForm, RedirectUrisOptionalMixin, ExtendedValidatorsMixin):
    title = StrippedCharField(required=True, error_messages=error_messages)
    description = StrippedCharField(required=False, max_length=250, error_messages=error_messages)
    scopes = ScopesField(required=True, error_messages=error_messages)
    icon = forms.fields.URLField(
        required=False,
        max_length=1024,
        error_messages=error_messages,
    )
    icon_id = StrippedCharField(required=False, error_messages=error_messages)
    icon_file = LimitedFileField(
        required=False,
        error_messages=error_messages,
    )
    homepage = forms.fields.URLField(
        required=False,
        max_length=1024,
        error_messages=error_messages,
    )
    is_yandex = BooleanField()
    require_platform = BooleanField()
    platforms = forms.MultipleChoiceField(
        choices=((value, value) for value in (PLATFORM_ANDROID, PLATFORM_IOS, PLATFORM_TURBOAPP, PLATFORM_WEB)),
        required=False,
        error_messages=error_messages,
    )
    ios_app_id = CollectionField(
        validator=StrippedCharField(
            required=True,
            validators=[
                validators.RegexValidator(
                    regex=IOS_APP_ID_REGEX,
                ),
            ],
            error_messages=error_messages,
        ),
        max_count=5,
        error_messages=error_messages,
    )
    ios_appstore_url = RedirectUriField(required=False, error_messages=error_messages)
    android_package_name = CollectionField(
        validator=StrippedCharField(
            required=True,
            validators=[
                validators.RegexValidator(
                    regex=ANDROID_PACKAGE_NAME_REGEX,
                ),
            ],
            error_messages=error_messages,
        ),
        max_count=5,
        error_messages=error_messages,
    )
    android_cert_fingerprints = CollectionField(
        validator=forms.CharField(
            required=True,
            validators=[
                validators.RegexValidator(
                    regex=ANDROID_FINGERPRINT_REGEX,
                ),
            ],
            error_messages=error_messages,
        ),
        max_count=5,
        error_messages=error_messages,
    )
    android_appstore_url = RedirectUriField(required=False, error_messages=error_messages)
    turboapp_base_url = RedirectUriField(required=False, error_messages=error_messages)
    owner_uids = CollectionField(
        validator=UidField(error_messages=error_messages),
        max_count=5,
    )
    owner_groups = CollectionField(
        validator=DepartmentGroupField(error_messages=error_messages),
        max_count=5,
    )
    contact_email = forms.EmailField(required=False, error_messages=error_messages)

    def __init__(self, visible_scopes=None, *args, **kwargs):
        super(EditClientForm, self).__init__(*args, **kwargs)
        self.fields['scopes'].choices = (visible_scopes or {}).items()
        self.fields['icon_file'].max_upload_size = settings.MAX_ICON_FILE_SIZE

    def clean_icon_id(self):
        data = self.cleaned_data['icon_id']
        if data and data.count('/') != 1:
            raise ValidationError('invalid')

        return data

    def clean(self):
        cleaned_data = super(EditClientForm, self).clean()
        if (
            cleaned_data['is_yandex'] and
            not (cleaned_data['icon_id'] or cleaned_data.get('icon_file')) and
            not (self.has_error('icon_id') or self.has_error('icon_file'))
        ):
            self.add_error('icon', 'required')

        if cleaned_data.get('require_platform') and not cleaned_data.get('platforms'):
            self.add_error('platforms', 'required')

        # если есть платформа - требуем обязательные для неё поля
        self.require_if_has_value(
            cleaned_data,
            collection_field='platforms',
            value=PLATFORM_ANDROID,
            required_fields=['android_package_name', 'android_cert_fingerprints'],
        )
        self.require_if_has_value(
            cleaned_data,
            collection_field='platforms',
            value=PLATFORM_IOS,
            required_fields=['ios_app_id'],
        )
        self.require_if_has_value(
            cleaned_data,
            collection_field='platforms',
            value=PLATFORM_TURBOAPP,
            required_fields=['turboapp_base_url'],
        )
        self.require_if_has_value(
            cleaned_data,
            collection_field='platforms',
            value=PLATFORM_WEB,
            required_fields=['redirect_uri'],
        )

        # если нет платформы - зануляем все её поля
        if PLATFORM_ANDROID not in cleaned_data.get('platforms', []):
            cleaned_data.update(
                android_package_name=[],
                android_cert_fingerprints=[],
                android_appstore_url='',
            )
        if PLATFORM_IOS not in cleaned_data.get('platforms', []):
            cleaned_data.update(
                ios_app_id=[],
                ios_appstore_url='',
            )
        if PLATFORM_TURBOAPP not in cleaned_data.get('platforms', []):
            cleaned_data.update(
                turboapp_base_url='',
            )
        if PLATFORM_WEB not in cleaned_data.get('platforms', []):
            cleaned_data.update(
                redirect_uri=[],
            )

        return cleaned_data


class AuthorizeSubmitPreliminaryForm(BaseApiFormWithClient, RedirectUriOptionalMixin):
    """
    Форма только для приложения, redirect_uri и response_type. Нужна, чтобы провалидировать redirect_uri
    до валидации всех остальных параметров и удаления параметра code из redirect_uri
    """
    response_type = forms.ChoiceField(
        widget=forms.Select(),
        choices=(
            (RESPONSE_TYPE_CODE, RESPONSE_TYPE_CODE),
            (RESPONSE_TYPE_TOKEN, RESPONSE_TYPE_TOKEN),
        ),
        required=True,
        error_messages=error_messages,
    )


class AuthorizeSubmitForm(AuthorizeSubmitPreliminaryForm,
                          DeviceInfoOptionalMixin,
                          LanguageMixin,
                          CodeChallengeMixin):
    requested_scopes = ScopesField(required=False, error_messages=error_messages)
    state = forms.fields.CharField(required=False, max_length=2048, error_messages=error_messages)
    payment_auth_scheme = forms.fields.CharField(required=False, max_length=20, error_messages=error_messages)
    fingerprint = StrippedCharField(required=False, error_messages=error_messages)

    def __init__(self, visible_scopes=None, *args, **kwargs):
        super(AuthorizeSubmitForm, self).__init__(*args, **kwargs)
        self.fields['requested_scopes'].choices = (visible_scopes or {}).items()


class AuthorizeGetStateForm(BaseApiForm, RequestIdMixin):
    """Форма для uid и request_id"""


class AuthorizeCommitForm(BaseApiForm, RequestIdMixin):
    """Форма для uid, request_id и скоупов"""
    granted_scopes = ScopesField(required=False, error_messages=error_messages)
    payment_auth_retpath = RedirectUriField(required=False, error_messages=error_messages)

    def __init__(self, visible_scopes=None, *args, **kwargs):
        super(AuthorizeCommitForm, self).__init__(*args, **kwargs)
        self.fields['granted_scopes'].choices = (visible_scopes or {}).items()


class IssueAppPasswordForm(BaseApiFormWithClient,
                           DeviceInfoOptionalMixin,
                           LanguageMixin):
    """Форма для uid, client_id, device_id, device_name и language"""
    device_id = DeviceIdField(required=True)


class RevokeDeviceForm(BaseApiForm):
    """Форма для удаления токенов устройства"""
    device_id = DeviceIdField(required=False)


class ListTokenGroupsForm(BaseApiForm, LanguageMixin):
    """Форма выдачи токенов по группам"""
    hide_yandex_device_tokens = BooleanField()


class DeviceAuthorizeForm(BaseApiForm, LanguageMixin, ClientIdOptionalMixin):
    """Форма для процесса подтверждения кода, выданного на другом девайсе"""
    code = StrippedCharField(
        required=True,
        error_messages=error_messages,
    )


__all__ = (
    'UidForm',
    'BaseApiForm',
    'BaseApiFormWithOptionalClient',
    'BaseApiFormWithClient',
    'UidLanguageForm',
    'LanguageForm',
    'ListClientsForm',
    'ListTokensForm',
    'ClientInfoForm',
    'ClientsInfoForm',
    'UidClientForm',
    'TokenForm',
    'TokensForm',
    'UserInfoForm',
    'EditClientForm',
    'AuthorizeSubmitPreliminaryForm',
    'AuthorizeSubmitForm',
    'AuthorizeGetStateForm',
    'AuthorizeCommitForm',
    'IssueAppPasswordForm',
    'RevokeDeviceForm',
    'ListTokenGroupsForm',
    'DeviceAuthorizeForm',
    'RESPONSE_TYPE_CODE',
    'RESPONSE_TYPE_TOKEN',
)
