# -*- coding: utf-8 -*-

from urllib.parse import (
    parse_qs,
    urlencode,
    urlparse,
    urlunparse,
)

from passport.backend.oauth.core.common.blackbox_helpers import try_convert_uids_to_logins
from passport.backend.oauth.core.db.client import (
    PLATFORM_ANDROID,
    PLATFORM_IOS,
    PLATFORM_TURBOAPP,
    PLATFORM_WEB,
)
from passport.backend.oauth.core.db.eav.types import DB_NULL
from passport.backend.oauth.core.db.token import (
    check_if_is_refreshable,
    get_ttl_by_scopes,
)
from passport.backend.utils.time import datetime_to_integer_unixtime


def client_to_response(
    client,
    language='ru',
    for_creator=False,
    group_scopes_to_one_fake_service=False,
    add_token_ttl=False,
):
    result = {
        'id': client.display_id,
        'title': client.get_title(language),
        'icon': client.icon,
        'icon_id': client.icon_id,
        'icon_url': client.get_icon_url(),
        'homepage': client.homepage,
        'callback': client.default_redirect_uri or '',  # для совместимости
        'redirect_uris': client.redirect_uris,
        'description': client.get_description(language),
        'create_time': datetime_to_integer_unixtime(client.created),
        'is_yandex': client.is_yandex,
        'is_deleted': client.is_deleted,
        'scopes': scopes_to_response(
            client.scopes,
            language,
            group_to_one_fake_service=group_scopes_to_one_fake_service,
        ),
        'platforms': {},
    }
    if PLATFORM_IOS in client.platforms:
        result['platforms'][PLATFORM_IOS] = {
            'app_ids': client.ios_app_ids,
            'appstore_url': client.ios_appstore_url,
            'universal_link_domains': client.universal_link_domains,
            'custom_urlschemes': [
                client.platform_specific_urlscheme,
            ],
        }
    if PLATFORM_ANDROID in client.platforms:
        result['platforms'][PLATFORM_ANDROID] = {
            'package_names': client.android_package_names,
            'cert_fingerprints': client.android_cert_fingerprints,
            'appstore_url': client.android_appstore_url,
            'custom_urlschemes': [
                client.platform_specific_urlscheme,
            ],
        }
    if PLATFORM_TURBOAPP in client.platforms:
        result['platforms'][PLATFORM_TURBOAPP] = {
            'base_url': client.turboapp_base_url,
        }
    if PLATFORM_WEB in client.platforms:
        result['platforms'][PLATFORM_WEB] = {
            'redirect_uris': client.redirect_uris,
        }
    if for_creator:
        result.update(
            secret=client.secret,
            approval_status=client.approval_status,
            blocked=client.is_blocked,
            owners=[
                {
                    'uid': uid,
                    'login': login,
                }
                for uid, login in try_convert_uids_to_logins(client.owner_uids).items()
            ],
            owner_groups=list(client.owner_groups),
            contact_email=client.contact_email if client.contact_email else None,
        )
    if add_token_ttl:
        token_ttl = get_ttl_by_scopes(client.scopes)
        result.update(token_ttl=token_ttl)
        if token_ttl:
            result.update(is_ttl_refreshable=check_if_is_refreshable(client.scopes))

    return result


def clients_to_response(clients, language='ru', for_creator=False, add_token_ttl=False):
    return sorted(
        [
            client_to_response(
                client,
                language=language,
                for_creator=for_creator,
                add_token_ttl=add_token_ttl,
            ) for client in clients
        ],
        key=lambda c: c['title'],
    )


def scope_to_response(scope, language='ru', with_slug=False, short=False, with_tags=False):
    scope_dict = {
        'title': scope.get_title(language, with_slug=with_slug, short=short),
        'requires_approval': scope.requires_approval,
        'ttl': scope.ttl,
        'is_ttl_refreshable': scope.is_ttl_refreshable,
    }

    if with_tags:
        scope_dict['tags'] = scope.tags

    return scope_dict


def scopes_to_response(
    scopes,
    language='ru',
    with_slugs=False,
    short=False,
    group_to_one_fake_service=False,
    with_tags=False,
):
    result = {}
    for scope in scopes:
        if group_to_one_fake_service:
            service_title = '-'
        else:
            service_title = scope.get_service_title(language, with_slug=with_slugs)

        result.setdefault(service_title, {})[scope.keyword] = scope_to_response(
            scope,
            language,
            with_slug=with_slugs,
            short=short,
            with_tags=with_tags,
        )
    return result


def token_to_response(token, client, language='ru'):
    if token.client_id != client.id:
        raise ValueError('Token doesn\'t belong to client')
    return {
        'id': token.id,
        'scopes': scopes_to_response(token.scopes, language),
        'client': client_to_response(client, language),
        'device_id': token.device_id if token.device_id != DB_NULL else None,
        'device_name': token.device_name or None,
        'create_time': datetime_to_integer_unixtime(token.created),
        'issue_time': datetime_to_integer_unixtime(token.issued),
        'is_app_password': token.is_app_password,
    }


def account_to_response(bb_response):
    return {
        'uid': bb_response['uid'],
        'login': bb_response['login'],
        'display_login': bb_response['display_login'],
        'display_name': bb_response['display_name']['name'],
        'default_avatar_id': bb_response['display_name'].get('avatar', {}).get('default'),
        'is_avatar_empty': bb_response['display_name'].get('avatar', {}).get('empty', True),
    }


def remove_param_from_url(url, param):
    url_parts = urlparse(url)
    query = parse_qs(url_parts.query)
    if param in query:
        del query[param]
    url_parts = url_parts._replace(query=urlencode(query, doseq=True))
    return urlunparse(url_parts)
