# -*- coding: utf-8 -*-
from datetime import datetime
import logging
from operator import itemgetter

from django.conf import settings
from passport.backend.core.builders.blackbox.utils import get_attribute
from passport.backend.core.builders.money_api import (
    get_money_payment_auth_api,
    MoneyApiUnknownSchemeError,
    PAYMENT_AUTH_SUCCEEDED_STATUS,
)
from passport.backend.core.builders.passport import BasePassportError
from passport.backend.oauth.api.api.iface.base import BaseIfaceApiView
from passport.backend.oauth.api.api.iface.errors import (
    AccessDeniedError,
    AccountIsChildError,
    AppIDNotMatchedError,
    BadIconFormatError,
    ClientNotApplicableError,
    DescriptionMissingError,
    FingerprintNotMatchedError,
    PaymentAuthNotPassedError,
    PaymentAuthRequiredError,
    PaymentAuthRetpathMissingError,
    PaymentAuthSchemeUnknownError,
    RedirectUriInsecureError,
    RedirectUriMissingError,
    RequestedScopesInvalidError,
    RequestedScopesMissingError,
    RequestedScopesTooManyError,
    RequestNotFoundError,
    ScopesNotMatchedError,
    TokenLimitExceededError,
    TokenOwnerRequiredError,
)
from passport.backend.oauth.api.api.iface.forms import (
    AuthorizeCommitForm,
    AuthorizeGetStateForm,
    AuthorizeSubmitForm,
    AuthorizeSubmitPreliminaryForm,
    BaseApiForm,
    BaseApiFormWithClient,
    BaseApiFormWithOptionalClient,
    ClientInfoForm,
    ClientsInfoForm,
    DeviceAuthorizeForm,
    EditClientForm,
    IssueAppPasswordForm,
    LanguageForm,
    ListClientsForm,
    ListTokenGroupsForm,
    ListTokensForm,
    RESPONSE_TYPE_CODE,
    RESPONSE_TYPE_TOKEN,
    RevokeDeviceForm,
    TokensForm,
    UserInfoForm,
)
from passport.backend.oauth.api.api.iface.utils import (
    account_to_response,
    client_to_response,
    clients_to_response,
    remove_param_from_url,
    scopes_to_response,
    token_to_response,
)
from passport.backend.oauth.core.api.errors import (
    BackendFailedError,
    ClientNotFoundError,
)
from passport.backend.oauth.core.common.avatars import (
    AvatarsMdsApiBadImageFormatError,
    AvatarsMdsApiTemporaryError,
    get_avatars_mds_api,
)
from passport.backend.oauth.core.common.blackbox import (
    get_revoke_time_from_bb_response,
    REVOKER_APP_PASSWORDS,
    REVOKER_TOKENS,
    REVOKER_WEB_SESSIONS,
)
from passport.backend.oauth.core.common.constants import (
    GRANT_TYPE_CODE,
    GRANT_TYPE_DEVICE_CODE,
    GRANT_TYPE_FRONTEND_ASSERTION,
    TOKEN_TYPE,
)
from passport.backend.oauth.core.common.error_logs import log_warning
from passport.backend.oauth.core.common.passport import get_passport
from passport.backend.oauth.core.common.portallib import is_yandex_ip
from passport.backend.oauth.core.common.utils import (
    now,
    split_by_predicate,
)
from passport.backend.oauth.core.db.client import (
    Client,
    is_redirect_uri_insecure,
    list_clients_by_creator,
    make_icon_id,
    make_icon_key,
    PLATFORM_ANDROID,
    PLATFORM_IOS,
    PLATFORM_TURBOAPP,
    will_require_approval,
)
from passport.backend.oauth.core.db.device_info import (
    app_platform_to_name,
    AppPlatform,
    parse_app_platform,
)
from passport.backend.oauth.core.db.eav import (
    CREATE,
    DELETE,
    EntityNotFoundError,
    UPDATE,
)
from passport.backend.oauth.core.db.eav.types import DB_NULL
from passport.backend.oauth.core.db.errors import (
    ExpiredRequestError,
    TokenLimitExceededError as TokenLimitExceeded,
    VerificationCodeCollisionError,
    WrongRequestUserError,
)
from passport.backend.oauth.core.db.limits import (
    AccessDeniedError as AccessDenied,
    assert_is_allowed,
    assert_is_not_child,
    assert_is_not_spammer,
)
from passport.backend.oauth.core.db.request import (
    accept_request,
    code_challenge_method_to_name,
    code_strength_to_name,
    CodeStrength,
    does_code_look_valid,
    get_request,
    Request,
)
from passport.backend.oauth.core.db.scope import (
    get_payment_auth_app_ids,
    is_payment_auth_required,
    Scope,
)
from passport.backend.oauth.core.db.token import (
    check_if_is_refreshable,
    delete_single_token,
    get_already_granted_scopes,
    get_ttl_by_scopes,
    issue_token,
    list_app_passwords,
    list_tokens_by_user_and_device,
    list_tokens_with_clients_by_user,
    Token,
)
from passport.backend.oauth.core.db.token.normal_token import split_tokens_to_devices
from passport.backend.oauth.core.db.turboapp import is_redirect_uri_turboapp
from passport.backend.oauth.core.logs.historydb import to_event_log
from passport.backend.oauth.core.logs.statbox import (
    StatboxLogger,
    to_statbox,
)


BASE_GRANT = 'iface_api.base'
ACCEPT_DEVICE_CODES_GRANT = 'iface_api.accept_device_codes'
ISSUE_TOKENS_GRANT = 'iface_api.issue_tokens'
ISSUE_APP_PASSWORDS_GRANT = 'iface_api.issue_app_passwords'
LIST_TOKENS_GRANT = 'iface_api.list_tokens'
REVOKE_TOKENS_GRANT = 'iface_api.revoke_tokens'
VIEW_CLIENTS_GRANT = 'iface_api.view_clients'
EDIT_CLIENTS_GRANT = 'iface_api.edit_clients'


log = logging.getLogger('api')


class Settings(BaseIfaceApiView):
    """
    Настройки, необходимые фронтенду
    """
    required_grants = [BASE_GRANT]
    allowed_methods = ['GET']
    base_form = None

    def process_request(self, request):
        self.response_values.update(
            invalidate_tokens_on_callback_change=settings.INVALIDATE_TOKENS_ON_CLIENT_CALLBACK_CHANGE,
            x_token_scopes=[
                scope.keyword
                for scope in Scope.list(show_hidden=True)
                if scope.has_xtoken_grant
            ],
        )


class AllScopes(BaseIfaceApiView):
    """
    Список всех скоупов, включая скрытые.
    """
    required_grants = [BASE_GRANT]
    allowed_methods = ['GET']
    base_form = LanguageForm

    def process_request(self, request):
        language = self.form_values['language']
        scopes = Scope.list()
        self.response_values['all_scopes'] = scopes_to_response(scopes, language)


class ClientsInfo(BaseIfaceApiView):
    """
    Возвращает общую информацию о приложениях (в том числе, удалённых)
    """
    required_grants = [VIEW_CLIENTS_GRANT]
    allowed_methods = ['GET']
    base_form = ClientsInfoForm

    def process_request(self, request):
        clients_info = {}
        language = self.form_values['language']

        for client_id in self.form_values['client_id']:
            client = Client.by_display_id(client_id, allow_deleted=True)
            if client is None:
                clients_info.update({client_id: None})
            else:
                clients_info.update(
                    {
                        client_id: client_to_response(client, language, False),
                    },
                )
        self.response_values['clients'] = clients_info


class RevokeToken(BaseIfaceApiView):
    """
    Отзывает указанные токены. Защищена вводом пароля для имеющих пароль аккаунтов.
    """
    required_grants = [REVOKE_TOKENS_GRANT]
    allowed_methods = ['POST']
    base_form = TokensForm
    check_verification_age_if_have_password = True

    def process_request(self, request):
        self.get_user()
        datetime_now = datetime.now()
        for token_id in self.form_values['token_id']:
            try:
                token = Token.by_id(token_id)
                if token.uid != self.uid:
                    raise TokenOwnerRequiredError()
                client = token.get_client()  # TODO: избавиться от лишнего запроса
            except EntityNotFoundError:
                continue  # приложения или токена нет - токен уже невалиден

            delete_single_token(token, client, request.env, reason='iface_revoke_token', dt=datetime_now)


class AuthorizeSubmit(BaseIfaceApiView):
    """
    Начало процесса выдачи токена (кода)
    """
    required_grants = [ISSUE_TOKENS_GRANT]
    allowed_methods = ['POST']
    base_form = AuthorizeSubmitPreliminaryForm  # остальное позже

    def _ios_app_id_matches_client(self, app_id):
        for trusted_app_id in self.client.ios_app_ids:
            # В iOS нет возможности получить префикс (aka Team ID) запустившего приложения. Поэтому сейчас iOS передаёт
            # app_id без префикса. Но в настройках приложения лежат app_id с префиксом - надо его отрезать.
            trusted_app_id_without_prefix = trusted_app_id.split('.', 1)[-1]
            if app_id in {trusted_app_id, trusted_app_id_without_prefix}:
                return True
        return False

    def validate_auth_sdk_params(self):
        """
        Если запрос пришёл из AuthSDK - сверим переданные параметры с настройками приложения.

        N.B. AuthSDK передаёт app_id дважды: в query - от приложения, в котором происходит авторизация;
         в теле запроса - от вызвавшего приложения. Проверять надо только второй.
        """
        app_platform = parse_app_platform(self.form_values['app_platform'])
        statbox_logger = StatboxLogger(
            mode='validate_auth_sdk_params',
            client_id=self.client.display_id,
            client_creator=self.client.uid,
            app_platform=app_platform_to_name(app_platform),
        )
        error_class = None
        are_params_passed = True

        if app_platform == AppPlatform.Android:
            if not (
                'fingerprint' in self.request.POST and self.form_values['fingerprint'] and
                'app_id' in self.request.POST and self.form_values['app_id']
            ):
                are_params_passed = False
            elif self.form_values['fingerprint'] not in self.client.android_cert_fingerprints:
                error_class = FingerprintNotMatchedError
                statbox_logger.bind(passed_value=self.form_values['fingerprint'])
            elif self.form_values['app_id'] not in self.client.android_package_names:
                error_class = AppIDNotMatchedError
                statbox_logger.bind(passed_value=self.form_values['app_id'])
        elif app_platform == AppPlatform.iOS:
            if not ('app_id' in self.request.POST and self.form_values['app_id']):
                are_params_passed = False
            elif not self._ios_app_id_matches_client(self.form_values['app_id']):
                error_class = AppIDNotMatchedError
                statbox_logger.bind(passed_value=self.form_values['app_id'])

        if error_class is not None:
            if settings.AUTH_SDK_PARAM_VALIDATION_DRY_RUN:
                statbox_logger.log(status='warning', error=error_class.errors[0])
            else:
                statbox_logger.log(status='error', error=error_class.errors[0])
                raise error_class()
        elif not are_params_passed:
            statbox_logger.log(status='no_data')
        else:
            statbox_logger.log(status='ok')

    def process_request(self, request):
        self.get_user()
        self.response_values.update(
            account=account_to_response(self.user_info['bb_response']),
        )
        self.get_client_by_display_id()

        is_token_response = self.form_values['response_type'] == RESPONSE_TYPE_TOKEN
        redirect_uri = self.form_values['redirect_uri'] or self.client.default_redirect_uri
        if not is_token_response:
            redirect_uri = remove_param_from_url(url=redirect_uri, param='code')

        self.assert_client_is_allowed(redirect_uri=redirect_uri)

        self.process_form_with_scope_arg(
            AuthorizeSubmitForm,
            data=request.REQUEST,
            files=request.FILES,
            show_hidden=True,
        )

        is_auth_sdk = is_turboapp = False
        if is_redirect_uri_turboapp(redirect_uri):
            is_turboapp = True
        elif parse_app_platform(self.form_values['app_platform']) in (AppPlatform.Android, AppPlatform.iOS):
            is_auth_sdk = True

        self.response_values.update(
            client=client_to_response(
                self.client,
                language=self.form_values['language'],
                for_creator=False,
                group_scopes_to_one_fake_service=is_turboapp,  # фикс уязвимости в коде АМ iOS (MOBDEVAUTH-7712)
            ),
        )

        requested_scopes = self.form_values['requested_scopes']
        if is_turboapp:
            if not self.form_values['redirect_uri']:
                raise RedirectUriMissingError()

            if not requested_scopes:
                raise RequestedScopesMissingError()
            elif len(requested_scopes) > settings.TURBO_APP_MAX_SCOPE_COUNT:
                raise RequestedScopesTooManyError()
            elif any([not scope.allowed_for_turboapps for scope in requested_scopes]):
                raise RequestedScopesInvalidError()

        scopes = requested_scopes or self.client.scopes
        if not scopes.issubset(self.client.scopes):
            raise ScopesNotMatchedError()
        try:
            assert_is_not_spammer(
                self.user_info['bb_response'],
                grant_type=GRANT_TYPE_CODE,
                client=self.client,
                ip=request.env.user_ip,
            )
            assert_is_allowed(
                scope_list=scopes,
                grant_type=GRANT_TYPE_CODE,
                client_id=self.client.display_id,
                ip=request.env.user_ip,
                uid=self.uid,
            )
        except AccessDenied:
            raise AccessDeniedError()

        try:
            assert_is_not_child(
                bb_response=self.user_info['bb_response'],
                grant_type=GRANT_TYPE_CODE,
                client=self.client,
                ip=request.env.user_ip,
            )
        except AccessDenied:
            raise AccountIsChildError()

        if (
            not settings.ALLOW_ISSUE_TOKEN_FOR_INSECURE_REDIRECT_URI and
            is_redirect_uri_insecure(redirect_uri) and
            (
                (is_token_response and self.client.created > settings.ALLOW_INSECURE_URIS_FOR_CLIENTS_CREATED_BEFORE) or
                self.form_values['code_challenge']
            )
        ):
            raise RedirectUriInsecureError()

        if is_auth_sdk:
            self.validate_auth_sdk_params()

        already_granted_scopes = get_already_granted_scopes(
            uid=self.uid,
            client=self.client,
            device_id=self.form_values['device_id'],
            revoke_time=get_revoke_time_from_bb_response(self.user_info['bb_response'], REVOKER_TOKENS),
        )
        scopes_to_issue = scopes | already_granted_scopes

        with CREATE(Request.create(
            uid=self.uid,
            client=self.client,
            scopes=scopes_to_issue,
            redirect_uri=redirect_uri,
            is_token_response=is_token_response,
            state=self.form_values['state'],
            device_id=self.form_values['device_id'],
            device_name=self.form_values['device_name'],
        )) as token_request:
            token_request.code_challenge = self.form_values['code_challenge']
            token_request.code_challenge_method = self.form_values['code_challenge_method']
            token_request.payment_auth_scheme = self.form_values['payment_auth_scheme']
            token_request.login_id = self.user_info['login_id']
        self.response_values['request_id'] = token_request.display_id

        requested_scopes = scopes_to_issue - already_granted_scopes
        # Подтверждение пользователя запрашиваем, если...
        require_user_confirm = (
            # ... появились новые скоупы (и приложение не в вайтлисте)
            (
                bool(requested_scopes) and
                self.client.display_id not in settings.SKIP_USER_CONFIRM_FOR_CLIENTS
            ) or
            # ... есть риск утечки токена в зловредное приложение, использующее не свою урлсхему
            redirect_uri.startswith(self.client.platform_specific_urlscheme) or
            # ... есть риск утечки токена в зловредное приложение из-за бага в одной из версий AuthSDK
            is_auth_sdk
        )

        self.response_values.update(
            require_user_confirm=require_user_confirm,
            already_granted_scopes=scopes_to_response(
                already_granted_scopes,
                self.form_values['language'],
                short=is_turboapp,
                group_to_one_fake_service=is_turboapp,  # фикс уязвимости в коде АМ iOS (MOBDEVAUTH-7712)
            ),
            requested_scopes=scopes_to_response(
                requested_scopes,
                self.form_values['language'],
                short=is_turboapp,
                group_to_one_fake_service=is_turboapp,  # фикс уязвимости в коде АМ iOS (MOBDEVAUTH-7712)
            ),
        )
        to_statbox(
            mode='issue_request',
            status='ok',
            reason='iface_authorize',
            response_type=self.form_values['response_type'],
            token_request_id=token_request.display_id,
            uid=token_request.uid,
            client_id=self.client.display_id,
            redirect_uri=redirect_uri,
            scopes=','.join(map(str, token_request.scopes)),
            user_ip=request.env.user_ip,
            user_agent=request.env.user_agent,
            yandexuid=request.env.cookies.get('yandexuid'),
        )


class AuthorizeGetState(BaseIfaceApiView):
    """
    Получение состояния текущего процесса выдачи токена (кода)
    """
    required_grants = [ISSUE_TOKENS_GRANT]
    allowed_methods = ['GET', 'POST']
    base_form = AuthorizeGetStateForm

    def get_request_and_client(self):
        try:
            self.token_request = get_request(display_id=self.form_values['request_id'], uid=self.uid)
        except (ExpiredRequestError, WrongRequestUserError):
            raise RequestNotFoundError()

        if self.token_request.is_invalidated_by_user_logout(
            revoke_time=get_revoke_time_from_bb_response(self.user_info['bb_response'], REVOKER_WEB_SESSIONS),
        ):
            raise RequestNotFoundError()

        try:
            self.client = self.token_request.get_client()
        except EntityNotFoundError:
            raise ClientNotFoundError()
        self.assert_client_is_allowed(redirect_uri=self.token_request.redirect_uri)
        self.response_values.update(
            response_type=RESPONSE_TYPE_TOKEN if self.token_request.is_token_response else RESPONSE_TYPE_CODE,
            redirect_uri=self.token_request.redirect_uri,
        )

    def process_request(self, request):
        self.get_user()
        self.get_request_and_client()


class AuthorizeCommit(AuthorizeGetState):
    """
    Завершение процесса выдачи токена (кода)
    """
    required_grants = [ISSUE_TOKENS_GRANT]
    allowed_methods = ['POST']
    base_form = BaseApiForm  # остальное вручную

    def process_request(self, request):
        self.get_user()
        self.process_form_with_scope_arg(
            AuthorizeCommitForm,
            data=request.REQUEST,
            files=request.FILES,
            show_hidden=True,
        )
        self.get_request_and_client()

        scopes = self.form_values.get('granted_scopes') or self.token_request.scopes
        if not scopes.issubset(self.token_request.scopes):
            raise ScopesNotMatchedError()

        payment_auth_scope_addendum = None
        if is_payment_auth_required(scopes):
            money_payment_auth_api = get_money_payment_auth_api()
            if not self.token_request.payment_auth_context_id:
                if not self.form_values['payment_auth_retpath']:
                    raise PaymentAuthRetpathMissingError()

                try:
                    rv = money_payment_auth_api.create_auth_context(
                        uid=self.uid,
                        client_id=self.client.display_id,
                        scopes=[s.keyword for s in scopes],
                        retpath=self.form_values['payment_auth_retpath'],
                        request_id=self.token_request.display_id,
                        scheme=self.token_request.payment_auth_scheme,
                    )
                except MoneyApiUnknownSchemeError:
                    raise PaymentAuthSchemeUnknownError()
                payment_auth_context_id = rv['authContextId']
                payment_auth_url = rv['redirectUri']

                with UPDATE(self.token_request):
                    # Запоминаем набор скоупов, который пользователь подтвердил - потом его взять будет неоткуда
                    self.token_request.set_scopes(scopes)
                    self.token_request.payment_auth_context_id = payment_auth_context_id

                self.response_values.update(
                    payment_auth_context_id=payment_auth_context_id,
                    payment_auth_app_ids=get_payment_auth_app_ids(scopes),
                    payment_auth_url=payment_auth_url,
                )
                raise PaymentAuthRequiredError()
            else:
                rv = money_payment_auth_api.check_auth_context(
                    auth_context_id=self.token_request.payment_auth_context_id,
                    scheme=self.token_request.payment_auth_scheme,
                )
                if not (
                    rv['status'] == PAYMENT_AUTH_SUCCEEDED_STATUS and
                    rv['passportRequestId'] == self.token_request.display_id
                ):
                    raise PaymentAuthNotPassedError()
                payment_auth_scope_addendum = rv['scopeAddendum']

        if self.token_request.is_token_response:
            # запросили токен - отдаём токен
            try:
                token = issue_token(
                    self.uid,
                    self.client,
                    grant_type=GRANT_TYPE_CODE,
                    env=request.env,
                    device_id=self.token_request.device_id,
                    device_name=self.token_request.device_name,
                    scopes=scopes,
                    tokens_revoked_at=get_revoke_time_from_bb_response(
                        self.user_info['bb_response'],
                        REVOKER_TOKENS,
                    ),
                    allow_reuse=True,  # важно для турбоаппов, см. PASSP-27185
                    allow_fallback_to_stateless=(
                        self.client.is_yandex or
                        settings.ALLOW_FALLBACK_TO_STATELESS_TOKENS_FOR_NON_YANDEX_CLIENTS_AND_PUBLIC_GRANT_TYPES
                    ),
                    payment_auth_context_id=self.token_request.payment_auth_context_id,
                    payment_auth_scope_addendum=payment_auth_scope_addendum,
                    login=self.user_info['bb_response']['login'],
                    login_id=self.token_request.login_id,
                    redirect_uri=self.token_request.redirect_uri,
                )
                self.response_values.update({
                    'state': self.token_request.state,
                    'access_token': token.access_token,
                    'token_type': TOKEN_TYPE,
                    'expires_in': token.ttl,
                })
                if token.scopes != self.token_request.scopes:
                    self.response_values.update(
                        scope=' '.join(map(str, token.scopes)),
                    )
            except AccessDenied:
                raise AccessDeniedError()
            finally:
                with DELETE(self.token_request):
                    pass
        else:
            # запросили код - отдаём код
            try:
                token_request = accept_request(
                    request=self.token_request,
                    scopes=scopes,
                    payment_auth_scope_addendum=payment_auth_scope_addendum,
                )
            except VerificationCodeCollisionError:
                raise BackendFailedError()
            self.response_values.update({
                'state': token_request.state,
                'code': token_request.code,
            })
            to_statbox(
                mode='issue_code',
                status='ok',
                token_request_id=token_request.display_id,
                code_strength=code_strength_to_name(token_request.code_strength),
                code_challenge_method=code_challenge_method_to_name(token_request.code_challenge_method),
                uid=token_request.uid,
                client_id=self.client.display_id,
                scopes=','.join(map(str, scopes)),
                user_ip=request.env.user_ip,
                user_agent=request.env.user_agent,
                yandexuid=request.env.cookies.get('yandexuid'),
            )


class BaseEditClientView(BaseIfaceApiView):
    required_grants = [EDIT_CLIENTS_GRANT]

    def process_request(self, request):
        self.get_user()
        self.client = None
        if self.form_values.get('client_id'):
            self.get_client_by_display_id(require_creator=True)
            self.assert_client_is_editable()

        self.process_form_with_scope_arg(EditClientForm, data=request.REQUEST, files=request.FILES, client=self.client)

        if self.client is not None:
            redirect_uris_added = bool(
                set(self.form_values['redirect_uri']) - set(self.client.redirect_uris) or
                (
                    PLATFORM_IOS not in self.client.platforms and
                    self.form_values['ios_app_id']
                ) or (
                    PLATFORM_ANDROID not in self.client.platforms and
                    self.form_values['android_package_name']
                ) or (
                    PLATFORM_TURBOAPP not in self.client.platforms and
                    self.form_values['turboapp_base_url']
                )
            )
            scopes_removed = not self.client.scopes.issubset(self.form_values['scopes'])
            # если сузился набор scope, то будут инвалидированы все токены приложения
            self.response_values.update(
                redirect_uris_added=redirect_uris_added,
                invalidate_tokens=scopes_removed,
            )

        will_approval_be_required = will_require_approval(self.form_values['scopes'], client=self.client)
        if (
            will_approval_be_required and
            settings.REQUIRE_DESCRIPTION_FOR_CLIENTS_WAITING_FOR_APPROVAL and
            not self.form_values['description']
        ):
            raise DescriptionMissingError()
        self.response_values.update(
            will_require_approval=will_approval_be_required,
        )

    def upload_icon(self):
        key = make_icon_key(client_id=self.client.display_id)
        try:
            group_id = get_avatars_mds_api().upload_from_file(
                key=key,
                file_=self.request.FILES['icon_file'],
            )
        except AvatarsMdsApiBadImageFormatError:
            raise BadIconFormatError()
        return make_icon_id(group_id, key)

    @property
    def can_manage_yandex_clients(self):
        return bool(
            get_attribute(self.user_info['bb_response'], settings.BB_ATTR_ACCOUNT_IS_CORPORATE) and
            is_yandex_ip(self.request.env.user_ip)
        )

    def fill_platform_data(self, client):
        # web
        client._redirect_uris = self.form_values['redirect_uri']
        client._callback = None
        # ios
        client.ios_default_app_id = self.form_values['ios_app_id'][0] if self.form_values['ios_app_id'] else ''
        client.ios_extra_app_ids = self.form_values['ios_app_id'][1:]
        client.ios_appstore_url = self.form_values['ios_appstore_url']
        # android
        client.android_default_package_name = self.form_values['android_package_name'][0] if self.form_values['android_package_name'] else ''
        client.android_extra_package_names = self.form_values['android_package_name'][1:]
        client.android_cert_fingerprints = self.form_values['android_cert_fingerprints']
        client.android_appstore_url = self.form_values['android_appstore_url']
        # turboapp
        client.turboapp_base_url = self.form_values['turboapp_base_url']


class CreateClient(BaseEditClientView):
    """
    Создаёт новое приложение
    """
    allowed_methods = ['POST']
    base_form = BaseApiForm  # остальное вручную

    def process_request(self, request):
        super(CreateClient, self).process_request(request)

        with CREATE(Client.create(
            uid=self.uid,
            redirect_uris=self.form_values['redirect_uri'],
            scopes=self.form_values['scopes'],
            default_title=self.form_values['title'],
            default_description=self.form_values['description'],
            icon=self.form_values['icon'],
            icon_id=None,
            homepage=self.form_values['homepage'],
        )) as self.client:
            self.fill_platform_data(self.client)
            self.client.owner_uids = self.form_values['owner_uids']
            self.client.owner_groups = self.form_values['owner_groups']
            if self.can_manage_yandex_clients:
                self.client.is_yandex = bool(self.form_values['is_yandex'])
            if 'icon_file' in request.FILES:
                self.client.icon_id = self.upload_icon()
            self.client.contact_email = self.form_values['contact_email']

        self.response_values.update(client_id=self.client.display_id)

        to_statbox(
            mode='create_client',
            status='ok',
            client_id=self.client.display_id,
            creator_uid=self.client.uid,
            client_title=self.form_values['title'],
            client_description=self.form_values['description'],
            client_scopes=','.join(s.keyword for s in self.form_values['scopes']),
            client_redirect_uris=', '.join(self.form_values['redirect_uri']),
        )


class ValidateClientChanges(BaseEditClientView):
    """
    Валидация данных перед созданием/редактированием приложения
    """
    allowed_methods = ['POST']
    base_form = BaseApiFormWithOptionalClient  # остальное вручную

    def process_request(self, request):
        super(ValidateClientChanges, self).process_request(request)

        token_ttl = get_ttl_by_scopes(self.form_values['scopes'])
        self.response_values.update(
            token_ttl=token_ttl,
        )
        if token_ttl:
            self.response_values.update(
                is_ttl_refreshable=check_if_is_refreshable(self.form_values['scopes']),
            )


class EditClient(BaseEditClientView):
    """
    Редактирует существующее приложение
    """
    allowed_methods = ['POST']
    base_form = BaseApiFormWithClient  # остальное вручную

    def process_request(self, request):
        super(EditClient, self).process_request(request)

        invalidate_tokens = self.response_values['invalidate_tokens']
        redirect_uris_added = self.response_values['redirect_uris_added']
        old_title = self.client.default_title
        old_icon_id = self.client.icon_id
        old_scopes = self.client.scopes
        old_redirect_uris = self.client.redirect_uris

        if 'icon_file' in request.FILES:
            icon_id = self.upload_icon()
        else:
            icon_id = self.form_values['icon_id']

        with UPDATE(self.client) as client:
            client.modified = now()
            client.default_title = self.form_values['title']
            client.icon = self.form_values['icon']
            client.icon_id = icon_id
            client.homepage = self.form_values['homepage']
            client.default_description = self.form_values['description']
            if invalidate_tokens:
                self.protect_yandex_client_from_destruction()
                client.glogout()
            client.set_scopes(self.form_values['scopes'])
            self.fill_platform_data(client)
            client.owner_uids = self.form_values['owner_uids']
            client.owner_groups = self.form_values['owner_groups']
            if self.can_manage_yandex_clients:
                client.is_yandex = bool(self.form_values['is_yandex'])
            client.contact_email = self.form_values['contact_email']

        if redirect_uris_added:
            if settings.SEND_EMAILS_ON_CLIENT_EDIT:
                try:
                    get_passport().oauth_client_edited_send_notifications(
                        uid=self.uid,
                        client_id=client.display_id,
                        client_title=old_title,
                        redirect_uris_changed=redirect_uris_added,
                        scopes_changed=set(self.client.scopes) != set(old_scopes),
                        user_ip=request.env.user_ip,
                        cookies=request.env.raw_cookies,
                        host=request.env.host,
                    )
                except BasePassportError as e:
                    log_warning('Failed to send email via passport: %s' % e)

        if old_icon_id and old_icon_id != icon_id:
            # Попытаемся убрать за собой мусор. Не получится - не беда, хранилище большое.
            try:
                old_group_id, old_key = old_icon_id.split('/')
                get_avatars_mds_api().delete(group_id=old_group_id, key=old_key)
            except AvatarsMdsApiTemporaryError:
                pass

        to_statbox(
            mode='edit_client',
            status='ok',
            client_id=self.form_values['client_id'],
            client_title=self.form_values['title'],
            client_description=self.form_values['description'],
            client_scopes=','.join(s.keyword for s in self.form_values['scopes']),
            client_redirect_uris=', '.join(self.form_values['redirect_uri']),
        )
        if invalidate_tokens:
            to_statbox(
                mode='invalidate_tokens',
                status='ok',
                target='client',
                client_id=client.display_id,
            )
            to_event_log(
                action='change',
                target='client',
                client_id=self.client.display_id,
                old_scopes=','.join(s.keyword for s in old_scopes),
                new_scopes=','.join(s.keyword for s in self.form_values['scopes']),
                old_redirect_uris=', '.join(old_redirect_uris),
                new_redirect_uris=', '.join(self.form_values['redirect_uri']),
            )


class DeleteClient(BaseEditClientView):
    """
    Удаляет приложение
    """
    required_grants = [EDIT_CLIENTS_GRANT]
    allowed_methods = ['POST']
    base_form = BaseApiFormWithClient
    check_password_verification_age = True

    def process_request(self, request):
        self.get_user()
        self.get_client_by_display_id(require_creator=True)
        self.assert_client_is_editable()
        self.protect_yandex_client_from_destruction()
        with UPDATE(self.client) as client:
            client.deleted = now()
        to_statbox(
            mode='invalidate_tokens',
            status='ok',
            target='client',
            client_id=self.form_values['client_id'],
        )
        to_statbox(
            mode='delete_client',
            status='ok',
            client_id=self.form_values['client_id'],
        )
        to_event_log(
            action='delete',
            target='client',
            client_id=self.form_values['client_id'],
            scopes=','.join(s.keyword for s in self.client.scopes),
        )


class RevokeDeviceTokens(BaseIfaceApiView):
    required_grants = [REVOKE_TOKENS_GRANT]
    allowed_methods = ['POST']
    base_form = RevokeDeviceForm

    check_verification_age_if_have_password = True

    def process_request(self, request):
        self.get_user()
        datetime_now = datetime.now()
        device_tokens_with_clients = list_tokens_by_user_and_device(
            self.uid,
            self.form_values['device_id'],
            revoked_at=get_revoke_time_from_bb_response(
                self.user_info['bb_response'],
                REVOKER_TOKENS,
            ),
        )
        # TODO: удаление одной транзакцией
        for token, client in device_tokens_with_clients:
            delete_single_token(
                token,
                client,
                request.env,
                reason='iface_revoke_device_tokens',
                dt=datetime_now,
            )


class ListTokens(BaseIfaceApiView):
    """
    Возвращает список токенов и ПП пользователя (валидных и протухших, вместе с приложениями)
    """
    required_grants = [LIST_TOKENS_GRANT]
    allowed_methods = ['GET']
    base_form = ListTokensForm

    def process_request(self, request):
        self.get_user()
        language = self.form_values['language']
        all_tokens_with_clients = [
            (is_valid, token_to_response(token, client, language))
            for token, client, is_valid in list_tokens_with_clients_by_user(
                uid=self.uid,
                tokens_revoked_at=get_revoke_time_from_bb_response(
                    self.user_info['bb_response'],
                    REVOKER_TOKENS,
                ),
                app_passwords_revoked_at=get_revoke_time_from_bb_response(
                    self.user_info['bb_response'],
                    REVOKER_APP_PASSWORDS,
                ),
            )
            if is_valid or token.is_app_password  # из протухших нам нужны только ПП
        ]
        valid_tokens, expired_tokens = split_by_predicate(
            sequence=all_tokens_with_clients,
            predicate=itemgetter(0),
            item_getter=itemgetter(1),
        )
        self.response_values.update(
            tokens=valid_tokens,
            expired_tokens=expired_tokens,
        )


class ListTokenGroups(BaseIfaceApiView):
    """
    Возвращает список валидных токенов и ПП пользователя, разбитых по группам:
        * принадлежащие одному устройству;
        * приложения без device_id;
        * пароли приложений.
    """
    required_grants = [LIST_TOKENS_GRANT]
    allowed_methods = ['GET']
    base_form = ListTokenGroupsForm

    def should_show_device_token(self, token):
        if self.form_values['hide_yandex_device_tokens']:
            # показываем только токены, не привязанные к х-токену
            return token.x_token_id is None
        else:
            # показываем всё
            return True

    def fill_device_tokens_to_response(self, device_tokens, language):
        """
        :param device_tokens: [{'token': <token_data>, 'client': <client_data>}]
        :type device_tokens: list
        :type language: basestring
        """
        device_infos = split_tokens_to_devices(
            tokens_with_clients=[
                (token_client_dict['token'], token_client_dict['client'])
                for token_client_dict in device_tokens
            ]
        )
        self.response_values['device_tokens'] = [
            {
                'device_id': device_info['device_id'],
                'device_name': device_info['device_name'],
                'app_platform': device_info['app_platform'],
                'has_xtoken': device_info['has_xtoken'],
                'tokens': [
                    token_to_response(token, client, language)
                    for token, client in device_info['tokens_with_clients']
                    if self.should_show_device_token(token)
                ],
            } for device_info in device_infos
        ]

    def get_split_tokens_to_groups(self):
        """
        :rtype: dict
        """
        all_tokens = list_tokens_with_clients_by_user(
            self.uid,
            tokens_revoked_at=get_revoke_time_from_bb_response(
                self.user_info['bb_response'],
                REVOKER_TOKENS,
            ),
            app_passwords_revoked_at=get_revoke_time_from_bb_response(
                self.user_info['bb_response'],
                REVOKER_APP_PASSWORDS,
            ),
        )
        result = {
            'app_passwords': [],
            'device_tokens': [],
            'other_tokens': [],
        }

        for token, client, is_valid in all_tokens:
            if not is_valid:
                continue
            if token.is_app_password:
                result['app_passwords'].append({'token': token, 'client': client})
            elif token.device_id != DB_NULL:
                result['device_tokens'].append({'token': token, 'client': client})
            else:
                result['other_tokens'].append({'token': token, 'client': client})
        return result

    def process_request(self, request):
        self.get_user()
        language = self.form_values['language']
        token_groups = self.get_split_tokens_to_groups()

        device_tokens = token_groups.pop('device_tokens', [])
        self.fill_device_tokens_to_response(device_tokens, language)

        result = {}
        for token_type, tokens in token_groups.items():
            result[token_type] = [token_to_response(token['token'], token['client'], language) for token in tokens]
        self.response_values.update(**result)


class NewClientSecret(BaseIfaceApiView):
    """
    Создаёт новый секрет для приложения
    """
    required_grants = [EDIT_CLIENTS_GRANT]
    allowed_methods = ['POST']
    base_form = BaseApiFormWithClient
    check_password_verification_age = True

    def process_request(self, request):
        self.get_user()
        self.get_client_by_display_id(require_creator=True)
        self.assert_client_is_editable()
        self.protect_yandex_client_from_destruction()
        with UPDATE(self.client) as client:
            client.make_new_secret()
            client.modified = now()
        self.response_values['secret'] = self.client.secret


class UndoNewClientSecret(BaseIfaceApiView):
    """
    Устанавливает старый секрет для приложения взамен свежесозданного
    """
    required_grants = [EDIT_CLIENTS_GRANT]
    allowed_methods = ['POST']
    base_form = BaseApiFormWithClient
    check_password_verification_age = True

    def process_request(self, request):
        self.get_user()
        self.get_client_by_display_id(require_creator=True)
        self.assert_client_is_editable()
        self.protect_yandex_client_from_destruction()
        with UPDATE(self.client) as client:
            client.restore_old_secret()
            client.modified = now()
        self.response_values['secret'] = self.client.secret


class GlogoutClient(BaseIfaceApiView):
    """
    Отзывает все токены данного приложения
    """
    required_grants = [EDIT_CLIENTS_GRANT]
    allowed_methods = ['POST']
    base_form = BaseApiFormWithClient
    check_password_verification_age = True

    def process_request(self, request):
        self.get_user()
        self.get_client_by_display_id(require_creator=True)
        self.assert_client_is_editable()
        self.protect_yandex_client_from_destruction()
        with UPDATE(self.client) as client:
            client.glogout()
            client.modified = now()


class IssueAppPassword(BaseIfaceApiView):
    """
    Выдаёт токен с алиасом - "пароль для приложений". Используется grant_type=frontend_assertion.
    Защищена вводом пароля для имеющих пароль аккаунтов.
    """
    allowed_methods = ['POST']
    base_form = IssueAppPasswordForm
    required_grants = [ISSUE_APP_PASSWORDS_GRANT]
    check_verification_age_if_have_password = True

    SCOPES_TO_TYPE_MAPPING = (
        ({'app_password:smtp', 'app_password:imap', 'app_password:pop3'}, 'mail'),
        ({'app_password:webdav'}, 'disk'),
        ({'app_password:calendar'}, 'calendar'),
        ({'app_password:carddav'}, 'abook'),
        ({'app_password:carddav', 'app_password:calendar'}, 'abook'),
        ({'app_password:xmpp'}, 'chat'),
        ({'app_password:magnitola'}, 'magnitola'),
    )

    def get_app_type(self):
        for scope_keywords, app_type in self.SCOPES_TO_TYPE_MAPPING:
            client_scope_keywords = set([
                scope.keyword
                for scope in self.client.scopes
            ])
            if client_scope_keywords == scope_keywords:
                return app_type
        raise ClientNotApplicableError()

    def process_request(self, request):
        self.get_user()
        self.get_client_by_display_id()
        app_type = self.get_app_type()
        try:
            token = issue_token(
                uid=self.uid,
                client=self.client,
                grant_type=GRANT_TYPE_FRONTEND_ASSERTION,
                env=self.request.env,
                device_id=self.form_values['device_id'],
                device_name=self.form_values['device_name'],
                login=self.user_info['bb_response']['login'],
                make_alias=True,
                raise_error_if_limit_exceeded=True,
                app_passwords_revoked_at=get_revoke_time_from_bb_response(
                    self.user_info['bb_response'],
                    REVOKER_APP_PASSWORDS,
                ),
            )
            self.response_values = {
                'token_alias': token.alias,
                'token': token_to_response(
                    token=token,
                    client=self.client,
                    language=self.form_values['language'],
                ),
            }
        except AccessDenied:
            raise AccessDeniedError()
        except TokenLimitExceeded:
            raise TokenLimitExceededError()

        try:
            get_passport().app_password_created_send_notifications(
                uid=token.uid,
                app_type=app_type,
                app_name=token.device_name,
                user_ip=self.request.env.user_ip,
            )
        except BasePassportError as e:
            log_warning('Failed to send email via passport: %s' % e)


class CountAppPasswords(BaseIfaceApiView):
    """
    Возвращает количество валидных ПП пользователя
    """
    allowed_methods = ['GET']
    base_form = BaseApiForm
    required_grants = [LIST_TOKENS_GRANT]

    def process_request(self, request):
        self.get_user()
        self.response_values['app_passwords_count'] = len(
            list_app_passwords(
                self.uid,
                revoked_at=get_revoke_time_from_bb_response(
                    self.user_info['bb_response'],
                    REVOKER_APP_PASSWORDS,
                ),
            ),
        )


class ClientInfo(BaseIfaceApiView):
    """
    Возвращает информацию о приложении. Создателю приложения отдается больше полей
    """
    allowed_methods = ['GET']
    base_form = ClientInfoForm
    required_grants = [VIEW_CLIENTS_GRANT]

    def process_request(self, request):
        self.get_user(optional=True)
        self.get_client_by_display_id()
        language = self.form_values['language']
        viewed_by_owner = self.client.is_created_by(self.uid) or self.client.is_owned_by(self.uid)
        can_be_edited = self.client.can_be_edited(
            uid=self.uid,
            user_ip=self.request.env.user_ip,
        )
        token_ttl = get_ttl_by_scopes(self.client.scopes)

        self.response_values = {
            'client': client_to_response(self.client, language, viewed_by_owner),
            'viewed_by_owner': viewed_by_owner,
            'can_be_edited': can_be_edited,
            'token_ttl': token_ttl,
        }
        if can_be_edited:
            is_ip_internal = is_yandex_ip(request.env.user_ip)
            show_slugs = is_ip_internal and not self.form_values['force_no_slugs']
            self.response_values.update(
                extra_visible_scopes=scopes_to_response(
                    self.client.extra_visible_scopes,
                    language=language,
                    with_slugs=show_slugs,
                ),
            )
        if token_ttl:
            self.response_values.update(
                is_ttl_refreshable=check_if_is_refreshable(self.client.scopes),
            )


class UserInfo(BaseIfaceApiView):
    """
    Информация о пользователе:
     - список скоупов, доступных пользователю для создания приложений
     - тип ip-адреса (внешний/внутренний)
     - доступно ли пользователю управление флажком яндексовости приложения
    """
    allowed_methods = ['GET']
    base_form = UserInfoForm
    required_grants = [BASE_GRANT]

    def process_request(self, request):
        self.get_user()
        language = self.form_values['language']
        visible_scopes = Scope.list(
            show_hidden=False,
            uid=self.uid,
        )
        is_ip_internal = is_yandex_ip(request.env.user_ip)
        is_user_corporate = bool(
            get_attribute(self.user_info['bb_response'], settings.BB_ATTR_ACCOUNT_IS_CORPORATE),
        )
        allow_register_yandex_clients = is_ip_internal and is_user_corporate
        show_slugs = is_ip_internal and not self.form_values['force_no_slugs']
        self.response_values.update(
            visible_scopes=scopes_to_response(visible_scopes, language, with_slugs=show_slugs, with_tags=True),
            is_ip_internal=is_ip_internal,
            allow_register_yandex_clients=allow_register_yandex_clients,
        )


class ListOwnedClients(BaseIfaceApiView):
    """
    Возвращает список приложений, созданных пользователем, и список приложений, к которым у него есть доступ
    """
    allowed_methods = ['GET']
    base_form = ListClientsForm
    required_grants = [VIEW_CLIENTS_GRANT]

    def process_request(self, request):
        self.get_user()
        language = self.form_values['language']
        self.response_values.update(
            created_clients=clients_to_response(
                clients=list_clients_by_creator(self.uid),
                language=language,
                for_creator=True,
                add_token_ttl=True,
            ),
            owned_clients=clients_to_response(
                clients=[
                    client
                    for client in Client.by_owner(self.uid)
                    if not client.is_created_by(self.uid)
                ],
                language=language,
                for_creator=True,
                add_token_ttl=True,
            ),
        )


class BaseDeviceAuthorizeView(BaseIfaceApiView):
    required_grants = [ACCEPT_DEVICE_CODES_GRANT]
    allowed_methods = ['POST']
    base_form = DeviceAuthorizeForm

    def get_request_and_client(self):
        if self.form_values['client_id']:
            self.client = Client.by_display_id(self.form_values['client_id'])
            if self.client is None:
                # приложение удалено - значит, и валидных реквестов для него нет
                raise RequestNotFoundError()
        else:
            self.client = None  # получим позже

        code = self.form_values['code']
        if not does_code_look_valid(code):
            log.debug('Code is obviously invalid: %r', code)
            raise RequestNotFoundError()

        self.token_request = Request.by_verification_code(
            client_id=self.client.id if self.client else None,
            code=self.form_values['code'],
        )

        if not self.token_request or self.token_request.is_accepted or self.token_request.is_expired:
            raise RequestNotFoundError()

        if self.client is None:
            try:
                self.client = self.token_request.get_client()
            except EntityNotFoundError:
                # приложение удалено - значит, и реквест невалиден
                raise RequestNotFoundError()

        self.assert_client_is_allowed(ignore_redirect_uri=True)

    def get_and_validate_user_and_request_and_client(self):
        self.get_user()
        self.get_request_and_client()
        try:
            assert_is_not_spammer(
                self.user_info['bb_response'],
                grant_type=GRANT_TYPE_DEVICE_CODE,
                client=self.client,
                ip=self.request.env.user_ip,
            )
            assert_is_allowed(
                scope_list=self.token_request.scopes,
                grant_type=GRANT_TYPE_DEVICE_CODE,
                client_id=self.client.display_id,
                ip=self.request.env.user_ip,
                uid=self.uid,
            )
        except AccessDenied:
            raise AccessDeniedError()

        try:
            assert_is_not_child(
                bb_response=self.user_info['bb_response'],
                grant_type=GRANT_TYPE_DEVICE_CODE,
                client=self.client,
                ip=self.request.env.user_ip,
            )
        except AccessDenied:
            raise AccountIsChildError()

        self.response_values.update(
            account=account_to_response(self.user_info['bb_response']),
            client=client_to_response(
                self.client,
                language=self.form_values['language'],
                for_creator=False,
            ),
            device_name=self.token_request.device_name,
        )


class DeviceAuthorizeSubmit(BaseDeviceAuthorizeView):
    """
    Начало процесса подтверждения кода, выданного на другом девайсе
    """

    def process_request(self, request):
        self.get_and_validate_user_and_request_and_client()
        already_granted_scopes = get_already_granted_scopes(
            uid=self.uid,
            client=self.client,
            device_id=self.token_request.device_id,
            revoke_time=get_revoke_time_from_bb_response(self.user_info['bb_response'], REVOKER_TOKENS),
        )
        scopes_to_issue = self.token_request.scopes | already_granted_scopes
        requested_scopes = scopes_to_issue - already_granted_scopes

        # Обычно запрашиваем подтверждение всегда, так как здесь отсутствует защита доверенным redirect_uri
        require_user_confirm = True
        if (
            self.client.is_yandex and
            self.token_request.code_strength in {CodeStrength.BelowMediumWithCRC, CodeStrength.MediumWithCRC}
        ):
            # Но для приложений Яндекса нет необходимости запрашивать подтверждение скоупов,
            # а CRC защищает от опечатки и случайного подтверждения чужого кода. Поэтому подтверждение можно не просить.
            require_user_confirm = False

        self.response_values.update(
            require_user_confirm=require_user_confirm,
            already_granted_scopes=scopes_to_response(
                already_granted_scopes,
                self.form_values['language'],
            ),
            requested_scopes=scopes_to_response(
                requested_scopes,
                self.form_values['language'],
            ),
        )


class DeviceAuthorizeCommit(BaseDeviceAuthorizeView):
    """
    Успешное завершение процесса подтверждения кода, выданного на другом девайсе
    """

    def process_request(self, request):
        self.get_and_validate_user_and_request_and_client()

        accept_request(
            request=self.token_request,
            uid=self.uid,
        )

        to_statbox(
            mode='accept_device_code',
            status='ok',
            token_request_id=self.token_request.display_id,
            uid=self.uid,
            client_id=self.client.display_id,
            user_ip=request.env.user_ip,
            user_agent=request.env.user_agent,
            yandexuid=request.env.cookies.get('yandexuid'),
        )
