# -*- coding: utf-8 -*-
"""
Изменяет скоупы заданных приложений и всех их токенов (без инвалидации последних)
"""
import json
import logging
from time import sleep

from django.core.management import BaseCommand
from passport.backend.oauth.core.db.client import Client
from passport.backend.oauth.core.db.eav import (
    BaseDBError,
    UPDATE,
)
from passport.backend.oauth.core.db.scope import Scope
from passport.backend.oauth.core.db.token import (
    check_if_is_refreshable,
    Token,
)


log = logging.getLogger('console')


def parse_scopes(scope_keywords):
    return set([
        Scope.by_keyword(keyword)
        for keyword in scope_keywords.split(' ')
        if keyword
    ])


def scopes_to_response(scopes):
    return ','.join(sorted([scope.keyword for scope in scopes]))


def _make_config_entry(data):
    return {
        'client_display_id': data.get('client_display_id'),
        'old_scopes': parse_scopes(data.get('old_scopes') or ''),
        'new_scopes': parse_scopes(data.get('new_scopes') or ''),
        'scopes_to_add': parse_scopes(data.get('scopes_to_add') or ''),
    }


def load_config(filename):
    result = []

    with open(filename, 'r') as f:
        data = json.load(f)

    for entry in data:
        result.append(_make_config_entry(entry))

    return result


def check_config(config_data, ignore_old_scopes):
    if not config_data:
        log.error('No clients to process')
        exit(1)

    for entry in config_data:
        if not (
            entry.get('client_display_id') and
            (ignore_old_scopes or entry.get('old_scopes')) and
            (entry.get('new_scopes') or entry.get('scopes_to_add'))
        ):
            log.error('Incomplete config entry: %s', entry)
            exit(1)

    if len(set(item['client_display_id'] for item in config_data)) != len(config_data):
        log.error('Duplicate clients in config')
        exit(1)


def get_and_check_client(client_display_id, old_scopes, ignore_old_scopes=False):
    client = Client.by_display_id(display_id=client_display_id)
    if not client:
        log.error('Client %s not found' % client_display_id)
        exit(1)

    if not ignore_old_scopes and client.scopes != old_scopes:
        log.error(
            'Client %s has unexpected scopes: %s (instead of %s)' % (
                client_display_id,
                scopes_to_response(client.scopes),
                scopes_to_response(old_scopes),
            ),
        )
        exit(1)

    return client


def update_client(client, new_scopes, scopes_to_add, retries=None):
    new_scope_ids = set(client.scope_ids)
    if new_scopes:
        new_scope_ids = set(scope.id for scope in new_scopes)
    if scopes_to_add:
        new_scope_ids.update(set(scope.id for scope in scopes_to_add))

    if new_scope_ids == set(client.scope_ids):
        log.info('Client %s already has these scopes' % client.display_id)
        return

    try:
        with UPDATE(client, retries=retries):
            client.scope_ids = new_scope_ids
            client.services = list(set(scope.service_name for scope in client.scopes))
            # Статус модерации не меняем, токены не инвалидируем.
    except BaseDBError as e:
        log.error('Unable to edit client %s: %s' % (client.display_id, e))
        exit(1)

    log.info('Client %s edited' % client.display_id)


def process_clients(config_data, ignore_old_scopes=False, retries=None):
    for item in config_data:
        client = get_and_check_client(item['client_display_id'], item['old_scopes'], ignore_old_scopes=ignore_old_scopes)
        item.update(
            client=client,
            client_id=client.id,
        )

    for item in config_data:
        update_client(item['client'], item['new_scopes'], item['scopes_to_add'], retries=retries)


def process_tokens(config_data, ignore_old_scopes=False, last_processed_id=None,
                   chunk_size=5000, use_pauses=True, pause_length=1.0, retries=None):
    config_by_client_id = {item['client_id']: item for item in config_data}

    total_processed = 0
    try:
        for token_chunk in Token.iterate_by_chunks(
            chunk_size=chunk_size,
            last_processed_id=last_processed_id,
            retries=retries,
        ):
            for token in token_chunk:
                if token.client_id not in config_by_client_id:
                    continue

                old_scopes = config_by_client_id[token.client_id]['old_scopes']
                new_scopes = config_by_client_id[token.client_id]['new_scopes']
                scopes_to_add = config_by_client_id[token.client_id]['scopes_to_add']

                result_scopes = token.scopes
                if new_scopes:
                    result_scopes = new_scopes
                if scopes_to_add:
                    result_scopes.update(scopes_to_add)

                is_refreshable = check_if_is_refreshable(result_scopes)

                if result_scopes == token.scopes and token.is_refreshable == is_refreshable:
                    # токен уже корректен
                    continue

                if token.scopes != old_scopes:
                    if ignore_old_scopes:
                        if not set(token.scopes).issubset(result_scopes):
                            # Во избежание труднообнаружимых проблем скоупы токенов только расширяем
                            log.warning(
                                'Unable to shrink token %s scopes(current: %s, desired: %s)',
                                token.id,
                                scopes_to_response(token.scopes),
                                scopes_to_response(result_scopes),
                            )
                            continue
                    else:
                        log.warning(
                            'Token %s has unexpected scopes: %s (instead of %s)',
                            token.id,
                            scopes_to_response(token.scopes),
                            scopes_to_response(old_scopes),
                        )
                        continue

                try:
                    with UPDATE(token, retries=retries):
                        token.scope_ids = set([scope.id for scope in result_scopes])
                        token.is_refreshable = is_refreshable
                    total_processed += 1
                    log.info(
                        'Token %s edited (attempt total: %s)',
                        token.id,
                        total_processed,
                    )
                except BaseDBError as e:
                    log.warning('Unable to edit token %s: %s' % (token.id, e))

            if use_pauses:
                sleep(pause_length)

    except BaseDBError as e:
        log.error('Error while getting token chunk: %s', e)
        exit(1)


class Command(BaseCommand):
    help = 'Changes clients\' scopes and their tokens\' ones. Does not invalidate tokens.'

    def add_arguments(self, parser):  # pragma: no cover
        parser.add_argument(
            '--chunk_size',
            action='store',
            dest='chunk_size',
            type=int,
            default=5000,
            help='Number of objects read from DB at a time',
        )
        parser.add_argument(
            '--client_id',
            action='store',
            dest='client_display_id',
            help='Id of client to edit',
        )
        parser.add_argument(
            '--old_scopes',
            action='store',
            dest='old_scopes',
            default='',
            help='Current client\'s scopes (space-delimited)',
        )
        parser.add_argument(
            '--new_scopes',
            action='store',
            dest='new_scopes',
            default='',
            help='Scopes to set (space-delimited)',
        )
        parser.add_argument(
            '--scopes_to_add',
            action='store',
            dest='scopes_to_add',
            default='',
            help='Scopes to add (space-delimited)',
        )
        parser.add_argument(
            '--config_file',
            action='store',
            dest='config_file',
            help='Config to load client_ids and scopes from',
        )
        parser.add_argument(
            '--use_pauses',
            action='store_true',
            dest='use_pauses',
            default=False,
            help='Use pauses between processing token chunks',
        )
        parser.add_argument(
            '--pause_length',
            action='store',
            dest='pause_length',
            type=float,
            default=1.0,
            help='Pause length (in seconds)',
        )
        parser.add_argument(
            '--ignore_client_scopes',
            action='store_true',
            dest='ignore_client_scopes',
            default=False,
            help='Does not validate client\'s current scopes',
        )
        parser.add_argument(
            '--ignore_token_scopes',
            action='store_true',
            dest='ignore_token_scopes',
            default=False,
            help='Does not strictly validate token\'s current scopes',
        )
        parser.add_argument(
            '--retries',
            action='store',
            dest='retries',
            type=int,
            default=None,
            help='DB retries count',
        )
        parser.add_argument(
            '--last_processed_id',
            action='store',
            dest='last_processed_id',
            type=int,
            default=None,
            help='Token id to start from (for restarting script after errors)',
        )
        parser.add_argument(
            '--skip_tokens',
            action='store_true',
            dest='skip_tokens',
            default=False,
            help='Does not edit tokens\' scopes',
        )

    def handle(self, *args, **options):
        try:
            config_data = []

            if options['client_display_id']:
                config_data.append(_make_config_entry(options))

            if options['config_file']:
                config_data += load_config(options['config_file'])

            check_config(
                config_data,
                ignore_old_scopes=(
                    options['ignore_client_scopes'] and
                    (options['ignore_token_scopes'] or options['skip_tokens'])
                ),
            )

            process_clients(
                config_data=config_data,
                ignore_old_scopes=options['ignore_client_scopes'],
                retries=options['retries'],
            )
            if not options['skip_tokens']:
                process_tokens(
                    config_data=config_data,
                    ignore_old_scopes=options['ignore_token_scopes'],
                    chunk_size=options['chunk_size'],
                    last_processed_id=options['last_processed_id'],
                    use_pauses=options['use_pauses'],
                    pause_length=options['pause_length'],
                    retries=options['retries'],
                )
            log.info('Done.')
        except Exception as e:
            log.error('Unhandled error: %s', e, exc_info=True)
            exit(1)
