# -*- coding: utf-8 -*-
import csv
from datetime import (
    datetime,
    timedelta,
)
import logging
from time import sleep

from django.conf import settings
from django.core.management.base import BaseCommand
from passport.backend.core.builders.blackbox.blackbox import BaseBlackboxError
from passport.backend.oauth.core.common.blackbox import (
    get_blackbox,
    get_revoke_time_from_bb_response,
    REVOKER_APP_PASSWORDS,
    REVOKER_TOKENS,
)
from passport.backend.oauth.core.db.client import Client
from passport.backend.oauth.core.db.eav import (
    BaseDBError,
    DELETE,
    EntityNotFoundError,
    UPDATE,
)
from passport.backend.oauth.core.db.request import Request
from passport.backend.oauth.core.db.token import Token
from passport.backend.utils.time import unixtime_to_datetime


log = logging.getLogger('management.db_cleaner')


TIMEOUT_LENGTH = 1.0

STORE_APP_PASSWORDS_FOR = timedelta(days=90)
DELETE_OFFSET = timedelta(hours=1)

# Эти токены не удаляем: они нужны для тестов
TOKEN_WHITELIST = set([
    # тестинг, для ЧЯ (PASSP-12554)
    'a8076505f2dc4599af9e6d2034f07a82',  # удалённое приложение
    'AQAAAADueB3oAAAKj_qdTRJxNkkIt5Z7NKjHobU',  # приложение, помеченное удалённым
    '68a6e1fb0ae843a4919c2e9078785984',   # удалённый аккаунт
    'AQAAAAAAARNkAAAIovuzUgxjxkydtjuKdEhQJks',  # протухший токен
    'AQAAAADue-Q5AAAIovB8T5lMzEGovKjrFoQkeWE',  # токен, убитый глогаутом
    'AQAAAADufFocAAAIoshbz4JP40aFoOwgb6t90tE',  # токен, убитый revoke_tokens
])
APP_PASSWORD_WHITELIST = set([
    'nahocccdqqszxpqx',  # аккаунт с revoker.app_passwords
])


def load_glogouts(glogouts_csv_filename):
    log.debug('Loading glogout times from file...')
    with open(glogouts_csv_filename, 'rb') as csv_file:
        csv_reader = csv.reader(csv_file, strict=True)
        result = {}
        for row in csv_reader:
            uid, glogout = row
            uid, glogout = int(uid), int(glogout)
            if uid in result:
                raise ValueError('Duplicate uid in file: %r', row)
            if glogout > 1:
                result[uid] = glogout
    log.debug('Glogout times loaded.')
    return result


def try_drop_tokens(tokens, retries=None, simplified=False, uid_to_glogout=None):
    deleted_clients, deleted_users = set(), set()
    valid_clients = {}
    valid_users_revoke_times = {}
    dropped_count, expired_count = 0, 0

    for token in tokens:
        to_drop = False

        # Проверяем наличие в белом списке
        if token.access_token in TOKEN_WHITELIST or token.alias in APP_PASSWORD_WHITELIST:
            continue

        # Проверяем TTL
        # Делаем зазор между токенами, которые можно реюзать, и токенами, которые пора вычищать
        if token.expires and token.expires < datetime.now() - DELETE_OFFSET:
            to_drop = True

        uid = token.uid
        if simplified:
            # Собираем времена отзыва сущностей
            if uid and uid_to_glogout and uid in uid_to_glogout:
                valid_users_revoke_times[uid] = {
                    REVOKER_TOKENS: unixtime_to_datetime(uid_to_glogout[uid]),
                    REVOKER_APP_PASSWORDS: unixtime_to_datetime(uid_to_glogout[uid]),
                }
        else:
            # Проверяем существование клиента
            if not to_drop:
                if token.client_id is None or token.client_id in deleted_clients:
                    to_drop = True
                elif token.client_id not in valid_clients:
                    try:
                        client = Client.by_id(token.client_id)
                        valid_clients[client.id] = client
                    except EntityNotFoundError:
                        deleted_clients.add(token.client_id)
                        to_drop = True

            # Проверяем глогаут клиента
            if not to_drop:
                if token.is_invalidated_by_client_glogout(client=valid_clients[token.client_id]):
                    to_drop = True

            # Проверяем существование юзера и собираем времена отзыва сущностей
            # Для обезличенных токенов этот блок пропускаем
            if uid and not to_drop:
                if uid not in valid_users_revoke_times and uid not in deleted_users:
                    try:
                        bb_response = get_blackbox().userinfo(
                            uid=uid,
                            ip='127.0.0.1',
                            dbfields=settings.BLACKBOX_DBFIELDS,
                            attributes=settings.BLACKBOX_ATTRIBUTES,
                            need_display_name=False,
                        )
                    except BaseBlackboxError as e:
                        log.warning('Blackbox request failed: %s. Skipping current token %s.' % (e, token.id))
                        continue

                    if not bb_response['uid']:
                        deleted_users.add(uid)
                    else:
                        valid_users_revoke_times[uid] = {
                            REVOKER_TOKENS: get_revoke_time_from_bb_response(
                                bb_response,
                                REVOKER_TOKENS,
                            ),
                            REVOKER_APP_PASSWORDS: get_revoke_time_from_bb_response(
                                bb_response,
                                REVOKER_APP_PASSWORDS,
                            ),
                        }

                if uid in deleted_users:
                    to_drop = True

        # Проверяем времена отзыва сущностей
        # Для обезличенных токенов этот блок пропускаем
        if not to_drop and uid and uid in valid_users_revoke_times:
            if not token.is_app_password:
                logout_time = valid_users_revoke_times[uid].get(REVOKER_TOKENS, 1) - DELETE_OFFSET
                if token.is_invalidated_by_user_logout(logout_time):
                    to_drop = True
            else:
                logout_time = valid_users_revoke_times.get(uid, {}).get(REVOKER_APP_PASSWORDS, 1)
                if token.is_invalidated_by_user_logout(logout_time) and not token.expires:
                    # ПП, убитые глогаутом, удаляем не сразу: их надо показывать на УД
                    try:
                        with UPDATE(token, retries=retries):
                            token.expires = logout_time + STORE_APP_PASSWORDS_FOR
                        expired_count += 1
                    except BaseDBError as e:
                        log.warning('Failed to expire token %s: %s' % (token.id, e))

        if to_drop:
            try:
                with DELETE(token, retries=retries):
                    pass
                dropped_count += 1
            except BaseDBError as e:
                log.warning('Failed to drop token %s: %s' % (token.id, e))

    # TODO: важна ли нам статистика по причинам удаления (ttl, приложение или юзер)?
    # Пока считаем, что не важна.
    return dropped_count, expired_count


def try_drop_requests(requests, retries=None):
    # проверяем только ttl: невалидные быстро отомрут по естественным причинам
    dropped_count = 0

    for request in requests:
        # Делаем зазор в час между реквестами, которые можно использовать,
        # и реквестами, которые пора вычищать
        if request.expires and request.expires < datetime.now() - DELETE_OFFSET:
            try:
                with DELETE(request, retries=retries):
                    pass
                dropped_count += 1
            except BaseDBError as e:
                log.warning('Failed to drop request %s: %s' % (request.id, e))

        # Существование юзера не проверяем - реквест сам отомрёт по TTL

    return dropped_count


def log_chunk(chunk):
    if chunk:
        log.debug(
            'Trying to drop %s objects (ids from %s to %s)' % (
                len(chunk),
                chunk[0].id,
                chunk[-1].id,
            ),
        )


class Command(BaseCommand):
    help = 'Drops expired and invalidated objects'

    def add_arguments(self, parser):
        parser.add_argument(
            '--chunk_size',
            action='store',
            dest='chunk_size',
            type=int,
            default=5000,
            help='Number of objects read from DB at a time',
        )
        parser.add_argument(
            '--tokens',
            action='store_true',
            dest='clean_tokens',
            default=False,
            help='Clean tokens',
        )
        parser.add_argument(
            '--requests',
            action='store_true',
            dest='clean_requests',
            default=False,
            help='Clean requests',
        )
        parser.add_argument(
            '--use_pauses',
            action='store_true',
            dest='use_pauses',
            default=False,
            help='Use pauses between processing token chunks',
        )
        parser.add_argument(
            '--pause_length',
            action='store',
            dest='pause_length',
            type=float,
            default=1.0,
            help='Pause length (in seconds)',
        )
        parser.add_argument(
            '--retries',
            action='store',
            dest='retries',
            type=int,
            default=None,
            help='DB retries count',
        )
        parser.add_argument(
            '--last_processed_id',
            action='store',
            dest='last_processed_id',
            type=int,
            default=None,
            help='Object id to start from (for restarting script after errors)',
        )
        parser.add_argument(
            '--simplified',
            action='store_true',
            dest='simplified',
            default=False,
            help='Do not perform complex checks',
        )
        parser.add_argument(
            '--glogouts_csv_file',
            action='store',
            dest='glogouts_csv_filename',
            default=None,
            help='CSV file with fields `uid` and `glogout_unixtime`. Used only with --simplified.',
        )

    def handle(self, *args, **options):
        try:
            log.info('Task started')
            dropped_token_count, dropped_request_count, expired_token_count = 0, 0, 0

            if options['clean_tokens']:
                log.info('Cleaning tokens...')

                if options['simplified'] and options['glogouts_csv_filename']:
                    uid_to_glogout = load_glogouts(glogouts_csv_filename=options['glogouts_csv_filename'])
                else:
                    uid_to_glogout = {}

                for token_chunk in Token.iterate_by_chunks(
                    chunk_size=options['chunk_size'],
                    last_processed_id=options['last_processed_id'],
                    retries=options['retries'],
                ):
                    log_chunk(token_chunk)
                    result = try_drop_tokens(
                        token_chunk,
                        retries=options['retries'],
                        simplified=options['simplified'],
                        uid_to_glogout=uid_to_glogout,
                    )
                    dropped_token_count += result[0]
                    expired_token_count += result[1]
                    if options['use_pauses']:
                        sleep(options['pause_length'])

            if options['clean_requests']:
                log.info('Cleaning requests...')
                for request_chunk in Request.iterate_by_chunks(
                    chunk_size=options['chunk_size'],
                    last_processed_id=options['last_processed_id'],
                    retries=options['retries'],
                ):
                    log_chunk(request_chunk)
                    dropped_request_count += try_drop_requests(
                        request_chunk,
                        retries=options['retries'],
                    )
                    if options['use_pauses']:
                        sleep(options['pause_length'])

            message = 'Task complete. Dropped %d tokens, %s requests; %d tokens set to expire soon' % (
                dropped_token_count,
                dropped_request_count,
                expired_token_count,
            )
            log.info(message)
        except Exception as e:
            log.error('Unhandled error: %s', e, exc_info=True)
            exit(1)
