# -*- coding: utf-8 -*-
from datetime import (
    date,
    timedelta,
)
import logging
import os

from django.conf import settings
from django.core.management.base import BaseCommand
from passport.backend.oauth.core.common.zookeeper import run_exclusively
from passport.backend.oauth.core.db.client import Client
from passport.backend.utils.time import datetime_to_integer_unixtime
import yt.wrapper as yt


log = logging.getLogger('management.dump_clients_to_yt')


DATA_SCHEMA = [
    {'name': 'client_id', 'type': 'string', 'required': True},
    {'name': 'title', 'type': 'string', 'required': True},
    {'name': 'scopes', 'type_v3': {'type_name': 'list', 'item': 'string'}},
    {'name': 'redirect_urls', 'type_v3': {'type_name': 'list', 'item': 'string'}},
    {'name': 'homepage', 'type': 'string', 'required': False},
    {'name': 'creator_uid', 'type': 'uint64', 'required': True},
    {'name': 'create_time', 'type': 'datetime', 'required': True},
    {'name': 'modification_time', 'type': 'datetime', 'required': True},
    {'name': 'is_yandex', 'type': 'boolean', 'required': True},
]


def client_to_dict(client):
    result = {
        'client_id': client.display_id,
        'title': client.get_title(),
        'scopes': [scope.keyword for scope in client.scopes],
        'redirect_urls': client.redirect_uris,
        'homepage': client.homepage or None,
        'creator_uid': client.uid,
        'create_time': datetime_to_integer_unixtime(client.created),
        'modification_time': datetime_to_integer_unixtime(client.modified),
        'is_yandex': client.is_yandex,
    }
    # проверим, что все нужные поля описаны в схеме
    assert set(result.keys()) == set([item['name'] for item in DATA_SCHEMA])
    return result


def get_yt_client():
    config = {}
    if settings.YT_OAUTH_TOKEN:
        config.update(token=settings.YT_OAUTH_TOKEN)
    else:
        log.warning('YT_OAUTH_TOKEN not set in secrets. Trying to use token from ~/.yt/token')

    return yt.YtClient(
        proxy=settings.YT_DEFAULT_CLUSTER,
        config=config,
    )


def dump_to_yt(chunk_size, retries=None, force_overwrite=False):
    yt_client = get_yt_client()

    target_dir = settings.YT_CLIENTS_TABLE_DIR
    if not yt_client.exists(target_dir):
        yt_client.mkdir(target_dir, recursive=True)

    target_path = os.path.join(target_dir, str(date.today()))
    if yt_client.exists(target_path):
        if force_overwrite:
            log.info(f'Removing {target_path} before rewriting it...')
            yt_client.remove(target_path)
        else:
            log.warning(f'Table {target_path} already exists. Aborting task.')
            return

    expiration_time = date.today() + timedelta(days=settings.YT_CLIENTS_TABLE_TTL_DAYS)
    yt_client.create(
        type='table',
        path=target_path,
        attributes={
            'schema': DATA_SCHEMA,
            'expiration_time': str(expiration_time),
        },
    )

    log.debug('Processing clients...')
    for client_chunk in Client.iterate_by_chunks(
        chunk_size=chunk_size,
        retries=retries,
    ):
        log.debug(
            'Processing %s clients (ids from %s to %s)' % (
                len(client_chunk),
                client_chunk[0].id,
                client_chunk[-1].id,
            ),
        )
        rows = []
        for client in client_chunk:
            rows.append(client_to_dict(client))

        yt_client.write_table(yt.TablePath(target_path, append=True), rows)


class Command(BaseCommand):
    help = 'Uploads OAuth clients\' info to YT table'

    def add_arguments(self, parser):
        parser.add_argument(
            '--chunk_size',
            action='store',
            dest='chunk_size',
            type=int,
            default=1000,
            help='Number of objects read from DB at a time',
        )
        parser.add_argument(
            '--retries',
            action='store',
            dest='retries',
            type=int,
            default=None,
            help='DB retries count',
        )
        parser.add_argument(
            '--force-overwrite',
            action='store_true',
            dest='force_overwrite',
            default=False,
            help='Overwrite table if it exists',
        )

    @run_exclusively('/passport/oauth/management/dump_clients_to_yt', treat_rc_as_prod=False)
    def handle(self, *args, **options):
        try:
            log.info('Task started')

            if not settings.YT_CLIENTS_TABLE_DIR:
                log.warning('Should not dump tables for current environment. Aborting task.')
                exit(0)

            dump_to_yt(
                chunk_size=options['chunk_size'],
                retries=options['retries'],
                force_overwrite=options['force_overwrite'],
            )
            log.info('Task complete')
        except Exception as e:
            log.error('Unhandled error: %s', e, exc_info=True)
            exit(1)
