# -*- coding: utf-8 -*-
from collections import defaultdict
import logging
import sys
from time import sleep
import traceback

from django.conf import settings
from django.core.management.base import BaseCommand
from passport.backend.core.builders.staff import (
    Staff,
    StaffAuthorizationInvalidError,
    StaffEntityNotFoundError,
)
from passport.backend.oauth.core.common.constants import GRANT_TYPE_PASSPORT_ASSERTION
from passport.backend.oauth.core.common.zookeeper import run_exclusively
from passport.backend.oauth.core.db.client import Client
from passport.backend.oauth.core.db.eav import (
    BaseDBError,
    UPDATE,
)
from passport.backend.oauth.core.db.token import (
    issue_token,
    TOKEN_TYPE_STATELESS,
)
from passport.backend.oauth.core.env import Environment
from passport.backend.oauth.core.logs.graphite import GraphiteLogger


log = logging.getLogger('management.owner_groups_updater')


MAX_UID_COUNT = 300


class BaseStaffRequesterError(Exception):
    """Базовая ошибка"""


class InvalidGroupError(BaseStaffRequesterError):
    """В БД записано невалидное (или несуществующее) имя группы"""


class InvalidTokenError(BaseStaffRequesterError):
    """Авторизационный токен внезапно протух"""


class UidsRequester(object):
    def __init__(self):
        self._token = self._issue_token()
        self._builder = Staff(
            url=settings.STAFF_API_URL,
            timeout=settings.STAFF_API_TIMEOUT,
            retries=settings.STAFF_API_RETRIES,
            graphite_logger=GraphiteLogger(service='staff'),
            ca_cert=settings.SSL_CA_CERT,
        )
        self._cache = {}

    def _issue_token(self):
        client = Client.by_display_id(settings.STAFF_OAUTH_CLIENT_ID)
        if not client:
            log.error('Failed to issue token: client "%s" not found', settings.STAFF_OAUTH_CLIENT_ID)
            sys.exit(1)
        token = issue_token(
            uid=settings.STAFF_ROBOT_UID,
            client=client,
            grant_type=GRANT_TYPE_PASSPORT_ASSERTION,
            env=Environment(
                user_ip='127.0.0.1',
                consumer_ip='127.0.0.1',
                host='127.0.0.1',
                user_agent=None,
                raw_cookies={},
                cookies={},
                request_id=None,
                authorization=None,
                service_ticket=None,
                user_ticket=None,
            ),
            token_type=TOKEN_TYPE_STATELESS,
        )
        return token.access_token

    def _get_uids(self, group):
        if ':' not in group:
            raise InvalidGroupError(group)

        group_type, group_name = group.split(':', 1)
        if group_type not in settings.DEPARTMENT_GROUP_PREFIXES:
            try:
                department_id = self._builder.get_department_id(
                    oauth_token=self._token,
                    department_url=group_name,
                )
            except StaffEntityNotFoundError:
                raise InvalidGroupError(group)

            rv = self._builder.get_team(
                oauth_token=self._token,
                department_id=department_id,
                fields=['uid'],
                max_pages=6,  # 300 юзеров
            )
            return [item['uid'] for item in rv]
        else:
            raise InvalidGroupError(group)

    def get_uids(self, group):
        if group not in self._cache:
            try:
                self._cache[group] = self._get_uids(group)
            except StaffAuthorizationInvalidError:
                raise InvalidTokenError()

        return self._cache[group]


def try_update_clients(target_classes, chunk_size, pause_length=1, retries=None):
    updated_counts = defaultdict(int)
    uids_requester = UidsRequester()
    for model_class in target_classes:
        model_class_name = model_class.__name__
        log.info('Updating %ss', model_class_name)
        for client_chunk in model_class.iterate_by_chunks(
            chunk_size=chunk_size,
            retries=retries,
        ):
            log_chunk(client_chunk, model_class_name)
            for client in client_chunk:
                try:
                    if client.owner_groups:
                        owner_group_uids = set()
                        for group in client.owner_groups:
                            try:
                                owner_group_uids.update(uids_requester.get_uids(group))
                            except InvalidGroupError:
                                log.info('Bad group: %s', group)
                            except InvalidTokenError:
                                log.error('Token expired during operation')
                                sys.exit(1)

                        if len(owner_group_uids) >= MAX_UID_COUNT:
                            log.warning(
                                'Too many uids for %s %s: %d/%d',
                                model_class_name,
                                getattr(client, 'display_id', client.id),
                                len(owner_group_uids),
                                MAX_UID_COUNT,
                            )
                            owner_group_uids = list(owner_group_uids)[:MAX_UID_COUNT]

                        with UPDATE(client):
                            client._owner_group_uids = owner_group_uids
                        updated_counts[model_class_name] += 1
                except BaseDBError as e:
                    log.warning('DB error: %s', e)
                except Exception as e:
                    log.error('Unhandled error: %s\n%s', e, traceback.format_exc())

            if pause_length:
                sleep(pause_length)

    return updated_counts


def log_chunk(chunk, model_class_name):
    if chunk:
        log.debug(
            'Trying to update %s %ss (ids from %s to %s)' % (
                len(chunk),
                model_class_name,
                chunk[0].id,
                chunk[-1].id,
            ),
        )


class Command(BaseCommand):
    help = 'Updates owner lists for selected types of clients'

    def add_arguments(self, parser):
        parser.add_argument(
            '--chunk_size',
            action='store',
            dest='chunk_size',
            type=int,
            default=1000,
            help='Number of objects read from DB at a time',
        )
        parser.add_argument(
            '--use_pauses',
            action='store_true',
            dest='use_pauses',
            default=False,
            help='Use pauses between processing token chunks',
        )
        parser.add_argument(
            '--pause_length',
            action='store',
            dest='pause_length',
            type=float,
            default=1.0,
            help='Pause length (in seconds)',
        )
        parser.add_argument(
            '--retries',
            action='store',
            dest='retries',
            type=int,
            default=None,
            help='DB retries count',
        )

    @run_exclusively('/passport/oauth/management/update_owner_groups')
    def handle(self, *args, **options):
        try:
            log.info('Task started')

            counts = try_update_clients(
                target_classes=[Client],
                chunk_size=options['chunk_size'],
                pause_length=options['pause_length'],
                retries=options['retries'],
            )
            log.info('Task complete. %d OAuth clients updated', counts['Client'])
        except Exception as e:
            log.error('Unhandled error: %s', e, exc_info=True)
            exit(1)
