# -*- coding: utf-8 -*-
from django.conf import settings
from django.http import HttpResponse
from django.utils.deprecation import MiddlewareMixin
from passport.backend.core.builders.blackbox.blackbox import BaseBlackboxError
from passport.backend.core.builders.captcha import CaptchaError
from passport.backend.core.builders.money_api import BaseMoneyApiError
from passport.backend.core.builders.passport import PassportTemporaryError
from passport.backend.core.builders.tvm import TVMTemporaryError
from passport.backend.oauth.api.api.old.error_descriptions import TOKENS_LIMIT_EXCEEDED
from passport.backend.oauth.api.api.old.views import api_error
from passport.backend.oauth.core.common.constants import (
    BAD_FLOW,
    INVALID_REQUEST,
)
from passport.backend.oauth.core.common.error_logs import log_error
from passport.backend.oauth.core.db.eav import BaseDBError
from passport.backend.oauth.core.db.errors import TokenLimitExceededError
from passport.backend.oauth.core.db.limits import AccessDeniedError


TEMPORARILY_UNAVAILABLE_STUB = 'Service temporarily unavailable'
INTERNAL_ERROR_STUB = 'Internal server error'


class ErrorsMiddleware(MiddlewareMixin):
    def process_exception(self, request, exception):
        if issubclass(
            exception.__class__,
            (BaseBlackboxError, CaptchaError, BaseDBError, TVMTemporaryError, PassportTemporaryError, BaseMoneyApiError),
        ):
            return HttpResponse(TEMPORARILY_UNAVAILABLE_STUB, status=503)
        elif issubclass(exception.__class__, AccessDeniedError):
            return api_error(request, BAD_FLOW)  # unauthorized_client (rev-10)
        elif issubclass(exception.__class__, TokenLimitExceededError):
            return api_error(request, INVALID_REQUEST, description=TOKENS_LIMIT_EXCEEDED)
        else:
            log_error(request)
            return HttpResponse(INTERNAL_ERROR_STUB, status=500)


class LocalePerDomainMiddleware(MiddlewareMixin):
    def process_request(self, request):
        hostname = request.headers['Host'].split(':', 1)[0].lower()
        tld = next(
            (tld for tld in settings.OAUTH_TLDS if hostname.endswith('.%s' % tld)),
            'ru',
        )
        request.TLD = tld
        request.LANGUAGE_CODE = settings.TLD_TO_LANG_MAPPING.get(tld, settings.DEFAULT_LANGUAGE)
