import logging
from typing import (
    Dict,
    Optional,
)

from django.conf import settings
from passport.backend.core.builders.antifraud import (
    AntifraudApi,
    BaseAntifraudApiError,
    ScoreAction,
    ScoreResponse,
)
from passport.backend.core.utils.decorators import cached_property
from passport.backend.utils.common import noneless_dict


log = logging.getLogger('api')


class AntifraudScoreManager:
    enabled: bool
    score_params: Dict
    score_response: Optional[ScoreResponse]

    def __init__(self, initial_data: Optional[Dict] = None):
        self.score_params = noneless_dict(initial_data or {})
        self.executed = False
        self.score_response = None

    @cached_property
    def antifraud_api(self):
        return AntifraudApi()

    def bind(self, **kwargs):
        self.score_params.update(noneless_dict(**kwargs))

    def get_antifraud_score(self) -> Optional[ScoreResponse]:
        if not settings.ANTIFRAUD_SCORE_ENABLE:
            return
        log.debug('Sending data to antifraud score')
        try:
            score_response = self.antifraud_api.score(self.score_params)
            self.score_response = score_response
            log.debug('Got antifraud score: {}'.format(score_response))
            return score_response
        except BaseAntifraudApiError as err:
            log.error('Antifraud score request error: {}'.format(err))
            return

    def check_antifraud_score(self) -> bool:
        self.get_antifraud_score()
        if self.score_response is None:
            # score_response может быть None если отключен ANTIFRAUD_SCORE_ENABLE,
            # либо при ошибке хождения в ручку. В обоих случаях выдаём разрешение
            return True
        else:
            return self.score_response.action != ScoreAction.DENY
