# -*- coding: utf-8 -*-
from django.conf import settings
from django.core.exceptions import ValidationError
from django.http import HttpResponseNotAllowed
from django.utils.encoding import smart_text
from passport.backend.oauth.api.api.forms import DeviceIdField
from passport.backend.oauth.api.api.old.bundle_views.errors import OldApiError
from passport.backend.oauth.api.api.old.error_descriptions import (
    CLIENT_NOT_FOUND,
    DEVICE_ID_INVALID,
    POST_PARAM_DUPLICATE,
    POST_PARAM_MISSING,
    POST_PARAM_NUMBER_REQUIRED,
    SCOPE_ERROR,
    SCOPES_POST_AND_CLIENT_MISMATCH,
    TOO_MANY_REQUESTS,
)
from passport.backend.oauth.api.api.old.errors import MalformedAuthorizationHeaderError
from passport.backend.oauth.api.api.old.utils import (
    ensure_only_one_entry,
    parse_auth_header,
)
from passport.backend.oauth.api.api.old.views import (
    api_error,
    api_success,
)
from passport.backend.oauth.core.api.base import BaseApiView
from passport.backend.oauth.core.common.constants import (
    AUTH_HEADER_NAME,
    BAD_CLIENT,
    BAD_FLOW,
    INVALID_REQUEST,
    INVALID_SCOPE,
)
from passport.backend.oauth.core.common.error_logs import log_warning
from passport.backend.oauth.core.common.kolmogor import (
    get_kolmogor_values,
    inc_kolmogor_values,
)
from passport.backend.oauth.core.db.client import (
    ApprovalStatus,
    Client,
)
from passport.backend.oauth.core.db.device_info import get_device_info
from passport.backend.oauth.core.db.scope import Scope


class BaseOldApiView(BaseApiView):
    def respond_error(self, exception):
        return api_error(self.request, status=exception.status, **exception.data)

    def respond_success(self):
        return api_success(self.request, self.response_values)

    def dispatch_request(self, request, http_method):
        try:
            self.request = request

            if http_method not in self.allowed_methods:
                log_warning('Method not allowed (%s): %s' % (request.method, request.path))
                return HttpResponseNotAllowed(self.allowed_methods)

            self.process_request(request)
        except OldApiError as e:
            return self.respond_error(e)
        # TODO: перенести сюда обработку ошибок из middleware

        return self.respond_success()

    def get_required_param(self, name):
        if name not in self.request.POST:
            raise OldApiError(INVALID_REQUEST, POST_PARAM_MISSING % name)
        return self.get_optional_param(name)

    def get_required_integer_param(self, name, error=INVALID_REQUEST):
        value = self.get_required_param(name)
        try:
            return int(value)
        except ValueError:
            self.statbox.log(status='error', reason='%s.invalid' % name, **{name: value})
            raise OldApiError(error, POST_PARAM_NUMBER_REQUIRED % name)

    def get_optional_param(self, name, default=None):
        try:
            ensure_only_one_entry(
                items=[name],
                params=self.request.POST,
            )
        except ValueError:
            raise OldApiError(INVALID_REQUEST, description=POST_PARAM_DUPLICATE % name)
        return self.request.POST.get(name, default)

    def get_optional_integer_param(self, name, default=None, error=INVALID_REQUEST):
        value = self.get_optional_param(name)
        try:
            return int(value) if value else default
        except ValueError:
            self.statbox.log(status='error', reason='%s.invalid' % name, **{name: value})
            raise OldApiError(error, POST_PARAM_NUMBER_REQUIRED % name)

    def get_optional_boolean_param(self, name, default=False):
        value = self.get_optional_param(name, '').lower()
        if value in ('yes', 'true', '1'):
            return True
        elif value in ('no', 'false', '0'):
            return False
        else:
            return default

    def is_client_secret_required(self, client):
        # Требуем секрет только от тех клиентов, у которых он есть.
        # Может перегружаться в потомках.
        return bool(client.secret)

    def get_client_from_request(self):
        if AUTH_HEADER_NAME in self.request.headers:
            status_for_error = 401
            try:
                display_id, secret = parse_auth_header(self.request.headers[AUTH_HEADER_NAME])
            except MalformedAuthorizationHeaderError as e:
                raise OldApiError(smart_text(e), '', status=status_for_error)
        else:
            status_for_error = 400
            display_id = self.get_required_param('client_id')
            secret = self.get_optional_param('client_secret')

        client = Client.by_display_id(display_id)
        if not client:
            raise OldApiError(BAD_CLIENT, CLIENT_NOT_FOUND, status=status_for_error)
        if client.is_blocked:
            raise OldApiError(BAD_CLIENT, 'Client blocked', status=status_for_error)
        if self.is_client_secret_required(client) and client.secret != secret:
            raise OldApiError(BAD_CLIENT, 'Wrong client secret', status=status_for_error)
        if client.approval_status not in [ApprovalStatus.NotRequired, ApprovalStatus.Approved]:
            raise OldApiError(BAD_FLOW, 'Client not approved', status=status_for_error)

        return client

    def get_and_bind_client(self):
        self.client = self.get_client_from_request()
        self.statbox.bind_context(client_id=self.client.display_id)

    def get_and_bind_device_info(self):
        self.device_info = get_device_info(self.request.REQUEST)
        try:
            self.device_id = DeviceIdField(required=False).clean(
                self.device_info.get('device_id'),
            )
        except ValidationError:
            raise OldApiError(INVALID_REQUEST, DEVICE_ID_INVALID)

        if self.device_id:
            self.device_name = self.device_info.get('device_name') or self.device_info.get('model_name')
        else:
            self.device_id = self.device_name = None

    def get_requested_scopes(self):
        scopes = self.get_optional_param('scope')
        if scopes:
            try:
                scopes = set(Scope.by_keyword(keyword) for keyword in scopes.split(' '))
            except ValueError:
                self.statbox.log(
                    status='error',
                    reason='scopes.invalid',
                    requested_scopes=scopes,
                )
                raise OldApiError(INVALID_SCOPE, SCOPE_ERROR)
            try:
                scopes = self.validate_requested_scopes(scopes)
            except ValidationError:
                raise OldApiError(INVALID_SCOPE, SCOPES_POST_AND_CLIENT_MISMATCH)
        return scopes

    def validate_requested_scopes(self, scopes):
        if not set(scopes).issubset(self.client.scopes):
            self.statbox.log(
                status='error',
                reason='scopes.not_match',
                scopes=scopes,
                client_scopes=self.client.scopes,
            )
            raise ValidationError('scopes.not_match')
        return scopes

    @property
    def rate_limit_counters(self):
        return {}  # может перегружаться в потомках

    def raise_rate_limit_error(self):
        raise OldApiError(
            error=INVALID_REQUEST,
            description=TOO_MANY_REQUESTS,
            status=429,
        )

    def check_counters(self, counter_to_key_mapping, uid=None, client_id=None):
        if not settings.ENABLE_RATE_LIMITS:
            return

        counter_values = get_kolmogor_values(
            space=settings.KOLMOGOR_KEYSPACE,
            keys=counter_to_key_mapping.values(),
        )
        if counter_values:
            for counter, limit in self.rate_limit_counters.items():
                counter_key = counter_to_key_mapping[counter]
                counter_value = counter_values[counter_key]
                if counter_value > limit:
                    self.statbox.get_child().log(
                        reason='rate_limit_exceeded',
                        status='error',
                        key=counter_key,
                        value=counter_value,
                        limit=limit,
                        uid=uid,
                        client_id=client_id,
                    )
                    self.raise_rate_limit_error()

    def increase_counters(self, counter_to_key_mapping):
        if not settings.ENABLE_RATE_LIMITS:
            return
        inc_kolmogor_values(
            space=settings.KOLMOGOR_KEYSPACE,
            keys=counter_to_key_mapping.values(),
        )
