# -*- coding: utf-8 -*-
from abc import (
    ABC,
    abstractmethod,
)
import json
import logging
import re
from typing import (
    Dict,
    Iterable,
    List,
    Type,
)

from passport.backend.core.utils.decorators import cached_property
from passport.backend.core.utils.experiments import is_experiment_enabled_by_login


log = logging.getLogger('api')


class BaseValidatorRule(ABC):
    def __init__(
        self, _statbox, _user_entered_login, experiment_denominator=None,
        statbox_alias=None, to_statbox=True, _path='', **_
    ):
        self.statbox = _statbox
        self.user_entered_login = _user_entered_login
        self.path = _path

        self.experiment_denominator = experiment_denominator
        self.statbox_alias = statbox_alias
        self.to_statbox = to_statbox

        self.executed = False
        self.errors = []
        self.error_fields = []

    @abstractmethod
    def _execute(self, data):
        pass

    def execute(self, data) -> None:
        self._execute(data)
        self.executed = True

    def _field_path(self, field):
        return '::'.join(x for x in (self.path, field) if x)

    def _make_subrule(self, subpath, rule_conf):
        try:
            return _rule_class_lookup[rule_conf['class']](
                _statbox=self.statbox,
                _user_entered_login=self.user_entered_login,
                _path=subpath,
                **rule_conf
            )
        except (TypeError, ValueError, KeyError) as err:
            msg = 'Error in rule conf {} {}: {} {}'.format(
                self.path, rule_conf, err.__class__, err,
            )
            raise ValueError(msg) from err

    def _bind_to_statbox(self, field, value):
        if self.to_statbox:
            self.statbox.bind(**{self.statbox_alias or self._field_path(field): str(value)})

    def _set_error(self, error_field, error):
        self.error_fields.append(error_field)
        self.errors.append(error)

    @property
    @abstractmethod
    def missing_value(self) -> bool:
        pass

    @property
    def ok(self) -> bool:
        assert self.executed
        return not bool(self.errors)

    def _get_field_value(self, data, field):
        if isinstance(data, dict):
            return data.get(field, None)
        else:
            if field.startswith('env.'):
                return getattr(data.env, field[4:], None)
            else:
                return data.REQUEST.get(field, None)

    @cached_property
    def experiment_dry_run(self) -> bool:
        if self.experiment_denominator is None:
            return False
        return not is_experiment_enabled_by_login(
            self.user_entered_login,
            self.experiment_denominator,
        )


class BaseMultipleFieldsValidatorRule(BaseValidatorRule, ABC):
    @property
    def missing_value(self) -> bool:
        raise TypeError(
            'Cannot check multiple fields validator for a missing value. '
            'Maybe it was used in ignore_for context?',
        )


class BaseSingleFieldValidatorRule(BaseValidatorRule, ABC):
    def __init__(self, field, **kwargs):
        super().__init__(**kwargs)
        self.field = field
        self._missing_value = True

    @property
    def missing_value(self) -> bool:
        return self._missing_value

    @abstractmethod
    def _validate_value(self, value):
        pass

    def _execute(self, data):
        value = self._get_field_value(data, self.field)
        if value is not None:
            self._missing_value = False
            self._bind_to_statbox(self.field, value)
            self._validate_value(value)


class RequiredFields(BaseMultipleFieldsValidatorRule):
    def __init__(self, fields, **kwargs):
        super().__init__(**kwargs)
        self.fields = fields

    def __str__(self):
        return 'require:{}'.format(self.fields)

    def _execute(self, data):
        for field in self.fields:
            if self._get_field_value(data, field) is None:
                self._set_error(
                    '{}:required'.format(self._field_path(field)),
                    'missing request field {}'.format(self._field_path(field)),
                )


class FieldDummy(BaseSingleFieldValidatorRule):
    """
    Нужно, чтобы без валидации вешать сайд-эффекты на поле.
    На момент написания этого докстринга сайд-эффект один: логирование.
    """
    def _validate_value(self, value):
        pass


class FieldValueMatch(BaseSingleFieldValidatorRule):
    def __init__(
        self, value_in=None, regex=None, lowercase=False,
        re_group_validators=None, **kwargs
    ):
        super().__init__(**kwargs)
        self.value_in = value_in
        self.regex = regex
        self.lowercase = lowercase
        self.re_group_validators = re_group_validators or []
        assert self.value_in or self.regex
        assert not self.re_group_validators or self.regex

        self.matched_value = None

    def __str__(self):
        if self.regex:
            return '{} re {}'.format(self.field, self.regex)
        elif self.matched_value:
            return '{} = {}'.format(self.field, self.matched_value)
        else:
            return '{} in {}'.format(self.field, self.value_in)

    def _run_regex_validators(self, matches: re.Match):
        for conf in self.re_group_validators:
            rule = self._make_subrule(self._field_path(self.field), conf)
            rule.execute(matches.groupdict())
            if not rule.ok:
                self.errors.extend(rule.errors)
                self.error_fields.extend(rule.error_fields)

    def _validate_value(self, value):
        if self.lowercase:
            value = value.lower()
        if self.regex:
            matches = self.regex.search(value)
            if matches:
                self._run_regex_validators(matches)
            else:
                self._set_error(
                    '{}:re'.format(self._field_path(self.field)),
                    'field \'{}\' value \'{}\' doesn\'t match re \'{}\''.format(
                        self._field_path(self.field), value, self.regex,
                    ),
                )
        if self.value_in:
            if value not in self.value_in:
                self._set_error(
                    '{}:value_in'.format(self._field_path(self.field)),
                    'field \'{}\' value \'{}\' is not in allowed values list'.format(
                        self._field_path(self.field), value,
                    ),
                )


class FieldValueJson(BaseSingleFieldValidatorRule):
    def __init__(self, validators=None, **kwargs):
        super().__init__(**kwargs)
        self.validators = validators or []

    def _run_validators(self, data):
        for conf in self.validators:
            rule = self._make_subrule(self._field_path(self.field), conf)
            rule.execute(data)
            if not rule.ok:
                self.errors.extend(rule.errors)
                self.error_fields.extend(rule.error_fields)

    def _validate_value(self, value):
        try:
            parsed = json.loads(value)
        except json.JSONDecodeError as err:
            self._set_error(
                '{}:json'.format(self._field_path(self.field)),
                'field \'{}\' value \'{}\' is not in JSON format: {}'.format(
                    self._field_path(self.field), value, err,
                ),
            )
        else:
            if self.validators:
                if isinstance(parsed, dict):
                    self._run_validators(parsed)
                else:
                    self._set_error(
                        '{}:json-dict'.format(self._field_path(self.field)),
                        'field \'{}\' value \'{}\' is not a JSON dict'.format(
                            self._field_path(self.field), value,
                        ),
                    )


RULE_CLASSES = [RequiredFields, FieldDummy, FieldValueMatch, FieldValueJson]
_rule_class_lookup: Dict[str, Type[BaseValidatorRule]] = {cls.__name__: cls for cls in RULE_CLASSES}


class StructureValidator:
    errors: List[str]
    error_fields: List[str]

    def __init__(self, statbox, user_entered_login, ignore_for=None, validators=None):
        self.statbox = statbox
        self.user_entered_login = user_entered_login
        self.ignore_for = ignore_for or []
        self.validators = validators or []

        self.errors = []
        self.error_fields = []
        self.experiment_dry_run = True
        self.ignored_by_rule = None
        self.executed = False

    def _make_rule(self, rule_conf):
        return _rule_class_lookup[rule_conf['class']](
            _statbox=self.statbox,
            _user_entered_login=self.user_entered_login,
            **rule_conf
        )

    def _make_rules(self, rules_conf) -> Iterable[BaseValidatorRule]:
        for rule_conf in rules_conf:
            try:
                yield self._make_rule(rule_conf)
            except (KeyError, ValueError, TypeError) as err:
                msg = 'Error in rule conf {}: {} {}'.format(
                    rule_conf, err.__class__, err,
                )
                raise ValueError(msg) from err

    @property
    def ok(self) -> bool:
        assert self.executed
        return not bool(self.errors)

    def validate(self, data) -> None:
        self._validate(data)
        self.executed = True

    def _validate(self, data):
        for rule in self._make_rules(self.ignore_for):
            rule.execute(data)
            if rule.ok and not rule.missing_value:
                self.ignored_by_rule = rule
                return
        for rule in self._make_rules(self.validators):
            rule.execute(data)
            if not rule.ok:
                self.errors.extend(rule.errors)
                self.error_fields.extend(rule.error_fields)
                if not rule.experiment_dry_run:
                    # Хотя бы одна ошибка по правилу, которому требуется
                    # реальное выполнение (либо не установлен эксперимент,
                    # либо юзер попал в эксперимент), отключает dry run на
                    # всём валидаторе.
                    self.experiment_dry_run = False
