# -*- coding: utf-8 -*-
from datetime import datetime
import logging
import re
from time import time
from typing import (
    Dict,
    Optional,
    Set,
)

from django.conf import settings
from django.core.exceptions import ValidationError
from django.utils.encoding import smart_bytes
from django.utils.functional import cached_property
from passport.backend.core.builders.blackbox import (
    BLACKBOX_FIND_BY_PHONE_ALIAS_FORCE_ON,
    BLACKBOX_LOGIN_V1_DISABLED_STATUS,
    BLACKBOX_LOGIN_V1_EXPIRED_STATUS,
    BLACKBOX_LOGIN_V1_SECOND_STEP_REQUIRED_STATUS,
    BLACKBOX_LOGIN_V1_SHOW_CAPTCHA_STATUS,
    BLACKBOX_LOGIN_V1_VALID_STATUS,
    BLACKBOX_OAUTH_VALID_STATUS,
    BLACKBOX_SESSIONID_NEED_RESET_STATUS,
    BLACKBOX_SESSIONID_VALID_STATUS,
    BLACKBOX_SESSIONID_WRONG_GUARD_STATUS,
    BlackboxInvalidParamsError,
    get_alias,
    get_attribute,
)
from passport.backend.core.builders.captcha import CaptchaLocateError
from passport.backend.core.builders.money_api import (
    get_money_payment_auth_api,
    PAYMENT_AUTH_SUCCEEDED_STATUS,
)
from passport.backend.core.builders.passport import BasePassportError
from passport.backend.core.builders.tvm import TVM_STATUS_OK
from passport.backend.core.geobase import (
    get_country_code_by_ip,
    Region,
)
from passport.backend.core.logging_utils.helpers import mask_sessionid
from passport.backend.core.logging_utils.loggers.statbox import (
    AntifraudLogger,
    CredentialsLogger,
)
from passport.backend.core.portallib import is_yandex_server_ip
from passport.backend.oauth.api.api.old.antifraud_score import AntifraudScoreManager
from passport.backend.oauth.api.api.old.bundle_views.base import BaseOldApiView
from passport.backend.oauth.api.api.old.bundle_views.errors import OldApiError
from passport.backend.oauth.api.api.old.bundle_views.validation import StructureValidator
from passport.backend.oauth.api.api.old.error_descriptions import (
    ACCOUNT_TYPE_FORBIDDEN,
    BLACKBOX_PARAMS_INVALID,
    CAPTCHA_ANSWER_WRONG,
    CAPTCHA_REQUIRED,
    CLIENT_ID_MISMATCH,
    CODE_EXPIRED,
    CODE_INVALID,
    CODE_NOT_ACTIVATED,
    CODE_VERIFIER_NOT_MATCHED,
    COOKIE_HEADER_MISSING,
    GRANT_TYPE_UNKNOWN,
    LOGIN_PASSWORD_INVALID,
    META_INVALID,
    MUTUALLY_EXCLUSIVE_PARAMS,
    PASSWORD_CHANGE_REQUIRED,
    PASSWORD_EXPIRED,
    PAYMENT_AUTH_NOT_PASSED,
    PAYMENT_AUTH_REQUIRED,
    POST_PARAM_MISSING,
    POST_PARAM_NUMBER_REQUIRED,
    SCOPE_X_TOKEN_GRANT_MISSING,
    SCOPES_CODE_AND_CLIENT_MISMATCH,
    SESSION_INVALID,
    SSH_KEYS_NOT_FOUND,
    SSH_SIGN_INVALID,
    TOKEN_EXPIRED,
    TOKEN_IS_STATELESS,
    TOKEN_NOT_BELONG_TO_CLIENT,
    TOKEN_WITHOUT_DEVICE_ID,
    TS_WRONG,
    UID_MISSING,
    USER_ACTION_REQUIRED,
    USER_NOT_FOUND,
)
from passport.backend.oauth.api.api.old.utils import (
    is_passport_domain,
    strip_to_yandex_domain,
)
from passport.backend.oauth.core.common import captcha
from passport.backend.oauth.core.common.blackbox import (
    get_blackbox,
    get_revoke_time_from_bb_response,
    REVOKER_TOKENS,
    REVOKER_WEB_SESSIONS,
)
from passport.backend.oauth.core.common.constants import (
    AUTHORIZATION_PENDING,
    BAD_FLOW,
    BAD_VERIFICATION_CODE,
    CAPTCHA_SOURCE_BLACKBOX,
    CAPTCHA_SOURCE_PROFILE,
    CAPTCHA_SOURCE_RATE_LIMIT,
    CAPTCHA_SOURCE_TEST,
    ERROR,
    GRANT_TYPE_ASSERTION,
    GRANT_TYPE_CLIENT_CREDENTIALS,
    GRANT_TYPE_CODE,
    GRANT_TYPE_DEVICE_CODE,
    GRANT_TYPE_PASSPORT_ASSERTION,
    GRANT_TYPE_PASSWORD,
    GRANT_TYPE_REFRESH_TOKEN,
    GRANT_TYPE_SESSIONID,
    GRANT_TYPE_SSH_KEY,
    GRANT_TYPE_XTOKEN,
    INVALID_GRANT,
    INVALID_REQUEST,
    INVALID_SCOPE,
    PAYMENT_AUTH_PENDING,
    TOKEN_TYPE,
    UNSUPPORTED_GRANT,
    UNSUPPORTED_TOKEN,
)
from passport.backend.oauth.core.common.counters import (
    COUNTER_CLIENT_ID,
    COUNTER_IP,
    COUNTER_UID,
)
from passport.backend.oauth.core.common.error_logs import log_warning
from passport.backend.oauth.core.common.passport import get_passport
from passport.backend.oauth.core.common.portallib import is_yandex_ip
from passport.backend.oauth.core.common.tvm import get_tvm
from passport.backend.oauth.core.common.utils import (
    int_or_default,
    update_url_params,
)
from passport.backend.oauth.core.db.config.grants_config import AccessDeniedError
from passport.backend.oauth.core.db.eav import DELETE
from passport.backend.oauth.core.db.eav.types import DB_NULL
from passport.backend.oauth.core.db.limits import (
    assert_is_allowed,
    assert_is_not_child,
    assert_is_not_spammer,
    check_grant,
    GrantsMissingError,
    restrict_non_yandex_clients,
)
from passport.backend.oauth.core.db.request import (
    code_strength_to_name,
    CodeStrength,
    CodeType,
    create_request,
    does_code_look_valid,
    Request,
)
from passport.backend.oauth.core.db.scope import (
    get_payment_auth_app_ids,
    has_x_token_scope,
    is_payment_auth_required,
    Scope,
)
from passport.backend.oauth.core.db.token import (
    get_access_token_from_refresh_token,
    invalidate_single_token,
    issue_token,
    list_tokens_with_clients_by_user,
    parse_token,
    should_issue_stateless,
    StatelessToken,
    TOKEN_TYPE_NORMAL,
    TOKEN_TYPE_STATELESS,
    try_refresh_token,
)
from passport.backend.oauth.core.logs.statbox import (
    StatboxLogger,
    to_statbox,
)
from passport.backend.utils.common import merge_dicts


log = logging.getLogger('api')


META_REGEX = re.compile(r'^[ -~]{0,1000}$')  # все ASCII-символы с кодами от 32 до 126
RFC_OTP_REGEX = re.compile(r'^[ -~]{1,20}$')
TRACK_REGEX = re.compile(r'^[!-~]{1,1000}$')  # то же, что выше, но без пробелов

# копипаста из passport.models.password, чтобы не тащить паспортные модели и их зависимости
PASSWORD_CHANGING_REASON_FLUSHED_BY_ADMIN = '2'


class BaseIssueTokenView(BaseOldApiView):
    allowed_methods = ['POST']
    grant_type: Optional[str] = None
    issue_refresh_token: bool = False
    add_uid_to_response: bool = False

    device_info: Optional[Dict[str, str]]
    device_id: Optional[str]
    device_name: Optional[str]
    requested_scopes: Optional[Set[Scope]]
    is_xtoken: bool

    def __init__(self):
        super(BaseIssueTokenView, self).__init__()
        self.request = None
        self.client = None
        self.device_info = None
        self.device_id = None
        self.device_name = None
        self.requested_scopes = None
        self.is_xtoken = False
        self.token = None

    @classmethod
    def is_allowed(cls):
        return True

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='issue_token',
            grant_type=self.grant_type,
        )

    @cached_property
    def antifraud_logger(self):
        return AntifraudLogger()

    @cached_property
    def credentials_logger(self):
        return CredentialsLogger()

    @cached_property
    def antifraud_score(self):
        return AntifraudScoreManager(
            initial_data=dict(
                channel='auth',
                sub_channel=settings.ANTIFRAUD_AUTH_SUB_CHANNEL,
                request='auth',
                grant_type=self.grant_type,
                surface='oauth_' + self.grant_type,
                t=int(time() * 1000),
            ),
        )

    @cached_property
    def blackbox(self):
        return get_blackbox()

    @cached_property
    def passport(self):
        return get_passport()

    @cached_property
    def fast_passport(self):
        return get_passport(timeout=settings.FAST_PASSPORT_TIMEOUT)

    @cached_property
    def region_info(self):
        user_ip = self.request.env.user_ip
        return Region(ip=str(user_ip))

    @cached_property
    def user_AS(self):
        region_info = self.region_info
        return region_info.AS_list[0] if region_info.AS_list else None

    @property
    def allow_fallback_to_stateless(self):
        return True

    def check_request_structure(self, request):
        pass

    def validate_client(self):
        pass

    def check_scope_and_client_grants(self, scopes=None):
        assert_is_allowed(
            scope_list=scopes or self.client.scopes,
            grant_type=self.grant_type,
            client_id=self.client.display_id,
            ip=self.request.env.consumer_ip,
            uid=None,
        )
        restrict_non_yandex_clients(
            grant_type=self.grant_type,
            client=self.client,
            user_ip=self.request.env.user_ip,
        )

    @property
    def can_ignore_counters(self):
        return False  # может переопределяться в потомках

    def _counter_to_key_mapping(self, uid, client_id, user_ip):
        counter_to_key_mapping = {}
        if COUNTER_UID in self.rate_limit_counters:
            counter_to_key_mapping[COUNTER_UID] = 'grant_type:%s:%s:%s' % (self.grant_type, COUNTER_UID, uid)
        if COUNTER_IP in self.rate_limit_counters:
            counter_to_key_mapping[COUNTER_IP] = 'grant_type:%s:%s:%s' % (self.grant_type, COUNTER_IP, user_ip)
        if COUNTER_CLIENT_ID in self.rate_limit_counters:
            counter_to_key_mapping[COUNTER_CLIENT_ID] = 'grant_type:%s:%s:%s' % (
                self.grant_type,
                COUNTER_CLIENT_ID,
                client_id,
            )
        return counter_to_key_mapping

    def try_check_counters(self, uid, client_id, user_ip):
        if (
            not self.rate_limit_counters or
            self.can_ignore_counters
        ):
            return

        self.check_counters(
            counter_to_key_mapping=self._counter_to_key_mapping(uid=uid, client_id=client_id, user_ip=user_ip),
            uid=uid,
            client_id=client_id,
        )

    def try_increase_counters(self, uid, client_id, user_ip):
        if not self.rate_limit_counters:
            return

        self.increase_counters(
            counter_to_key_mapping=self._counter_to_key_mapping(uid=uid, client_id=client_id, user_ip=user_ip),
        )

    def get_token_params(self, uid):
        meta = self.get_optional_param('x_meta')
        if meta and not META_REGEX.match(meta):
            raise OldApiError(INVALID_REQUEST, META_INVALID)

        force_issue_stateless = self.get_optional_boolean_param('x_stateless')
        if (
            force_issue_stateless or
            should_issue_stateless(
                uid=uid,
                client_id=self.client.display_id,
                client_create_time=self.client.created,
                app_id=self.device_info.get('app_id'),
                app_platform=self.device_info.get('app_platform'),
                app_version=self.device_info.get('app_version'),
                is_xtoken=self.is_xtoken,
            )
        ):
            token_type = TOKEN_TYPE_STATELESS
        else:
            token_type = TOKEN_TYPE_NORMAL

        app_platform = None
        if token_type == TOKEN_TYPE_NORMAL and self.is_xtoken:
            # Храним только для stateful х-токенов, так как требуется только для показа списка устройств
            app_platform = self.device_info.get('app_platform')

        return dict(
            meta=meta,
            raise_error_if_limit_exceeded=self.get_optional_param('on_limit') == ERROR,
            device_id=self.device_id,
            device_name=self.device_name,
            app_platform=app_platform,
            token_type=token_type,
        )

    def _check_account_type_limits_for_client(self, bb_response):
        """Некоторые неполноценные типы аккаунта могут использовать лишь ограниченное множество приложений"""
        for alias_type in settings.LIMITED_ACCOUNT_TYPES:
            if get_alias(bb_response, alias_type):
                try:
                    check_grant(
                        grant='client.%s' % self.client.display_id,
                        consumer='grant_type:%s/%s' % (self.grant_type, alias_type),
                        ip=self.request.env.consumer_ip,
                        service_ticket=None,  # консумер фейковый, поэтому тикет не проверяем
                    )
                except GrantsMissingError:
                    self.log_failed_auth(
                        uid=bb_response['uid'],
                        reason='account_type.forbidden',
                        account_type=alias_type,
                    )
                    raise OldApiError(INVALID_GRANT, ACCOUNT_TYPE_FORBIDDEN)

    def _check_client_limits_for_account_type(self, bb_response):
        """Некоторые служебные приложения могут использовать лишь ограниченное множество типов аккаунта"""
        if self.client.display_id not in settings.CLIENTS_LIMITED_BY_ACCOUNT_TYPE:
            return
        allowed_alias_types = settings.CLIENTS_LIMITED_BY_ACCOUNT_TYPE[self.client.display_id]
        if not any(
            get_alias(bb_response, alias_type)
            for alias_type in allowed_alias_types
        ):
            self.log_failed_auth(
                uid=bb_response['uid'],
                reason='account_type.forbidden',
                allowed_account_types=','.join(allowed_alias_types),
            )
            raise OldApiError(INVALID_GRANT, ACCOUNT_TYPE_FORBIDDEN)

    def check_is_spammer(self, bb_response, user_ip):
        try:
            assert_is_not_spammer(
                bb_response=bb_response,
                grant_type=self.grant_type,
                client=self.client,
                ip=user_ip,
            )
        except AccessDeniedError:
            self.log_failed_auth(reason='user.bad_karma')
            raise OldApiError(INVALID_GRANT, ACCOUNT_TYPE_FORBIDDEN)

    def check_is_child(self, bb_response, user_ip):
        try:
            assert_is_not_child(
                bb_response=bb_response,
                grant_type=self.grant_type,
                client=self.client,
                ip=user_ip,
            )
        except AccessDeniedError:
            self.log_failed_auth(reason='user.is_child')
            raise OldApiError(INVALID_GRANT, ACCOUNT_TYPE_FORBIDDEN)

    def check_account_type_limits(self, bb_response):
        self._check_account_type_limits_for_client(bb_response)
        self._check_client_limits_for_account_type(bb_response)

    def get_credentials(self):
        raise NotImplementedError()  # pragma: no cover

    def check_credentials(self, user_ip, **kwargs):
        raise NotImplementedError()  # pragma: no cover

    def check_antifraud_score(self):
        pass

    def bind_request_based_log_params(self):
        user_ip = self.request.env.user_ip
        self.statbox.bind_context(user_ip=user_ip)
        self.antifraud_score.bind(
            user_ip=user_ip,
            AS=self.user_AS,
            user_agent=self.request.env.user_agent,
            external_id='req-{}'.format(self.request.env.request_id),
            login_id=self.get_optional_param('login_id'),
            input_login=self.get_optional_param('username'),
        )

    def bind_scope_based_log_params(self):
        if self.requested_scopes:
            self.antifraud_score.bind(
                scopes=','.join(str(x) for x in self.requested_scopes),
            )

    def bind_extra_log_params(self, uid):
        params = dict(uid=uid, **self.device_info)
        self.statbox.bind(**params)
        self.antifraud_logger.bind_context(**params)
        self.antifraud_score.bind(**params)

    def log_failed_auth(self, reason, **kwargs):
        self.statbox.log(status='error', reason=reason, **kwargs)

    def issue_token(self, uid, password_passed=False, **kwargs):
        return issue_token(
            uid=uid,
            client=self.client,
            grant_type=self.grant_type,
            env=self.request.env,
            statbox_logger=self.statbox,
            antifraud_logger=self.antifraud_logger,
            credentials_logger=self.credentials_logger,
            allow_fallback_to_stateless=self.allow_fallback_to_stateless,
            password_passed=password_passed,
            **kwargs
        )

    def process_request(self, request):
        self.request = request
        user_ip = request.env.user_ip
        self.bind_request_based_log_params()

        self.get_and_bind_client()
        self.get_and_bind_device_info()

        self.validate_client()
        self.check_request_structure(request)
        self.requested_scopes = self.get_requested_scopes()
        self.bind_scope_based_log_params()
        self.is_xtoken = has_x_token_scope(self.requested_scopes or self.client.scopes)
        self.check_scope_and_client_grants(self.requested_scopes)

        uid, bb_response = self.check_credentials(
            user_ip=user_ip,
            **self.get_credentials()
        )
        if bb_response:
            self.check_account_type_limits(bb_response)
            self.check_is_spammer(bb_response, user_ip)
            self.check_is_child(bb_response, user_ip)

        self.try_check_counters(uid=uid, client_id=self.client.display_id, user_ip=user_ip)
        tokens_revoked_at = None
        if bb_response:
            tokens_revoked_at = get_revoke_time_from_bb_response(bb_response, REVOKER_TOKENS)

        self.bind_extra_log_params(uid)

        self.check_antifraud_score()

        token = self.issue_token(
            uid=uid,
            tokens_revoked_at=tokens_revoked_at,
            scopes=self.requested_scopes,
            login=bb_response['login'] if bb_response else None,
            **self.get_token_params(uid=uid)
        )

        self.token = token
        self.try_increase_counters(uid=uid, client_id=self.client.display_id, user_ip=user_ip)

        self.response_values.update(
            access_token=token.access_token,
            token_type=TOKEN_TYPE,
        )
        if token.expires:
            self.response_values.update(expires_in=token.ttl)

        if self.issue_refresh_token:
            # Важно выдавать refresh_token только после явного запроса разрешений у пользователя
            self.response_values.update(refresh_token=token.refresh_token)

        if self.add_uid_to_response:
            self.response_values.update(uid=uid)

        expected_scopes = self.requested_scopes or self.client.scopes
        if token.scopes != expected_scopes:
            self.response_values.update(scope=' '.join(map(str, token.scopes)))


class IssueTokenByAuthorizationCode(BaseIssueTokenView):
    grant_type = GRANT_TYPE_CODE
    issue_refresh_token = True

    @property
    def allow_fallback_to_stateless(self):
        return (
            self.client.is_yandex or
            settings.ALLOW_FALLBACK_TO_STATELESS_TOKENS_FOR_NON_YANDEX_CLIENTS_AND_PUBLIC_GRANT_TYPES
        )

    def get_credentials(self):
        verification_code = self.get_required_param('code')
        if not does_code_look_valid(verification_code):
            raise OldApiError(BAD_VERIFICATION_CODE, CODE_INVALID)

        return dict(
            code=verification_code,
            code_verifier=self.get_optional_param('code_verifier'),
        )

    def is_client_secret_required(self, client):
        # Не требуем секрет, если authorization_code защищён PKCE
        return (
            super(IssueTokenByAuthorizationCode, self).is_client_secret_required(client) and
            not self.get_optional_param('code_verifier')
        )

    def get_requested_scopes(self):
        pass  # скоупы берём не из запроса, а из token_request

    def check_credentials(self, user_ip, code, code_verifier):
        token_request = Request.by_verification_code(self.client.id, smart_bytes(code))
        if not token_request or token_request.is_expired:
            raise OldApiError(INVALID_GRANT, CODE_EXPIRED)
        if not token_request.is_accepted or not token_request.uid:
            # Недостижимый кейс, но важный: лучше перепроверить. Ошибку отдаём старую, для совместимости
            raise OldApiError(INVALID_GRANT, CODE_EXPIRED)
        self.statbox.bind(
            token_request_id=token_request.display_id,
            code_strength=code_strength_to_name(token_request.code_strength),
        )
        if not token_request.check_code_verifier(code_verifier):
            self.log_failed_auth(reason='code_verifier.not_matched')
            raise OldApiError(INVALID_GRANT, CODE_VERIFIER_NOT_MATCHED)
        if token_request.needs_activation:
            self.log_failed_auth(reason='code.not_activated')
            raise OldApiError(INVALID_GRANT, CODE_NOT_ACTIVATED)

        try:
            self.requested_scopes = self.validate_requested_scopes(token_request.scopes)
        except ValidationError:
            raise OldApiError(INVALID_SCOPE, SCOPES_CODE_AND_CLIENT_MISMATCH)

        if token_request.device_id:
            # значения из реквеста приоритетней
            self.device_id = token_request.device_id
            self.device_name = token_request.device_name
            self.device_info.update(
                device_id=self.device_id,
                device_name=self.device_name,
            )

        bb_response = self.blackbox.userinfo(
            uid=token_request.uid,
            ip=user_ip,
            dbfields=settings.BLACKBOX_DBFIELDS,
            attributes=settings.BLACKBOX_ATTRIBUTES,
            need_display_name=False,
        )
        if not bb_response['uid']:
            # юзер создал реквест и сразу удалился. Считаем реквест протухшим.
            self.log_failed_auth(reason='user.not_found')
            raise OldApiError(INVALID_GRANT, CODE_EXPIRED)
        elif token_request.is_invalidated_by_user_logout(
            revoke_time=get_revoke_time_from_bb_response(bb_response, REVOKER_WEB_SESSIONS),
        ):
            # юзер отозвал все веб-сессии. Считаем реквест протухшим.
            self.log_failed_auth(reason='request.expired')
            raise OldApiError(INVALID_GRANT, CODE_EXPIRED)

        self.token_request = token_request

        return bb_response['uid'], bb_response

    def get_token_params(self, uid):
        return dict(
            super(IssueTokenByAuthorizationCode, self).get_token_params(uid=uid),
            token_type=TOKEN_TYPE_NORMAL,  # не разрешаем получать stateless-токены с этим grant_type
            allow_reuse=True,  # важно для турбоаппов, см. PASSP-27185
            payment_auth_context_id=self.token_request.payment_auth_context_id,
            payment_auth_scope_addendum=self.token_request.payment_auth_scope_addendum,
            login_id=self.token_request.login_id or None,
            redirect_uri=self.token_request.redirect_uri,
        )

    def issue_token(self, **kwargs):
        token = super(IssueTokenByAuthorizationCode, self).issue_token(**kwargs)
        try:
            with DELETE(self.token_request):
                pass
        finally:
            return token


class IssueTokenByDeviceCode(BaseIssueTokenView):
    grant_type = GRANT_TYPE_DEVICE_CODE
    issue_refresh_token = True

    @property
    def allow_fallback_to_stateless(self):
        return (
            self.client.is_yandex or
            settings.ALLOW_FALLBACK_TO_STATELESS_TOKENS_FOR_NON_YANDEX_CLIENTS_AND_PUBLIC_GRANT_TYPES
        )

    def get_credentials(self):
        device_code = self.get_required_param('code')
        if len(device_code) != settings.REQUEST_ID_LENGTH:
            raise OldApiError(INVALID_GRANT, CODE_EXPIRED)
        return dict(
            code=device_code,
        )

    def get_requested_scopes(self):
        pass  # скоупы берём не из запроса, а из token_request

    def check_credentials(self, user_ip, code):
        token_request = Request.by_display_id(smart_bytes(code))
        if not token_request or not token_request.is_accepted or not token_request.uid:
            raise OldApiError(AUTHORIZATION_PENDING, USER_ACTION_REQUIRED)
        if token_request.is_expired:
            raise OldApiError(INVALID_GRANT, CODE_EXPIRED)
        self.statbox.bind(
            token_request_id=token_request.display_id,
        )

        if token_request.client_id != self.client.id:
            raise OldApiError(INVALID_GRANT, CLIENT_ID_MISMATCH)
        try:
            self.requested_scopes = self.validate_requested_scopes(token_request.scopes)
        except ValidationError:
            raise OldApiError(INVALID_SCOPE, SCOPES_CODE_AND_CLIENT_MISMATCH)

        if token_request.device_id:
            self.device_id = token_request.device_id
            self.device_name = token_request.device_name
            self.device_info.update(
                device_id=self.device_id,
                device_name=self.device_name,
            )

        bb_response = self.blackbox.userinfo(
            uid=token_request.uid,
            ip=user_ip,
            dbfields=settings.BLACKBOX_DBFIELDS,
            attributes=settings.BLACKBOX_ATTRIBUTES,
            need_display_name=False,
        )
        if not bb_response['uid']:
            # юзер заакцептил реквест и сразу удалился. Считаем реквест протухшим.
            self.log_failed_auth(reason='user.not_found')
            raise OldApiError(INVALID_GRANT, CODE_EXPIRED)
        elif token_request.is_invalidated_by_user_logout(
            revoke_time=get_revoke_time_from_bb_response(bb_response, REVOKER_WEB_SESSIONS),
        ):
            # юзер отозвал все веб-сессии. Считаем реквест протухшим.
            self.log_failed_auth(reason='request.expired')
            raise OldApiError(INVALID_GRANT, CODE_EXPIRED)

        self.token_request = token_request

        return bb_response['uid'], bb_response

    def get_token_params(self, uid):
        return dict(
            super(IssueTokenByDeviceCode, self).get_token_params(uid=uid),
            token_type=TOKEN_TYPE_NORMAL,  # не разрешаем получать stateless-токены с этим grant_type
        )

    def issue_token(self, **kwargs):
        token = super(IssueTokenByDeviceCode, self).issue_token(**kwargs)
        try:
            with DELETE(self.token_request):
                pass
        finally:
            return token


class IssueTokenByPassword(BaseIssueTokenView):
    grant_type = GRANT_TYPE_PASSWORD
    add_uid_to_response = True

    @cached_property
    def country(self):
        return get_country_code_by_ip(self.request.env.user_ip)

    @cached_property
    def is_yandex_user_or_server_ip(self):
        return (
            is_yandex_ip(self.request.env.user_ip) or
            is_yandex_server_ip(self.request.env.user_ip)
        )

    @property
    def rate_limit_counters(self):
        return {
            COUNTER_UID: settings.TOKEN_RATE_LIMIT_UID,
            COUNTER_IP: settings.TOKEN_RATE_LIMIT_IP,
        }

    def check_antifraud_score(self):
        result = self.antifraud_score.check_antifraud_score()
        if not result:
            log.debug('Denying based on antifraud score')
            antifraud_response = self.antifraud_score.score_response
            self.statbox.log(
                status='warning' if settings.ANTIFRAUD_SCORE_DRY_RUN else 'error',
                reason='antifraud_score_deny',
                antifraud_action=antifraud_response.action,
                antifraud_reason=antifraud_response.reason,
                antifraud_tags=','.join(antifraud_response.tags or []),
            )
            if not settings.ANTIFRAUD_SCORE_DRY_RUN:
                raise OldApiError(INVALID_GRANT, LOGIN_PASSWORD_INVALID)

    def get_captcha_params(self):
        params = self.request.POST
        if 'x_captcha_key' in params or 'x_captcha_answer' in params:
            if not ('x_captcha_key' in params and 'x_captcha_answer' in params):
                raise OldApiError(
                    INVALID_REQUEST,
                    POST_PARAM_MISSING % 'x_captcha_key or x_captcha_answer',
                )
        return {
            'captcha_key': self.get_optional_param('x_captcha_key'),
            'captcha_answer': self.get_optional_param('x_captcha_answer'),
            'captcha_scale_factor': int_or_default(self.get_optional_param('x_captcha_scale_factor'), default=1),
        }

    def get_credentials(self):
        return merge_dicts(
            self.get_captcha_params(),
            dict(
                username=self.get_required_param('username'),
                password=self.get_required_param('password'),
                authtype='oauthcreate',
            ),
        )

    @property
    def can_ignore_counters(self):
        return self.is_captcha_checked

    def raise_rate_limit_error(self, source=CAPTCHA_SOURCE_RATE_LIMIT):
        self.show_captcha(
            scale_factor=self.captcha_scale_factor,
            source=source,
        )

    @staticmethod
    def _get_field_value(request, field):
        if field.startswith('env.'):
            return getattr(request.env, field[4:], None)
        else:
            return request.REQUEST.get(field, None)

    def validate_client(self):
        is_outdated_client = self.client.display_id in settings.GRANT_TYPE_PASSWORD_OUTDATED_CLIENTS
        if is_outdated_client and not self.is_yandex_user_or_server_ip:
            if self.country in settings.OUTDATED_CLIENT_COUNTRIES_LANG_RU:
                message = settings.OUTDATED_CLIENT_MESSAGE_RU
            else:
                message = settings.OUTDATED_CLIENT_MESSAGE
            self.log_failed_auth(reason='client.outdated')
            raise OldApiError(error=BAD_FLOW, description=message)

    def log_failed_auth_to_antifraud(self, comment):
        self.antifraud_logger.log(
            channel='auth',
            sub_channel=settings.ANTIFRAUD_AUTH_SUB_CHANNEL,
            external_id='req-{}'.format(self.request.env.request_id),
            status='FAILED',
            comment=comment,
            grant_type=self.grant_type,
            client_id=self.client.display_id,
            is_client_yandex=self.client.is_yandex,
            device_id=self.device_id,
            scopes=','.join(map(str, self.requested_scopes or self.client.scopes)),
            password_passed=True,
            is_app_password=False,
            ip=self.request.env.user_ip,
            AS=self.user_AS,
            user_agent=self.request.env.user_agent,
        )

    def log_failed_auth(self, reason, **kwargs):
        super(IssueTokenByPassword, self).log_failed_auth(reason, **kwargs)
        self.log_failed_auth_to_antifraud(comment=reason)

    def check_request_structure(self, request):
        user_entered_login = self._get_field_value(request, 'username')
        cl_id_conf = settings.GRANT_TYPE_PASSWORD_REQUEST_VALIDATORS_BY_CLIENT_ID
        if self.client.display_id not in cl_id_conf:
            return
        validator = StructureValidator(
            statbox=self.statbox,
            user_entered_login=user_entered_login,
            **cl_id_conf[self.client.display_id]
        )
        validator.validate(request)
        if validator.ok and validator.ignored_by_rule:
            log.debug(
                'Client {} request structure validation ignored ({} is in ignore list)'.format(
                    self.client.display_id,
                    validator.ignored_by_rule,
                ),
            )
        if not validator.ok:
            log.debug(
                'Client {} request structure validation failed (dry_run={}): {}'.format(
                    self.client.display_id,
                    validator.experiment_dry_run,
                    ', '.join(validator.errors)
                ),
            )
            self.statbox.log(
                user_entered_login=user_entered_login,
                device_id=self.device_id,
                fields=','.join(validator.error_fields),
                status='warning' if validator.experiment_dry_run else 'error',
                reason='request.invalid_fields',
            )
            if not validator.experiment_dry_run:
                self.log_failed_auth_to_antifraud(comment='request.invalid_fields')
                raise OldApiError(BAD_FLOW, '', status=401)

    def try_check_captcha(self, login, captcha_key, captcha_answer, captcha_scale_factor):
        self.captcha_scale_factor = captcha_scale_factor
        captcha_checked = False
        if captcha_key or captcha_answer:
            if captcha_key and captcha_answer:
                # для специальных тестовых аккаунтов в тестовом окружении примем любую капчу
                if (
                    any(login.startswith(prefix) for prefix in settings.YANDEX_TEST_LOGIN_PREFIXES) and
                    settings.DEGRADATE_CAPTCHA_FOR_TEST_LOGINS
                ):
                    captcha_checked = True
                    log.debug('issue token: captcha accepted for test login "%s"' % login)
                else:
                    # проверим key и answer: с заведомо неправильными данными бессмысленно идти в капча-сервер
                    if (
                        len(captcha_key) <= settings.MAX_CAPTCHA_KEY_LENGTH and
                        len(captcha_answer) <= settings.MAX_CAPTCHA_ANSWER_LENGTH
                    ):
                        try:
                            captcha_checked = captcha.check(
                                key=captcha_key,
                                answer=captcha_answer,
                                request_id=self.request.env.request_id,
                            )
                        except CaptchaLocateError:
                            pass
            if not captcha_checked:
                self.log_failed_auth(reason='captcha.wrong_answer')
                captcha_key, captcha_url = captcha.generate(
                    scale_factor=captcha_scale_factor,
                    request_id=self.request.env.request_id,
                )
                raise OldApiError(
                    error='403',
                    description=CAPTCHA_ANSWER_WRONG,
                    status=403,
                    x_captcha_key=captcha_key,
                    x_captcha_url=captcha_url,
                )
        return captcha_checked

    def show_captcha(self, scale_factor, source):
        self.log_failed_auth(reason='captcha.required', captcha_source=source)
        captcha_key, captcha_url = captcha.generate(
            scale_factor=scale_factor,
            request_id=self.request.env.request_id,
        )
        raise OldApiError(
            error='403',
            description=CAPTCHA_REQUIRED,
            status=403,
            x_captcha_key=captcha_key,
            x_captcha_url=captcha_url,
        )

    def check_for_new_auth_on_device(self, uid, device_id, tokens_revoked_at):
        if not device_id:
            self.statbox.bind(auth_on_device='unknown')
            return True
        else:
            all_tokens_with_clients = list_tokens_with_clients_by_user(
                uid,
                tokens_revoked_at=tokens_revoked_at,
                app_passwords_revoked_at=datetime.now(),  # сэкономим: не будем получать ПП, они нам не нужны
            )
            if any(
                token.device_id == device_id
                for token, _, is_valid in all_tokens_with_clients
                if is_valid
            ):
                self.statbox.bind(auth_on_device='already_existed')
                return False
            else:
                self.statbox.bind(auth_on_device='new')
                return True

    def check_if_ip_is_trusted(self, ip, uid):
        current_region = Region(ip=str(ip))
        if current_region.AS_list and set(current_region.AS_list) & settings.BLACKLISTED_AS_LIST:
            self.statbox.bind(ip_check_result='AS_blacklisted')
            return False

        self.statbox.bind(ip_check_result='unknown')
        # TODO: возможно, научиться забирать профили пользователя и проверять IP по ним
        return True

    def is_password_expired(self, bb_response):
        """
        Нужно ли пользователю сменить пароль или создать новый
        (все причины, кроме подозрения о взломе)
        """
        if bb_response['status'] == BLACKBOX_LOGIN_V1_EXPIRED_STATUS:
            # протухший пароль у аккаунта с 67 сидом
            self.statbox.bind(password_expire_reason='strong_password_policy')
            return True
        elif bb_response['status'] == BLACKBOX_LOGIN_V1_VALID_STATUS:
            if get_attribute(bb_response, settings.BB_ATTR_PASSWORD_CREATING_REQUIRED):
                # автозарегистрированный пользователь
                self.statbox.bind(password_expire_reason='incomplete_autoregistered')
                return True
            elif get_alias(bb_response, 'pdd') and not get_attribute(bb_response, settings.BB_ATTR_PDD_AGREEMENT_ACCEPTED) and not get_alias(bb_response, 'portal'):
                # недорегистрированный ПДД
                self.statbox.bind(password_expire_reason='incomplete_pdd')
                return True
            elif get_attribute(bb_response, settings.BB_ATTR_PASSWORD_CHANGE_REASON) == PASSWORD_CHANGING_REASON_FLUSHED_BY_ADMIN:
                # ПДД, которому админ сбросил пароль
                self.statbox.bind(password_expire_reason='flushed_pdd')
                return True

        return False

    def check_credentials(self, user_ip, username, password, authtype,
                          captcha_key, captcha_answer, captcha_scale_factor):
        if not username:
            self.log_failed_auth(reason='login.empty')
            raise OldApiError(INVALID_GRANT, LOGIN_PASSWORD_INVALID)
        if not password:
            self.log_failed_auth(reason='password.empty')
            raise OldApiError(INVALID_GRANT, LOGIN_PASSWORD_INVALID)
        self.statbox.bind(login=username)

        self.is_captcha_checked = self.try_check_captcha(
            login=username,
            captcha_key=captcha_key,
            captcha_answer=captcha_answer,
            captcha_scale_factor=captcha_scale_factor,
        )

        # Для некоторых тестовых аккаунтов капчу показываем всегда, если она не разгадана
        if username.startswith(settings.YANDEX_CAPTCHA_ALWAYS_TEST_LOGIN_PREFIX) and not self.is_captcha_checked:
            self.show_captcha(captcha_scale_factor, source=CAPTCHA_SOURCE_TEST)

        # Для некоторых тестовых логинов в тестовом окружении капчу показывать не будем
        # (сделаем вид, что её разгадали)
        if (
            not self.is_captcha_checked and
            username.startswith(settings.YANDEX_CAPTCHA_NEVER_TEST_LOGIN_PREFIX) and
            settings.DEGRADATE_CAPTCHA_FOR_TEST_LOGINS
        ):
            log.debug('issue_token: skipping captcha for test login "%s"' % username)
            self.is_captcha_checked = True

        is_magnitola = (
            self.client.display_id == settings.AM_CLIENT_ID and
            self.device_info.get('app_id') in settings.MAGNITOLA_APP_IDS
        )
        if is_magnitola:
            authtype = 'magnitola'

        try:
            bb_response = self.blackbox.login(
                login=username,
                password=password,
                ip=user_ip,
                version=1,
                dbfields=settings.BLACKBOX_DBFIELDS,
                attributes=settings.BLACKBOX_ATTRIBUTES,
                need_aliases=True,
                need_display_name=False,
                captcha_already_checked=self.is_captcha_checked,
                authtype=authtype,
                find_by_phone_alias=BLACKBOX_FIND_BY_PHONE_ALIAS_FORCE_ON,
                country=self.country,
            )
        except BlackboxInvalidParamsError:
            self.log_failed_auth(reason='blackbox_params.invalid')
            raise OldApiError(INVALID_GRANT, BLACKBOX_PARAMS_INVALID)

        uid = bb_response.get('uid')
        if isinstance(uid, dict):
            # При некоторых статусах ответа (например, EXPIRED) билдер не парсит ответ.
            # Но нам важно знать uid, чтобы записать его в логи.
            uid = int(uid['value']) if uid.get('value') else None
        self.statbox.bind_context(uid=uid)
        self.antifraud_logger.bind_context(uid=uid)

        self.antifraud_score.bind(badauth_counts=bb_response.get('badauth_counts'))

        # Сначала проверяем необходимость капчи
        if bb_response['status'] == BLACKBOX_LOGIN_V1_SHOW_CAPTCHA_STATUS:
            if (
                self.is_yandex_user_or_server_ip or
                self.country in settings.GRANT_TYPE_PASSWORD_CAPTCHA_COUNTRIES
            ):
                self.show_captcha(captcha_scale_factor, source=CAPTCHA_SOURCE_BLACKBOX)
            else:
                self.log_failed_auth(
                    reason='captcha.unsupported_country',
                    captcha_source=CAPTCHA_SOURCE_BLACKBOX,
                )
                raise OldApiError(INVALID_GRANT, LOGIN_PASSWORD_INVALID)
        elif self.is_password_expired(bb_response):
            self.log_failed_auth(reason='password.expired')
            raise OldApiError('403', PASSWORD_EXPIRED, status=403)
        elif bb_response['status'] == BLACKBOX_LOGIN_V1_DISABLED_STATUS:
            # Пользователь заблокирован. Увы, из совместимости говорим, что пароль неверен :(
            self.log_failed_auth(reason='user.blocked')
            raise OldApiError(INVALID_GRANT, LOGIN_PASSWORD_INVALID)
        elif (
            get_attribute(bb_response, settings.BB_ATTR_SMS_2FA_ON) and
            self.client.display_id not in settings.ALLOW_SMS_2FA_IN_GT_PASSWORD_FOR_CLIENTS
        ):
            self.log_failed_auth(reason='sms_2fa.enabled')
            raise OldApiError(INVALID_GRANT, LOGIN_PASSWORD_INVALID)
        elif bb_response['status'] == BLACKBOX_LOGIN_V1_SECOND_STEP_REQUIRED_STATUS:
            self.log_failed_auth(
                reason='second_step.not_implemented',
                allowed_steps=','.join(bb_response['allowed_second_steps']),
            )
            raise OldApiError(INVALID_GRANT, LOGIN_PASSWORD_INVALID)
        elif bb_response['status'] != BLACKBOX_LOGIN_V1_VALID_STATUS:
            self.log_failed_auth(reason='password.invalid')
            raise OldApiError(INVALID_GRANT, LOGIN_PASSWORD_INVALID)
        elif get_attribute(bb_response, settings.BB_ATTR_PASSWORD_CHANGE_REASON):
            self.log_failed_auth(reason='password.change_required')
            raise OldApiError('403', PASSWORD_CHANGE_REQUIRED, status=403)
        elif is_magnitola and not get_attribute(bb_response, settings.BB_ATTR_APP_PASSWORDS_ON):
            self.log_failed_auth(reason='app_passwords.off')
            raise OldApiError(INVALID_GRANT, LOGIN_PASSWORD_INVALID)

        # Сходим в UFO, проверим, из доверенного ли окружения юзер пытается авторизоваться.
        is_ip_trusted = self.check_if_ip_is_trusted(
            ip=user_ip,
            uid=uid,
        )
        # Проверим, с нового ли устройства авторизация?
        is_new_device = self.check_for_new_auth_on_device(
            uid=uid,
            device_id=self.device_id,
            tokens_revoked_at=get_revoke_time_from_bb_response(bb_response, REVOKER_TOKENS),
        )

        if is_new_device:
            email_sent = False
            is_challenged = not is_ip_trusted
            try:
                rv = self.fast_passport.send_challenge_email(
                    uid=uid,
                    device_name=self.device_name,
                    user_ip=user_ip,
                    user_agent=self.request.env.user_agent,
                    is_challenged=is_challenged,
                )
                email_sent = rv.get('email_sent', False)
            except BasePassportError as e:
                log_warning('Failed to send challenge email via passport: %s' % e)

            self.statbox.bind(challenge_email_sent=email_sent, is_challenged=is_challenged)

            if is_challenged and not self.is_captcha_checked:
                self.show_captcha(captcha_scale_factor, source=CAPTCHA_SOURCE_PROFILE)

        return uid, bb_response

    def get_token_params(self, uid):
        params = dict(
            super(IssueTokenByPassword, self).get_token_params(uid=uid),
            password_passed=True,
        )
        if settings.ACCEPT_LOGIN_ID_FOR_GT_PASSWORD:
            params.update(
                login_id=self.get_optional_param('login_id'),
            )
        return params


class IssueTokenBySessionid(BaseIssueTokenView):
    grant_type = GRANT_TYPE_SESSIONID
    add_uid_to_response = True

    @property
    def rate_limit_counters(self):
        return {
            COUNTER_UID: settings.TOKEN_RATE_LIMIT_UID,
            COUNTER_IP: settings.TOKEN_RATE_LIMIT_IP,
        }

    def get_credentials(self):
        uid = self.get_optional_param('uid')
        try:
            uid = int(uid) if uid else None
        except ValueError:
            raise OldApiError(INVALID_REQUEST, POST_PARAM_NUMBER_REQUIRED % 'uid')

        host = self.get_required_param('host')

        cookies = self.request.env.cookies
        sessionid_from_cookie = cookies.get('Session_id') if cookies else None
        sessionid_from_param = self.get_optional_param('sessionid')

        if sessionid_from_param is None and not cookies:
            raise OldApiError(INVALID_REQUEST, COOKIE_HEADER_MISSING)

        sessionid = sessionid_from_cookie or sessionid_from_param
        all_cookies_passed = bool(cookies) and sessionid_from_param is None
        if all_cookies_passed:
            sessguard = cookies.get('sessguard') if cookies else None
        else:
            sessguard = None
            if settings.ALLOW_SKIP_SESSGUARD_FOR_LEGACY_GT_SESSIONID_USERS:
                # Заменяем точный хост (вида mail.yandex.ru) общим (вида yandex.ru).
                # Тогда sessguard не будет требоваться.
                host = strip_to_yandex_domain(host)

        # Запишем в statbox, уверены ли мы, что потребитель правильно передаёт все куки только в хедере
        self.statbox.bind_context(all_cookies_passed=all_cookies_passed)

        return dict(
            sessionid=sessionid,
            sessguard=sessguard,
            host=host,
            uid=uid,
        )

    def get_token_params(self, uid):
        return dict(
            super(IssueTokenBySessionid, self).get_token_params(uid=uid),
            login_id=self.login_id,
        )

    def check_credentials(self, user_ip, sessionid, sessguard, host, uid):
        if not sessionid:
            self.log_failed_auth(reason='sessionid.empty')
            raise OldApiError(INVALID_GRANT, SESSION_INVALID)

        if (
            settings.REQUIRE_PASSPORT_HOST_FOR_XTOKEN_BY_GT_SESSIONID and
            self.is_xtoken and
            not is_passport_domain(host)
        ):
            self.log_failed_auth(reason='host.not_passport')
            raise OldApiError(INVALID_GRANT, SESSION_INVALID)

        try:
            to_statbox(
                mode='check_cookies',
                host=host,
                have_sessguard=sessguard is not None,
                app_id=self.device_info.get('app_id'),
                sessionid=mask_sessionid(sessionid),
            )
            bb_response = self.blackbox.sessionid(
                sessionid=sessionid,
                sessguard=sessguard,
                host=host,
                ip=user_ip,
                dbfields=settings.BLACKBOX_DBFIELDS,
                attributes=settings.BLACKBOX_ATTRIBUTES,
                need_display_name=False,
                multisession=True,
                get_login_id=True,
                request_id=self.request.env.request_id,
            )
        except BlackboxInvalidParamsError:
            self.log_failed_auth(reason='blackbox_params.invalid')
            raise OldApiError(INVALID_GRANT, BLACKBOX_PARAMS_INVALID)

        # Сначала проверим статус всей куки
        if bb_response['cookie_status'] == BLACKBOX_SESSIONID_WRONG_GUARD_STATUS:
            self.log_failed_auth(reason='sessguard.invalid', cookie_status=bb_response['cookie_status'])
            raise OldApiError(INVALID_GRANT, SESSION_INVALID)
        elif bb_response['cookie_status'] not in [
            BLACKBOX_SESSIONID_VALID_STATUS,
            BLACKBOX_SESSIONID_NEED_RESET_STATUS,
        ]:
            self.log_failed_auth(reason='sessionid.invalid', cookie_status=bb_response['cookie_status'])
            raise OldApiError(INVALID_GRANT, SESSION_INVALID)

        # Теперь проверим наличие запрошенного юзера в куке
        uid = uid or bb_response['default_uid']
        users = bb_response['users']
        user_session = users.get(uid)
        if not user_session:
            self.log_failed_auth(
                reason='sessionid.no_uid',
                known_uids=','.join(map(str, sorted(users.keys()))),
            )
            raise OldApiError(INVALID_GRANT, UID_MISSING)

        # И наконец - статус данного юзера
        if user_session['status'] not in [
            BLACKBOX_SESSIONID_VALID_STATUS,
            BLACKBOX_SESSIONID_NEED_RESET_STATUS,
        ]:
            self.log_failed_auth(reason='sessionid.invalid', user_status=user_session['status'])
            raise OldApiError(INVALID_GRANT, SESSION_INVALID)

        self.login_id = bb_response['login_id']
        self.statbox.bind_context(old_authid=bb_response['authid']['id'])

        return uid, user_session


class IssueTokenByXToken(BaseIssueTokenView):
    grant_type = GRANT_TYPE_XTOKEN
    add_uid_to_response = True

    def get_credentials(self):
        return dict(
            access_token=self.get_required_param('access_token'),
            payment_auth_context_id=self.get_optional_param('payment_auth_context_id'),
        )

    def get_token_params(self, uid):
        return dict(
            super(IssueTokenByXToken, self).get_token_params(uid=uid),
            # ЧЯ не может проверить валидность stateless-хтокена по его id - передаём id только для обычных хтокенов
            x_token_id=self.x_token.id if not isinstance(self.x_token, StatelessToken) else None,
            payment_auth_context_id=self.payment_auth_context_id,
            payment_auth_scope_addendum=self.payment_auth_scope_addendum,
            login_id=self.login_id,
            passport_track_id=self.get_optional_param('passport_track_id'),
        )

    def check_payment_auth(self, uid, payment_auth_context_id):
        scopes = self.requested_scopes or self.client.scopes
        if not is_payment_auth_required(scopes):
            self.payment_auth_context_id = None
            self.payment_auth_scope_addendum = None
            return

        money_payment_auth_api = get_money_payment_auth_api()
        if not payment_auth_context_id:
            payment_auth_retpath = self.get_required_param('payment_auth_retpath')
            rv = money_payment_auth_api.create_auth_context(
                uid=uid,
                client_id=self.client.display_id,
                scopes=[s.keyword for s in scopes],
                retpath=payment_auth_retpath,
                request_id='-',
            )
            raise OldApiError(
                PAYMENT_AUTH_PENDING,
                PAYMENT_AUTH_REQUIRED,
                payment_auth_context_id=rv['authContextId'],
                payment_auth_url=rv['redirectUri'],
                payment_auth_app_ids=get_payment_auth_app_ids(scopes),
            )

        rv = money_payment_auth_api.check_auth_context(
            auth_context_id=payment_auth_context_id,
        )
        if not (
            rv['status'] == PAYMENT_AUTH_SUCCEEDED_STATUS and
            str(rv['uid']) == str(uid) and
            rv['clientId'] == self.client.display_id and
            set(rv['scope'].split()) == set([s.keyword for s in scopes])
        ):
            raise OldApiError(PAYMENT_AUTH_PENDING, PAYMENT_AUTH_NOT_PASSED)

        self.payment_auth_context_id = payment_auth_context_id
        self.payment_auth_scope_addendum = rv['scopeAddendum']

    def check_credentials(self, user_ip, access_token, payment_auth_context_id):
        if not access_token or len(access_token) < 5:
            # Не ходим в ЧЯ с пустым токеном. Также отсекаем 'null', 'None', '-', etc.
            raise OldApiError(INVALID_GRANT, TOKEN_EXPIRED)

        self.statbox.bind_context(**self.device_info)

        bb_response = get_blackbox().oauth(
            oauth_token=access_token,
            ip=user_ip,
            dbfields=settings.BLACKBOX_DBFIELDS,
            attributes=settings.BLACKBOX_ATTRIBUTES,
            need_aliases=True,
            need_token_attributes=True,
            need_client_attributes=True,
            need_display_name=False,
            get_login_id=True,
        )
        if bb_response['status'] != BLACKBOX_OAUTH_VALID_STATUS:
            self.log_failed_auth(
                reason='x_token.invalid',
                bb_status=bb_response['status'],
                bb_error=bb_response.get('error'),
            )
            # Передавая такой description, теряем в информативности, зато сохраняем совместимость
            raise OldApiError(INVALID_GRANT, TOKEN_EXPIRED)

        self.x_token = parse_token(bb_response)
        self.login_id = bb_response['login_id']
        self.statbox.bind_context(x_token_id=self.x_token.id)
        uid = self.x_token.uid

        if not self.x_token.has_xtoken_grant:
            self.log_failed_auth(
                uid=uid,
                reason='x_token.scope_missing',
                actual_scopes=';'.join(scope.keyword for scope in self.x_token.scopes),
            )
            raise OldApiError(INVALID_GRANT, SCOPE_X_TOKEN_GRANT_MISSING)

        if self.x_token.device_id and self.x_token.device_id != self.device_id:
            # Расхождение device_id. Залогируем для статистики
            self.statbox.log(
                uid=uid,
                action='device_id_verification',
                status='warning',
                x_token_device_id=self.x_token.device_id,
            )

        self.check_payment_auth(uid, payment_auth_context_id)

        return uid, bb_response


class IssueTokenByAssertion(BaseIssueTokenView):
    grant_type = GRANT_TYPE_ASSERTION

    def get_credentials(self):
        assertion = self.get_required_integer_param('assertion', error=INVALID_GRANT)
        self.statbox.bind(assertion=assertion)

        return dict(assertion=assertion)

    def check_credentials(self, user_ip, assertion):
        try:
            bb_response = self.blackbox.userinfo(
                uid=assertion,
                ip=user_ip,
                dbfields=settings.BLACKBOX_DBFIELDS,
                attributes=settings.BLACKBOX_ATTRIBUTES,
                need_display_name=False,
            )
            uid = bb_response['uid']
        except BlackboxInvalidParamsError:
            self.log_failed_auth(reason='blackbox_params.invalid')
            raise OldApiError(INVALID_GRANT, BLACKBOX_PARAMS_INVALID)

        if not uid:
            self.log_failed_auth(reason='user.not_found')
            raise OldApiError(INVALID_GRANT, USER_NOT_FOUND)

        if get_attribute(bb_response, settings.BB_ATTR_AVAILABLE) != '1':
            self.log_failed_auth(reason='user.disabled')
            raise OldApiError(INVALID_GRANT, USER_NOT_FOUND)

        return uid, bb_response


class IssueTokenByPassportAssertion(IssueTokenByAssertion):
    """Тот же assertion, но от Паспорта и без некоторых ограничений"""
    grant_type = GRANT_TYPE_PASSPORT_ASSERTION

    @property
    def rate_limit_counters(self):
        # IP не проверяем (на случай, если где-то он не пробрасывается правильно)
        return {
            COUNTER_UID: settings.TOKEN_RATE_LIMIT_UID,
        }

    def check_account_type_limits(self, bb_response):
        pass  # Паспорту можно всё

    def get_token_params(self, uid):
        is_xtoken_trusted = False
        if self.get_optional_boolean_param('set_is_xtoken_trusted'):
            if not self.is_xtoken:
                raise OldApiError(INVALID_REQUEST, 'Cannot set is_xtoken_trusted for non-xtoken')
            is_xtoken_trusted = True
        auth_source = self.get_optional_param('auth_source')

        return dict(
            super(IssueTokenByPassportAssertion, self).get_token_params(uid=uid),
            password_passed=self.get_optional_boolean_param('password_passed'),
            login_id=self.get_optional_param('login_id'),
            passport_track_id=self.get_optional_param('passport_track_id'),
            is_xtoken_trusted=is_xtoken_trusted,
            auth_source=auth_source,
        )

    def bind_extra_log_params(self, uid):
        super(IssueTokenByPassportAssertion, self).bind_extra_log_params(uid)
        self.statbox.bind(
            cloud_token=self.get_optional_param('cloud_token'),
        )


class IssueTokenBySshKey(BaseIssueTokenView):
    grant_type = GRANT_TYPE_SSH_KEY

    @cached_property
    def tvm(self):
        return get_tvm()

    @classmethod
    def is_allowed(cls):
        return settings.ALLOW_GRANT_TYPE_SSH_KEY

    def get_credentials(self):
        uid = self.get_optional_integer_param('uid')
        login = self.get_optional_param('login')
        ts = self.get_required_integer_param('ts')
        ssh_sign = self.get_required_param('ssh_sign')
        public_cert = self.get_optional_param('public_cert')
        self.statbox.bind(uid=uid, login=login)

        if uid and login:
            raise OldApiError(INVALID_REQUEST, MUTUALLY_EXCLUSIVE_PARAMS % 'uid and login')
        elif not uid and not login:
            raise OldApiError(INVALID_REQUEST, POST_PARAM_MISSING % 'uid or login')

        current_ts = int(time())
        if abs(current_ts - ts) > settings.TS_CHECK_DELTA:
            self.log_failed_auth(reason='ts.wrong')
            raise OldApiError(INVALID_REQUEST, TS_WRONG % current_ts)

        return dict(uid=uid, login=login, ts=ts, ssh_sign=ssh_sign, public_cert=public_cert)

    def check_credentials(self, user_ip, uid, login, ts, ssh_sign, public_cert):
        # Сначала проверим подпись
        if not ssh_sign:
            self.log_failed_auth(reason='ssh_sign.empty')
            raise OldApiError(INVALID_GRANT, SSH_SIGN_INVALID)
        rv = self.tvm.verify_ssh(
            uid=uid,
            login=login,
            string_to_sign='%s%s%s' % (ts, self.client.display_id, uid or login),
            signed_string=ssh_sign,
            public_cert=public_cert,
        )
        tvm_status = rv.get('status')
        if tvm_status != TVM_STATUS_OK:
            tvm_error = rv.get('error')
            if tvm_error == 'BB_REJECT':
                reason = 'user.not_found'
                error_description = USER_NOT_FOUND
            elif tvm_error == 'NO_SSH_KEY':
                reason = 'ssh_keys.not_found'
                error_description = SSH_KEYS_NOT_FOUND
            elif tvm_error == 'SSH_BROKEN':
                reason = 'ssh_sign.invalid'
                error_description = SSH_SIGN_INVALID
            else:
                reason = None
                error_description = rv.get('desc', '')
                self.statbox.bind(tvm_status=tvm_status, tvm_error=tvm_error)

            log.debug('TVM responded with %s', rv)
            self.log_failed_auth(reason=reason)
            raise OldApiError(INVALID_GRANT, error_description)

        self.statbox.bind(ssh_key_fingerprint=rv.get('fingerprint'))

        # Затем проверим uid/login и получим данные пользователя
        try:
            bb_response = self.blackbox.userinfo(
                uid=uid,
                login=login,
                ip=user_ip,
                dbfields=settings.BLACKBOX_DBFIELDS,
                attributes=settings.BLACKBOX_ATTRIBUTES,
                need_display_name=False,
            )
            uid = bb_response['uid']
        except BlackboxInvalidParamsError:
            # случай в реальности недостижимый, но перестрахуемся
            self.log_failed_auth(reason='blackbox_params.invalid')
            raise OldApiError(INVALID_GRANT, BLACKBOX_PARAMS_INVALID)

        if not uid:
            self.log_failed_auth(reason='user.not_found')
            raise OldApiError(INVALID_GRANT, USER_NOT_FOUND)

        return uid, bb_response


class IssueTokenByClientCredentials(BaseIssueTokenView):
    grant_type = GRANT_TYPE_CLIENT_CREDENTIALS

    @property
    def rate_limit_counters(self):
        return {
            COUNTER_CLIENT_ID: settings.TOKEN_RATE_LIMIT_CLIENT_ID,
        }

    def get_token_params(self, uid):
        # Не пробрасываем device_id и device_name из-за лимита на количество персонализированных токенов,
        # кроме того, сами по себе они нам мало интересны.
        params = super(IssueTokenByClientCredentials, self).get_token_params(uid=uid)
        params.pop('device_id', None)
        params.pop('device_name', None)
        return params

    def get_credentials(self):
        # А нет их.
        return {}

    def check_credentials(self, user_ip):
        # Передаём uid=0, чтобы в будущем иметь возможность показывать список токенов,
        # выданных на конкретное приложение
        return 0, None


class IssueTokenByRefreshToken(BaseIssueTokenView):
    grant_type = GRANT_TYPE_REFRESH_TOKEN
    issue_refresh_token = True

    def get_credentials(self):
        return dict(
            refresh_token=self.get_required_param('refresh_token'),
        )

    def check_credentials(self, user_ip, refresh_token):
        if refresh_token:
            access_token = get_access_token_from_refresh_token(refresh_token)
        else:
            access_token = None

        if not access_token:
            raise OldApiError(INVALID_GRANT, TOKEN_EXPIRED)

        bb_response = get_blackbox().oauth(
            oauth_token=access_token,
            ip=user_ip,
            dbfields=settings.BLACKBOX_DBFIELDS,
            attributes=settings.BLACKBOX_ATTRIBUTES,
            need_aliases=True,
            need_token_attributes=True,
            need_client_attributes=True,
            need_display_name=False,
        )
        if bb_response['status'] != BLACKBOX_OAUTH_VALID_STATUS:
            self.log_failed_auth(
                reason='refresh_token.invalid',
                bb_status=bb_response['status'],
                bb_error=bb_response.get('error'),
            )
            raise OldApiError(INVALID_GRANT, TOKEN_EXPIRED)

        self.bb_response = bb_response
        token_info = bb_response['oauth']
        token_id = token_info['token_id']
        client_id = token_info['client_id']
        uid = token_info['uid']
        self.statbox.bind_context(token_id=token_id)
        if client_id != self.client.display_id:
            self.log_failed_auth(
                reason='client.not_matched',
                refresh_token_client_id=client_id,
            )
            raise OldApiError(INVALID_GRANT, TOKEN_NOT_BELONG_TO_CLIENT)

        return uid, bb_response

    def check_account_type_limits(self, bb_response):
        pass  # если токен смогли получить - должны мочь и подновить

    def issue_token(self, **kwargs):
        self.requested_scopes = None  # входной параметр игнорируем, поле ответа scope отдавать не хотим
        token = parse_token(self.bb_response)
        return try_refresh_token(
            token,
            client=self.client,
            env=self.request.env,
            statbox=self.statbox,
        )

    def get_token_params(self, uid=None):
        return {}


class IssueTokenDispatcher(BaseOldApiView):
    allowed_methods = ['POST']

    VIEWS = {
        view.grant_type: view
        for view in (
            IssueTokenByAuthorizationCode,
            IssueTokenByDeviceCode,
            IssueTokenByPassword,
            IssueTokenBySessionid,
            IssueTokenByXToken,
            IssueTokenByAssertion,
            IssueTokenByPassportAssertion,
            IssueTokenBySshKey,
            IssueTokenByClientCredentials,
            IssueTokenByRefreshToken,
        )
    }

    def process_request(self, request):
        grant_type = self.get_required_param('grant_type')
        if grant_type not in self.VIEWS or not self.VIEWS[grant_type].is_allowed():
            raise OldApiError(UNSUPPORTED_GRANT, GRANT_TYPE_UNKNOWN)

        view = self.VIEWS[grant_type]()
        view.process_request(request)
        self.response_values = view.response_values


class RevokeTokenView(BaseOldApiView):
    allowed_methods = ['POST']

    def process_request(self, request):
        client = self.get_client_from_request()
        access_token = self.get_required_param('access_token')  # отступаю от RFC - там параметр называется token

        if not access_token:
            # нам передали пустую строку - она не является валидным токеном
            pass
        else:
            bb_response = get_blackbox().oauth(
                oauth_token=access_token,
                ip=request.env.user_ip,
                dbfields=settings.BLACKBOX_DBFIELDS,
                attributes=settings.BLACKBOX_ATTRIBUTES,
                need_token_attributes=True,
                need_client_attributes=True,
                need_display_name=False,
            )

            if bb_response['status'] != BLACKBOX_OAUTH_VALID_STATUS:
                # токена нет или он протух - всё уже сделано до нас
                pass
            else:
                token = parse_token(bb_response)

                if isinstance(token, StatelessToken):
                    raise OldApiError(UNSUPPORTED_TOKEN, TOKEN_IS_STATELESS)
                elif token.client_id != client.id:
                    # токен от другого клиента
                    raise OldApiError(INVALID_GRANT, TOKEN_NOT_BELONG_TO_CLIENT)
                elif token.device_id == DB_NULL:
                    # не позволяем отзывать токены, выданные без device_id
                    # TODO: может, разрешить также отзывать токены, не имеющие uid? Вряд ли.
                    raise OldApiError(UNSUPPORTED_TOKEN, TOKEN_WITHOUT_DEVICE_ID)
                else:
                    invalidate_single_token(token, client, request.env, reason='api_revoke_token')

        self.response_values.update(status='ok')


class IssueDeviceCodeView(BaseOldApiView):
    allowed_methods = ['POST']

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='issue_device_code',
        )

    @property
    def rate_limit_counters(self):
        return {
            COUNTER_IP: settings.DEVICE_CODE_RATE_LIMIT_IP,
        }

    def _counter_to_key_mapping(self, user_ip):
        return {
            COUNTER_IP: 'issue_device_code:%s:%s' % (COUNTER_IP, user_ip)
        }

    def is_client_secret_required(self, client):
        return False

    def check_scope_and_client_grants(self, scopes=None):
        assert_is_allowed(
            scope_list=scopes or self.client.scopes,
            grant_type=GRANT_TYPE_DEVICE_CODE,
            client_id=self.client.display_id,
            ip=self.request.env.consumer_ip,
            uid=None,
        )

    def make_verification_url(self, tld, client_id=None):
        verification_url = settings.CUSTOM_DEVICE_CODE_VERIFICATION_URLS_BY_TLD.get(
            tld,
            settings.DEFAULT_DEVICE_CODE_VERIFICATION_URL_TEMPLATE % {'tld': tld},
        )
        if client_id is not None:
            verification_url = update_url_params(verification_url, client_id=client_id)
        return verification_url

    def process_request(self, request):
        self.check_counters(self._counter_to_key_mapping(request.env.user_ip))
        self.get_and_bind_client()
        self.get_and_bind_device_info()
        requested_scopes = self.get_requested_scopes()
        self.check_scope_and_client_grants(requested_scopes)
        is_code_client_bound = self.get_optional_boolean_param('client_bound')

        token_request = create_request(
            client=self.client,
            scopes=requested_scopes,
            is_token_response=False,
            ttl=settings.DEVICE_CODE_TTL,
            device_id=self.device_id,
            device_name=self.device_name,
            make_code=True,
            code_strength=CodeStrength.BelowMedium if is_code_client_bound else CodeStrength.Medium,
            code_type=CodeType.ClientBound if is_code_client_bound else CodeType.Unique,
        )
        self.increase_counters(self._counter_to_key_mapping(request.env.user_ip))
        self.response_values.update(
            device_code=token_request.display_id,
            user_code=token_request.code,
            verification_url=self.make_verification_url(
                tld=request.TLD,
                client_id=self.client.display_id if is_code_client_bound else None,
            ),
            expires_in=token_request.ttl,
            interval=settings.DEVICE_CODE_POLL_INTERVAL,
        )
        self.statbox.log(
            status='ok',
            token_request_id=token_request.display_id,
            device_id=self.device_id,
            device_name=self.device_name,
        )
