# -*- coding: utf-8 -*-
import base64
import binascii
from collections import OrderedDict
import json
import re
from urllib.parse import urlencode

from django.utils.encoding import (
    smart_bytes,
    smart_text,
)
from passport.backend.oauth.api.api.old.errors import MalformedAuthorizationHeaderError
from passport.backend.oauth.core.common.constants import (
    BASIC_AUTH_REQUIRED,
    MALFORMED_AUTHORIZATION_HEADER,
)
import xmltodict


XML_INVALID_CHARS_REGEX = re.compile('[\x00-\x08\x0B\x0C\x0E-\x1F]')
XML_REPLACEMENT_FOR_INVALID_CHAR = '?'  # как и в ЧЯ

YANDEX_HOST_REGEX = re.compile(r'(.*)\.((yandex|yandex-team)\.[^\.]+(\.[^\.]+)?)')


def parse_auth_header(authorization):
    try:
        header_type, header_value = authorization.split()
    except ValueError:
        raise MalformedAuthorizationHeaderError(MALFORMED_AUTHORIZATION_HEADER)

    if header_type.lower() != 'basic':
        raise MalformedAuthorizationHeaderError(BASIC_AUTH_REQUIRED)

    try:
        client_id, client_secret = base64.b64decode(smart_bytes(header_value)).decode().split(':')
    except (TypeError, binascii.Error, UnicodeDecodeError, ValueError):
        raise MalformedAuthorizationHeaderError(MALFORMED_AUTHORIZATION_HEADER)

    return client_id, client_secret


def get_format(request):
    # Смотрим на параметр format в request
    # Если его нет, то считаем, что format=json (так написано в oauth 2.0 rev 5)
    format_ = request.GET.get('format', '') or request.POST.get('format', '')
    if not format_ or format_ not in ['json', 'form', 'xml']:
        format_ = 'json'
    return format_


def xml_value(value):
    if value is None:
        return ''
    elif isinstance(value, bool):
        return '1' if value else '0'
    else:
        return XML_INVALID_CHARS_REGEX.sub(XML_REPLACEMENT_FOR_INVALID_CHAR, smart_text(value))


def format_response(data, format_):
    if isinstance(data, dict):
        data = OrderedDict(sorted(data.items()))
    if not format_ or format_ == 'json':
        return 'application/json', json.dumps(data)
    elif format_ == 'form':
        return 'application/x-www-form-urlencoded', urlencode(data)
    elif format_ == 'xml':
        xml_data = xmltodict.unparse({'OAuth': data}, pretty=True) + '\n'
        return 'application/xml', xml_data
    raise ValueError('Unsupported format: %s' % format_)


def ensure_only_one_entry(items, params):
    """Проверяет, что некий параметр в GET/POST встречается только 1 раз. Если
    это не так, то поднимает исключение"""
    for (name, values) in params.lists():
        if name in items and len(values) > 1:
            raise ValueError(name)


def strip_to_yandex_domain(host):
    return YANDEX_HOST_REGEX.sub(r'\2', host)


def is_passport_domain(yandex_host):
    """
    ВНИМАНИЕ!!!
    Функция может возвращать True на некоторых неяндексовых хостах.
    Но если хост яндексовый (на него можно выставить авторизационные куки), то возвращаемый результат корректен.
    """
    # TODO: если начнём зависеть от паспортных сеттингов, то вместо этого заюзать PASSPORT_TLDS
    match = YANDEX_HOST_REGEX.match(yandex_host)
    if not match:
        return False
    return match.group(1).endswith('.passport') or match.group(1) == 'passport'
