# -*- coding: utf-8 -*-
import logging

from django.conf import settings
from django.http import (
    HttpResponse,
    HttpResponseForbidden,
    HttpResponseNotFound,
    HttpResponseServerError,
    JsonResponse,
)
from passport.backend.core.builders.blackbox.exceptions import BlackboxInvalidParamsError
from passport.backend.oauth.api.api.old.decorators import requires_grant
from passport.backend.oauth.api.api.old.error_descriptions import CLIENT_NOT_FOUND
from passport.backend.oauth.api.api.old.utils import (
    format_response,
    get_format,
)
from passport.backend.oauth.core.common.blackbox import (
    get_blackbox,
    get_revoke_time_from_bb_response,
    REVOKER_APP_PASSWORDS,
    REVOKER_TOKENS,
)
from passport.backend.oauth.core.common.constants import BAD_CLIENT
from passport.backend.oauth.core.db.client import (
    Client,
    list_clients_by_creator as get_clients_by_creator,
    PLATFORM_ANDROID,
    PLATFORM_IOS,
)
from passport.backend.oauth.core.db.token import list_clients_by_user as get_clients_by_user
from passport.backend.utils.time import datetime_to_integer_unixtime_nullable


log = logging.getLogger('api')


def api_error(request, error, status=400, description='', **kwargs):
    # NOTE Иду поперёк драфта, драфт (версии 07-10) не поддерживают xml и
    # ничего другого, только json.
    # Поддержка отдельных форматов основана на драфте версии 05.
    raw_response = {'error': error}
    if description:
        raw_response['error_description'] = description
    raw_response.update(kwargs)
    content_type, formatted_data = format_response(raw_response, get_format(request))
    response = HttpResponse(formatted_data, status=status)
    response['Cache-Control'] = 'no-store'  # описано в драфте
    response['Content-Type'] = content_type
    if status == 401:
        # NOTE Я не знаю, требуется ли указание realm или нет.
        # Надеюсь, что нет.
        response['WWW-Authenticate'] = 'Basic'
    return response


def api_success(request, data, status=200):
    content_type, formatted_data = format_response(data, get_format(request))
    response = HttpResponse(formatted_data, status=status)
    response['Cache-Control'] = 'no-store'  # описано в драфте
    response['Content-Type'] = content_type
    return response


def check_in(request):
    # Костыль: оставляем, пока АМ не оторвёт хождение в эту ручку
    return api_success(request, {'status': 'ok'})


@requires_grant('api.list_clients_by_user', default_consumer='intranet')
def list_clients_by_user(request, user_id):
    scope_keywords = request.GET.getlist('scope')
    if not scope_keywords:
        return api_error(request, 'Required parameter is missing: scope')

    try:
        bb_response = get_blackbox().userinfo(
            uid=user_id,
            ip=request.env.user_ip,
            dbfields=settings.BLACKBOX_DBFIELDS,
            attributes=settings.BLACKBOX_ATTRIBUTES,
            need_display_name=False,
        )
        uid = bb_response['uid']
    except BlackboxInvalidParamsError:
        uid = None

    if not uid:
        # нет юзера - нет приложений
        return api_success(request, [])

    tokens_revoked_at = get_revoke_time_from_bb_response(bb_response, REVOKER_TOKENS)
    app_passwords_revoked_at = get_revoke_time_from_bb_response(bb_response, REVOKER_APP_PASSWORDS)
    clients = [
        client
        for client in get_clients_by_user(
            uid=user_id,
            tokens_revoked_at=tokens_revoked_at,
            app_passwords_revoked_at=app_passwords_revoked_at,
        )
        if any(s for s in client.scopes if s.keyword in scope_keywords)
    ]
    display_ids = [client.display_id for client in clients]
    return api_success(request, display_ids)


@requires_grant('api.list_clients_by_creator', default_consumer='intranet')
def list_clients_by_creator(request, creator_uid):
    scope_keywords = request.GET.getlist('scope')
    if not scope_keywords:
        return api_error(request, 'Required parameter is missing: scope')
    clients = get_clients_by_creator(creator_uid)
    display_ids = set()
    for client in clients:
        if any(s for s in client.scopes if s.keyword in scope_keywords):
            display_ids.add(client.display_id)
    return api_success(request, list(display_ids))


@requires_grant('api.client_info', default_consumer='external')
def client_info(request, client_id):
    requested_locale = request.GET.get('locale') or request.LANGUAGE_CODE
    if requested_locale in settings.LANGUAGES:
        language = requested_locale
    else:
        language = settings.LANGUAGE_FALLBACK_MAPPING.get(
            requested_locale,
            settings.DEFAULT_LANGUAGE,
        )

    client = Client.by_display_id(display_id=client_id)
    if not client:
        return api_error(request, BAD_CLIENT, status=403, description=CLIENT_NOT_FOUND)
    client_as_dict = {
        'id': client.display_id,
        'name': client.get_title(language),
        'ctime': datetime_to_integer_unixtime_nullable(client.created),
        'mtime': datetime_to_integer_unixtime_nullable(client.modified),
        'icon': client.get_icon_url(),
        'homepage': client.homepage,
        'callback': client.default_redirect_uri,
        'description': client.get_description(language),
        'scope': [s.keyword for s in client.scopes],
        'localized_names': dict(
            (lang, client.get_title(lang))
            for lang in settings.LANGUAGES
        ),  # для совместимости
        'localized_scope': [s.get_title(language) for s in client.scopes],
        'is_yandex': client.is_yandex,
    }
    return JsonResponse(client_as_dict)


def ios_universal_links_associations(request, client_id):
    # https://developer.apple.com/library/content/documentation/General/Conceptual/AppSearch/UniversalLinks.html
    client = Client.by_display_id(display_id=client_id)
    if not client:
        return HttpResponseNotFound('Client not found')
    if PLATFORM_IOS not in client.platforms:
        return HttpResponseForbidden('Universal links disabled for client')

    details = []
    for app_id in client.ios_app_ids:
        app_id_without_bundle_prefix = app_id.split('.', 1)[-1]
        details.append({
            'appID': app_id,
            'paths': [
                '/auth/finish',
                '/magic-link/%s/finish' % app_id_without_bundle_prefix,
            ],
        })
    response_values = {
        'applinks': {
            'apps': [],
            'details': details,
        },
    }
    return JsonResponse(response_values, json_dumps_params=dict(indent=2))


def android_app_links_associations(request, client_id):
    # https://developer.android.com/training/app-links/index.html
    client = Client.by_display_id(display_id=client_id)
    if not client:
        return HttpResponseNotFound('Client not found')
    if PLATFORM_ANDROID not in client.platforms:
        return HttpResponseForbidden('App links disabled for client')
    response_values = [
        {
            'relation': [
                'delegate_permission/common.handle_all_urls',
            ],
            'target': {
                'namespace': 'android_app',
                'package_name': package_name,
                'sha256_cert_fingerprints': client.android_cert_fingerprints,
            },
        }
        for package_name in client.android_package_names
    ]
    return JsonResponse(response_values, json_dumps_params=dict(indent=2), safe=False)


def error_404(request, *args, **kwargs):
    return HttpResponseNotFound('Unknown url')


def error_500(request, *args, **kwargs):
    return HttpResponseServerError('Server error')
