# -*- coding: utf-8 -*-
from collections import OrderedDict
import json
import re

from django.conf import settings
from passport.backend.core.builders.blackbox import (
    get_alias,
    get_attribute,
)
from passport.backend.oauth.core.common.jwt import make_jwt
from passport.backend.oauth.core.common.utils import mask_string
import xmltodict


RE_EMAIL_SPLIT = re.compile(r'[_\-@\.]')
OAUTH_HEADER_PREFIXES = ['oauth', 'bearer']


class VerificationLevel(object):
    NO_VERIFIED = 10
    LOW_VERIFIED = 20
    HIGH_VERIFIED = 30


def mask_token(token):
    if not token:
        return token
    return mask_string(token, show_first_n=len(token) // 2)


def sort_emails(emails):
    return sorted(emails, key=lambda e: RE_EMAIL_SPLIT.split(e))


def parse_auth_header(header):
    if not header:
        return
    for prefix in OAUTH_HEADER_PREFIXES:
        if header.lower().startswith(prefix):
            return header[len(prefix):].strip()


def parse_bool_str(value):
    return value.lower() in ('yes', 'true', '1')


def _phone_to_response(phone, need_confirm_time=False):
    result = OrderedDict()
    result['id'] = phone['id']
    result['number'] = phone['attributes']['number']
    if need_confirm_time:
        result['confirmed'] = (
            phone['attributes'].get('admitted') or
            phone['attributes'].get('confirmed') or
            phone['attributes'].get('bound')
        )
    return result


def _get_payment_auth_scopes(all_scopes):
    result = []
    for service in settings.SERVICES_REQUIRING_PAYMENT_AUTH:
        prefix = service + ':'
        for scope in all_scopes:
            if scope.startswith(prefix):
                result.append(scope)
    return result


def parse_blackbox_response(response):
    retval = {
        'id': str(response['uid']),  # отдаём как строку для консистентности с ЧЯ и OAuth
        'login': response['subscriptions'][8]['login'],
        'client_id': response['oauth']['client_id'],
        'normalized_login': get_attribute(response, 'account.normalized_login'),
        'display_name': response['display_name']['name'],
        'default_avatar_id': response['display_name'].get('avatar', {}).get('default'),
        'is_avatar_empty': response['display_name'].get('avatar', {}).get('empty', True),
        'first_name': get_attribute(response, 'person.firstname') or '',
        'last_name': get_attribute(response, 'person.lastname') or '',
        'birthday': get_attribute(response, 'person.birthday'),
        'sex': {'m': 'male', 'f': 'female'}.get(get_attribute(response, 'person.gender')),
        'old_social_login': get_alias(response, 'social'),
        'has_yaru_sid': bool(get_attribute(response, 'subscription.37')),
        'has_plus': bool(get_attribute(response, 'account.have_plus')),
        'payment_auth_info': {
            'scopes': _get_payment_auth_scopes(response['oauth']['scope']),
            'context_id': response['oauth'].get('payment_auth_context_id'),
            'scope_addendum': response['oauth'].get('payment_auth_scope_addendum'),
            'meta': response['oauth']['meta'],
            'client_name': response['oauth']['client_name'],
        },
        'verification_level': _get_account_verification_level(response),
    }
    retval['real_name'] = ('%(first_name)s %(last_name)s' % retval).strip()

    all_emails = response.get('address-list', [])
    # Отдаем только провалидированные адреса
    # Кроме того, пока отдаём только дефолтный
    emails = [e for e in all_emails if e['validated'] and e['default']]
    default_emails = [e['address'] for e in emails if e['default']]  # список длины от 0 до 1
    retval.update(
        emails=sort_emails(e['address'] for e in emails),
        default_email=default_emails[0] if default_emails else '',
    )

    all_phones = response['phones'].values()
    phones = [p for p in all_phones if p['attributes'].get('bound')]
    default_phones = [p for p in phones if p['attributes'].get('secured')]  # список длины от 0 до 1; возможно, лучше использовать is_secured
    retval.update(
        phones=[
            _phone_to_response(phone, need_confirm_time=True)
            for phone in phones
        ],
        default_phone=_phone_to_response(default_phones[0]) if default_phones else None,
    )

    return retval


def strip_according_to_scopes(response, scopes, with_openid_identity=False, with_payment_auth_info=False):
    retval = {
        'id': response['id'],
        'login': response['login'],
        'client_id': response['client_id'],
    }

    if with_openid_identity:
        openid_identities = set()
        openid_identities.add('http://openid.yandex.ru/%s/' % response['normalized_login'])
        if response['has_yaru_sid']:
            openid_identities.add('http://%s.ya.ru/' % response['normalized_login'])
        if response['old_social_login']:
            openid_identities.add('http://openid.yandex.ru/%s/' % response['old_social_login'])

        retval['openid_identities'] = sorted(openid_identities)

    if with_payment_auth_info:
        if _get_payment_auth_scopes(scopes):
            retval['payment_auth_info'] = response['payment_auth_info']
        else:
            retval['payment_auth_info'] = None

    if 'login:info' in scopes:
        retval['display_name'] = response['display_name']
        retval['real_name'] = response['real_name']
        retval['first_name'] = response['first_name']
        retval['last_name'] = response['last_name']
        retval['sex'] = response['sex']
    if 'login:email' in scopes:
        retval['default_email'] = response['default_email']
        retval['emails'] = response['emails']
    if 'login:birthday' in scopes:
        retval['birthday'] = response['birthday']
    if 'login:avatar' in scopes:
        retval['default_avatar_id'] = response['default_avatar_id']
        retval['is_avatar_empty'] = response['is_avatar_empty']
    if 'login:full_phones' in scopes:
        retval['phones'] = response['phones']
        retval['default_phone'] = response['default_phone']
    if 'login:default_phone' in scopes:
        retval['default_phone'] = response['default_phone']
    if 'login:plus_subscriptions' in scopes:
        retval['has_plus'] = response['has_plus']
    if 'login:verification_level' in scopes:
        retval['verification_level'] = response['verification_level']

    return retval


def dict_to_xml(dictionary):
    list_to_item_name_mapping = {
        'emails': 'address',
        'openid_identities': 'identity',
        'phones': 'phone',
    }
    result = OrderedDict()
    for key, value in sorted(dictionary.items()):
        if key == 'payment_auth_info':
            # эти данные нужны только в JSON
            continue
        elif isinstance(value, (list, tuple)):
            item_name = list_to_item_name_mapping[key]
            result[key] = {item_name: value}
        else:
            result[key] = value
    return xmltodict.unparse({'user': result})


def dict_to_json(dictionary):
    return json.dumps(dictionary)


def dict_to_jwt(dictionary, jwt_secret, expires, issuer):
    payload = {
        'uid': int(dictionary['id']),
        'login': dictionary['login'],
    }

    field_to_attr_mapping = {
        'psuid': 'psuid',
        'real_name': 'name',
        'default_email': 'email',
        'birthday': 'birthday',
        'sex': 'gender',
        'display_name': 'display_name',
        'default_avatar_id': 'avatar_id',
        'default_phone': 'phone'
    }
    for field, attr in field_to_attr_mapping.items():
        if field in dictionary:
            payload[attr] = dictionary[field]

    return make_jwt(
        secret=jwt_secret,
        expires=expires,
        issuer=issuer,
        custom_fields=payload,
    )


def _get_account_verification_level(bb_response):
    karma = bb_response.get('karma')
    # https://wiki.yandex-team.ru/passport/karma/
    # префиксы, при которых аккаунт не является спамером
    if karma is not None and karma // 1000 not in [0, 2, 4, 6]:
        return VerificationLevel.NO_VERIFIED
    confirmed_phones = [p for p in bb_response['phones'].values() if p['attributes'].get('confirmed')]
    secure_phone = [p for p in confirmed_phones if p['attributes'].get('secured')]
    if secure_phone:
        return VerificationLevel.HIGH_VERIFIED
    elif confirmed_phones:
        return VerificationLevel.LOW_VERIFIED
    return VerificationLevel.NO_VERIFIED
