# -*- coding: utf-8 -*-
import logging

from django.conf import settings
from django.http import (
    HttpResponse,
    HttpResponseForbidden,
)
from passport.backend.core.builders.blackbox.constants import BLACKBOX_OAUTH_VALID_STATUS
from passport.backend.oauth.api.api.yandex_login.utils import (
    dict_to_json,
    dict_to_jwt,
    dict_to_xml,
    mask_token,
    parse_auth_header,
    parse_blackbox_response,
    parse_bool_str,
    strip_according_to_scopes,
)
from passport.backend.oauth.core.common.blackbox import get_blackbox
from passport.backend.oauth.core.common.utils import parse_datetime_to_unixtime
from passport.backend.oauth.core.db.client import Client
from passport.backend.oauth.core.db.limits import (
    check_grants,
    GrantsMissingError,
)
from passport.backend.oauth.core.db.psuid import make_psuid
from passport.backend.oauth.core.logs.statbox import to_statbox


log = logging.getLogger('yandex_login')

PAYMENT_AUTH_INFO_GRANT = 'api.get_payment_auth_info'


def auth_error(reason):
    error = HttpResponse(status=401)
    if reason:
        error['WWW-Authenticate'] = "OAuth realm='login' error='%s'" % reason
    return error


def xml_response(data):
    return HttpResponse(data, content_type='application/xml')


def json_response(data):
    return HttpResponse(data, content_type='application/json')


def plaintext_response(data):
    return HttpResponse(data, content_type='text/plain')


def info(request):
    format_ = request.REQUEST.get('format', 'json')
    if format_ not in ['json', 'xml', 'jwt']:
        return HttpResponse('format=<json|xml|jwt>', status=400)

    with_openid_identity = parse_bool_str(request.REQUEST.get('with_openid_identity', 'no'))

    with_payment_auth_info = parse_bool_str(request.REQUEST.get('with_payment_auth_info', 'no'))
    if with_payment_auth_info:
        try:
            check_grants(
                grants=[PAYMENT_AUTH_INFO_GRANT],
                consumer=settings.PAYMENT_AUTH_INFO_FAKE_CONSUMER,
                ip=request.env.consumer_ip,
                service_ticket=None,
            )
        except GrantsMissingError:
            log.warning('payment_auth_info was requested, but ip=%s is not permitted', request.env.consumer_ip)
            return HttpResponseForbidden('no_grants')

    oauth_token = request.REQUEST.get('oauth_token')
    if not oauth_token:
        auth_header = request.env.authorization
        oauth_token = parse_auth_header(auth_header)
        log.debug('Got OAuth token: %s', mask_token(oauth_token))

    jwt_secret = request.REQUEST.get('jwt_secret')

    if not oauth_token:
        return auth_error('')

    blackbox_info = get_blackbox().oauth(
        oauth_token=oauth_token,
        ip=request.env.user_ip,
        dbfields=[
            'subscription.suid.8',
            'subscription.login.8',
        ],
        attributes=[
            'person.firstname',
            'person.lastname',
            'person.gender',
            'person.birthday',
            'account.normalized_login',
            'subscription.37',  # подписка на Ярушку
            'account.have_plus',
        ],
        emails=True,  # вообще хватило бы getdefault вместо getall
        need_display_name=True,
        need_public_name=False,
        need_aliases=True,
        phones='bound',
        phone_attributes=[
            'number',
            'bound',
            'admitted',
            'confirmed',
            'is_default',
            'secured',
        ],
    )

    if blackbox_info['status'] != BLACKBOX_OAUTH_VALID_STATUS:
        log.debug('OAuth token invalid: %s', blackbox_info['error'])
        return auth_error(blackbox_info['error'])

    if not blackbox_info.get('uid'):
        log.debug('Got depersonalized OAuth token')
        return auth_error('unsupported_token_type')

    response_data = parse_blackbox_response(blackbox_info)
    response_data = strip_according_to_scopes(
        response=response_data,
        scopes=set(blackbox_info['oauth']['scope']),
        with_openid_identity=with_openid_identity,
        with_payment_auth_info=with_payment_auth_info,
    )

    client_id = blackbox_info['oauth']['client_id']
    # теоретически тут приложение можем не найти (случай гонок), но этим кейсом можно пренебречь
    client = Client.by_display_id(client_id)
    if client:
        response_data.update(
            psuid=make_psuid(
                uid=int(blackbox_info['uid']),
                client=client,
            ),
        )

    if jwt_secret is None and format_ == 'jwt':
        # Если JWT-секрет не передали, подпишем токен секретом приложения
        jwt_secret = client.secret if client else ''

    log.debug('Sending successful response for uid=%s, client_id=%s', blackbox_info['uid'], client_id)
    to_statbox(
        mode='get_info_by_token',
        status='ok',
        uid=blackbox_info['uid'],
        client_id=client_id,
        token_id=blackbox_info['oauth']['token_id'],
        format=format_,
        with_payment_auth_info=with_payment_auth_info,
    )

    if format_ == 'json':
        return json_response(dict_to_json(response_data))
    elif format_ == 'jwt':
        return plaintext_response(dict_to_jwt(
            response_data,
            jwt_secret=jwt_secret,
            expires=parse_datetime_to_unixtime(blackbox_info['oauth']['expire_time']),
            issuer=request.get_host(),
        ))
    else:
        return xml_response(dict_to_xml(response_data))
