# -*- coding: utf-8 -*-
import logging
import os

import yenv


def build_config_postfix(env_name, env_type):
    if env_type == 'rc':
        # для RC отдаём конфиги из продакшна
        actual_env_type = 'production'
    elif env_type == 'development':
        # для дева - из тестинга
        actual_env_type = 'testing'
    else:
        actual_env_type = env_type
    return actual_env_type if env_name in ['other', 'localhost'] else '%s.%s' % (env_name, actual_env_type)


def make_link_to_config(config_dir, config_name, config_extension='json', result_name=None):
    """
    Создаёт для каждой группы конфигов по две ссылки: на версии для текущих
    физического (local) и логического (local_admin) окружений
    """
    for link_postfix, file_postfix in (
        ('local', build_config_postfix(yenv.name, yenv.type)),
        ('local_admin', build_config_postfix('localhost', yenv.type)),  # админки для ятима пока нет
    ):
        source_file = '/usr/lib/yandex/%s/%s.%s.%s' % (config_dir, config_name, file_postfix, config_extension)
        logging.info('Using %s as %s (yenv.name = %s, yenv.type = %s)', source_file, link_postfix, yenv.name, yenv.type)

        dest_file = '/usr/lib/yandex/oauth-configs/%s.%s.%s' % (result_name or config_name, link_postfix, config_extension)
        if os.path.islink(dest_file):
            os.unlink(dest_file)
        elif os.path.isfile(dest_file):
            os.remove(dest_file)

        os.symlink(source_file, dest_file)


def create_links():
    logging.info('Creating links to JSON configs...')
    for config_dir, config_name, kwargs in [
        ('oauth-configs', 'scopes', {}),
        ('oauth-configs', 'acl_grants', {}),
        ('oauth-configs', 'client_lists', {'config_extension': 'yaml'}),
        ('oauth-configs', 'token_params', {'config_extension': 'yaml'}),
        ('oauth-grants', 'scope_grants', {}),
        ('tvm-api-grants', 'consumer_grants', {'result_name': 'tvm_api_grants'}),
    ]:
        make_link_to_config(config_dir=config_dir, config_name=config_name, **kwargs)
    logging.info('Links created.')


def configure_create_links_command(commander):
    commander.add_command(
        'create-links',
        create_links,
    )
