# -*- coding: utf-8 -*-
import json
import logging
import os

import requests


TANKER_URL = 'https://tanker-api.yandex-team.ru'

TANKER_PROJECT = 'oauth'
TANKER_SCOPE_KEYSET = 'scope'
TANKER_SCOPE_SHORT_KEYSET = 'scope_short'
TANKER_SERVICE_KEYSET = 'service'
TANKER_LANGS = ('ru', 'en', 'tr', 'uk')


def fetch():
    logging.info('Fetching tanker...')
    result = []
    for keyset_id in (TANKER_SCOPE_KEYSET, TANKER_SCOPE_SHORT_KEYSET, TANKER_SERVICE_KEYSET):
        resp = requests.get(
            TANKER_URL + '/keysets/tjson/',
            params={
                'keyset-id': keyset_id,
                'project-id': TANKER_PROJECT,
                'status': 'unapproved',
            },
            verify='/etc/ssl/certs/ca-certificates.crt',
        )
        json_data = json.loads(resp.content)
        keyset = {}
        for msgid, data in json_data['keysets'][keyset_id]['keys'].items():
            for lang in TANKER_LANGS:
                translation = data['translations'][lang].get('form', '')
                if lang not in keyset:
                    keyset[lang] = {}
                keyset[lang][msgid] = translation
        result.append(keyset)
    return result


def update_translations(base_dir):
    scopes, scopes_short, services = fetch()
    for data, filename in (
        (scopes, 'scope_translations.json'),
        (scopes_short, 'scope_short_translations.json'),
        (services, 'service_translations.json'),
    ):
        file_path = os.path.join(base_dir, filename)
        with open(file_path, 'w') as f:
            json.dump(data, f, indent=4, ensure_ascii=False, sort_keys=True, separators=(',', ': '))
            f.write('\n')
    logging.info('Done.')


def configure_update_translations_command(commander):
    commander.add_command(
        'update-translations',
        update_translations,
    ).add_argument(
        '-d', '--dir',
        dest='base_dir',
        type=str,
        default='../',
    )
