# -*- coding: utf-8 -*-
from collections import namedtuple
import json
import logging
import os
import sys

import yaml


ENV_NAMES = (
    'testing',
    'production',
    'intranet.testing',
    'intranet.production',
    'stress.stress',
)

LANGUAGES = ('ru', 'en', 'tr', 'uk')
LANGUAGES_YATEAM = ('ru', )

COLOR_GREEN = '\033[32m'
COLOR_YELLOW = '\033[33m'
COLOR_RED = '\033[91m'
COLOR_END = '\033[0m'

MAX_LIST_LENGTH = 5
PREFERRED_LIST_LENGTH = 3


Field = namedtuple('Field', 'type required')

SCOPE_FIELDS = {
    'keyword': Field(type=str, required=True),
    'default_title': Field(type=str, required=False),
    'ttl': Field(type=int, required=False),
    'is_ttl_refreshable': Field(type=bool, required=False),
    'requires_approval': Field(type=bool, required=False),
    'is_hidden': Field(type=bool, required=False),
    'visible_for_uids': Field(type=list, required=False),
    'visible_for_consumers': Field(type=list, required=False),
    'visible_for': Field(type=list, required=False),
    'has_xtoken_grant': Field(type=bool, required=False),
    'allowed_for_turboapps': Field(type=bool, required=False),
    'tags': Field(type=list, required=False),
}

STATELESS_TOKEN_RULE_FIELDS = {
    'client_id': Field(type=str, required=False),
    'app_id': Field(type=str, required=False),
    'app_platform': Field(type=str, required=False),
    'min_app_version': Field(type=str, required=False),
    'denominator': Field(type=int, required=False),
}

# для этих сервисов не требуем наличия переводов
TEST_SERVICES = ('test',)

# сервисы с такими именами завести нельзя
RESERVED_SERVICE_NAMES = (
    'bb',  # так называются фейковые скоупы в TVM User-тикетах
    'grant_type',  # так называются фейковые потребители в Грантушке
)


def json_duplicate_keys_checker(keys_values):
    result = {}
    for key, value in keys_values:
        if key in result:
            raise ValueError('Duplicate key %r' % key)
        else:
            result[key] = value
    return result


class ValidationError(AssertionError):
    """Конфиги не прошли проверку"""


class ConfigValidator(object):
    def __init__(self, config_dir, verbose_output=False, colorless_output=False):
        self.config_dir = config_dir
        self.scope_configs = {}
        self.acl_configs = {}
        self.token_params_configs = {}
        self.client_lists_configs = {}
        self.scope_translations = None
        self.scope_short_translations = None
        self.service_translations = None
        self.login_to_uid_mapping = None
        self.current_env_name = None
        self.is_verbose = verbose_output
        self.is_colorless = colorless_output
        self.infos = []
        self.warnings = []
        self.errors = []

    def error(self, message):
        self.errors.append('ERROR (%s): %s' % (self.current_env_name or 'global', message))

    def warning(self, message):
        self.warnings.append('WARNING (%s): %s' % (self.current_env_name or 'global', message))

    def info(self, message):
        self.infos.append('INFO (%s): %s' % (self.current_env_name or 'global', message))

    def make_warning(self, message):
        if self.is_strict:
            self.warning(message)
        else:
            self.info(message)

    def format_list(self, items):
        if self.is_verbose or len(items) <= MAX_LIST_LENGTH:
            return ', '.join(items)
        else:
            return '%s,.. (%d more)' % (', '.join(items[:PREFERRED_LIST_LENGTH]), len(items) - PREFERRED_LIST_LENGTH)

    def _read_json_config(self, filename, **json_load_kwargs):
        with open(
            os.path.join(self.config_dir, filename),
        ) as f:
            return json.load(f, **json_load_kwargs)

    def _read_yaml_config(self, filename):
        with open(
            os.path.join(self.config_dir, filename),
        ) as f:
            return yaml.safe_load(f)

    def load_configs(self):
        for env_name in ENV_NAMES:
            self.scope_configs[env_name] = self._read_json_config(
                'scopes.%s.json' % env_name,
                object_pairs_hook=json_duplicate_keys_checker,
            )
            self.acl_configs[env_name] = self._read_json_config('acl_grants.%s.json' % env_name)
            self.token_params_configs[env_name] = self._read_yaml_config('token_params.%s.yaml' % env_name)
            self.client_lists_configs[env_name] = self._read_yaml_config('client_lists.%s.yaml' % env_name)

        self.scope_translations = self._read_json_config('scope_translations.json')
        self.scope_short_translations = self._read_json_config('scope_short_translations.json')
        self.service_translations = self._read_json_config('service_translations.json')
        self.login_to_uid_mapping = self._read_json_config('login_to_uid_mapping.json')

    def print_result(self):
        for info in self.infos:
            logging.info(info)
        for warning in self.warnings:
            logging.warning(warning if self.is_colorless else '%s%s%s' % (COLOR_YELLOW, warning, COLOR_END))
        for error in self.errors:
            logging.error(error if self.is_colorless else '%s%s%s' % (COLOR_RED, error, COLOR_END))

        if self.errors:
            raise ValidationError(self.errors)

        if not self.warnings:
            logging.info('OK.' if self.is_colorless else '%sOK.%s' % (COLOR_GREEN, COLOR_END))

    @property
    def scopes(self):
        return self.scope_configs[self.current_env_name]

    @property
    def token_params(self):
        return self.token_params_configs[self.current_env_name]

    @property
    def client_lists(self):
        return self.client_lists_configs[self.current_env_name]

    @property
    def is_strict(self):
        return 'production' in self.current_env_name

    def validate(self):
        # Проверяем json на валидность
        try:
            self.load_configs()
        except ValueError as e:
            self.error('Malformed config file: %s' % e)
            self.print_result()

        # Проверяем конфиги скоупов во всех окружениях
        for self.current_env_name in ENV_NAMES:
            # проверим наличие и тип всех полей
            unknown_fields = []
            bad_field_values = []
            for scope in self.scopes.values():
                scope_keyword = scope.get('keyword', scope)
                service, _ = scope_keyword.split(':', 1)

                if service in RESERVED_SERVICE_NAMES:
                    self.error('Bad scope keyword (forbidden service name): %s' % scope_keyword)

                for field_name in scope:
                    if field_name not in SCOPE_FIELDS:
                        unknown_fields.append(field_name)
                        continue
                    field = SCOPE_FIELDS[field_name]
                    value = scope.get(field_name)
                    if value is None and not field.required:
                        continue
                    if not isinstance(value, field.type):
                        bad_field_values.append('%s -> %s (%s)' % (scope_keyword, field_name, repr(value)))

                for field_name, field in SCOPE_FIELDS.items():
                    if not scope.get(field_name) and field.required:
                        bad_field_values.append('%s -> %s (empty)' % (scope_keyword, field_name))

                # отдельно проверим visible_for_uids
                value = scope.get('visible_for_uids')
                if not all(isinstance(x, int) for x in value or []):
                    bad_field_values.append('%s -> %s (%s)' % (scope_keyword, 'visible_for_uids', repr(value)))
                # отдельно проверим visible_for
                value = scope.get('visible_for')
                bad_logins = [
                    x for x in value or []
                    if not isinstance(x, int) and x not in self.login_to_uid_mapping
                ]
                if bad_logins:
                    bad_field_values.append('%s -> %s (bad logins: %s)' % (
                        scope_keyword,
                        'visible_for',
                        bad_logins,
                    ))
                # visible_for / visible_for_uids без is_hidden не учитываются => ошибка, если так
                if scope.get('visible_for_uids') or scope.get('visible_for') and not scope.get('is_hidden'):
                    bad_field_values.append('%s -> %s (should be true if visible_for* is set' % (
                        scope_keyword,
                        'is_hidden',
                    ))
                # В ятиме не должно быть подновляемых скоупов
                if self.current_env_name.startswith('intranet') and not scope_keyword.startswith('deleted:'):
                    field = 'is_ttl_refreshable'
                    if scope.get('is_ttl_refreshable'):
                        bad_field_values.append('%s -> %s (unavailable at YT)' % (
                            scope_keyword,
                            field,
                        ))

            if bad_field_values:
                self.error('Bad scope fields: %s' % self.format_list(bad_field_values))
            if unknown_fields:
                self.warning('Unknown scope fields: %s' % self.format_list(unknown_fields))

            keywords = [s['keyword'] for s in self.scopes.values() if not s['keyword'].startswith('deleted:')]

            # проверим уникальность скоупов
            not_unique_keywords = [keyword for keyword in keywords if keywords.count(keyword) > 1]
            if not_unique_keywords:
                self.error('Not unique scope keywords: %s' % self.format_list(not_unique_keywords))

            scopes = set(keywords)
            scopes_for_turboapps = set([
                s['keyword']
                for s in self.scopes.values()
                if s.get('allowed_for_turboapps') and not s['keyword'].startswith('deleted:')
            ])
            services = set(scope.split(':', 1)[0] for scope in scopes)

            languages = LANGUAGES_YATEAM if self.current_env_name.startswith('intranet.') else LANGUAGES

            for scope_list, translations, alias_for_log in (
                (scopes, self.scope_translations, 'Translations'),
                (scopes_for_turboapps, self.scope_short_translations, 'Short translations'),
            ):
                # проверим наличие переводов для скоупов
                missing_translations = [
                    scope for scope in scope_list
                    if (
                        scope not in translations['ru'] and
                        not any(scope.startswith(service + ':') for service in TEST_SERVICES)
                    )
                ]
                if missing_translations:
                    self.error(
                        '%s missing for scopes: %s' % (alias_for_log, self.format_list(missing_translations)),
                    )

                # проверим непустоту всех переводов скоупов
                for lang in languages:
                    empty_translations = [
                        scope for scope in scope_list
                        if (
                            scope not in missing_translations and
                            not translations[lang].get(scope) and
                            not any(scope.startswith(service + ':') for service in TEST_SERVICES)
                        )
                    ]
                    if empty_translations:
                        self.make_warning(
                            '%s (%s) empty for scopes: %s' % (alias_for_log, lang, self.format_list(empty_translations)),
                        )

            # проверим наличие переводов для сервисов
            missing_translations = [
                service_ for service_ in services
                if (
                    service_ not in self.service_translations['ru'] and
                    service_ not in TEST_SERVICES
                )
            ]
            if missing_translations:
                self.error('Translations missing for services: %s' % self.format_list(missing_translations))

            # проверим непустоту всех переводов сервисов
            for lang in languages:
                empty_translations = [
                    service_ for service_ in services
                    if (
                        service_ not in missing_translations and
                        not self.service_translations[lang].get(service_) and
                        service_ not in TEST_SERVICES
                    )
                ]
                if empty_translations:
                    self.make_warning(
                        'Translations (%s) empty for services: %s' % (lang, self.format_list(empty_translations)),
                    )

            # Проверяем конфиги ACL
            for login_or_uid in self.acl_configs[self.current_env_name]:
                if login_or_uid not in self.login_to_uid_mapping and not login_or_uid.isdigit():
                    self.error('Unknown login: "%s"' % login_or_uid)

        # Проверяем отсутствие неиспользуемых переводов
        self.current_env_name = 'common'
        all_scopes = set()
        for scope_config in self.scope_configs.values():
            all_scopes.update(set(s['keyword'] for s in scope_config.values()))
        all_services = set(scope.split(':', 1)[0] for scope in all_scopes)

        # проверим переводы скоупов
        unused_translations = [key for key in self.scope_translations['ru'] if key not in all_scopes]
        if unused_translations:
            self.info('Unused scope translations: %s' % (self.format_list(unused_translations)))

        # проверим переводы сервисов
        unused_translations = [key for key in self.service_translations['ru'] if key not in all_services]
        if unused_translations:
            self.info('Unused service translations: %s' % (self.format_list(unused_translations)))

        # Проверяем отсутствие неиспользуемых логинов в маппинге
        all_logins_and_uids = set()
        for scope_config in self.scope_configs.values():
            for s in scope_config.values():
                all_logins_and_uids.update(set(s.get('visible_for', [])))
        for acl_config in self.acl_configs.values():
            all_logins_and_uids.update(acl_config.keys())
        unused_logins = [
            '"%s"' % login
            for login in self.login_to_uid_mapping
            if login not in all_logins_and_uids and not login.startswith('__comment')
        ]
        if unused_logins:
            self.info('Unused logins: %s' % (self.format_list(unused_logins)))

        # Проверяем настройки для токенов во всех окружениях
        for self.current_env_name in ENV_NAMES:
            # проверим наличие и тип всех полей
            unknown_fields = []
            bad_field_values = []
            for rule_name, rule in self.token_params['force_stateless'].items():
                for field_name in rule:
                    if field_name not in STATELESS_TOKEN_RULE_FIELDS:
                        unknown_fields.append(field_name)
                        continue
                    field = STATELESS_TOKEN_RULE_FIELDS[field_name]
                    value = rule.get(field_name)
                    if value is None and not field.required:
                        continue
                    if not isinstance(value, field.type):
                        bad_field_values.append('%s -> %s (%s)' % (rule_name, field_name, repr(value)))

                for field_name, field in STATELESS_TOKEN_RULE_FIELDS.items():
                    if not rule.get(field_name) and field.required:
                        bad_field_values.append('%s -> %s (empty)' % (rule_name, field_name))

                # должен присутствовать либо client_id, либо app_id
                if not rule.get('client_id') and not rule.get('app_id'):
                    bad_field_values.append('%s (neither `client_id` not `app_id` is specified)' % rule_name)

                # отдельно проверим client_id
                value = rule.get('client_id')
                if value and len(value) != 32:
                    bad_field_values.append('%s -> %s (malformed: %s)' % (
                        rule_name,
                        'client_id',
                        repr(value),
                    ))
                # отдельно проверим app_platform
                value = rule.get('app_platform')
                if value not in [None, 'android', 'ios']:
                    bad_field_values.append('%s -> %s (invalid choice: %s)' % (
                        rule_name,
                        'app_platform',
                        repr(value),
                    ))

            if bad_field_values:
                self.error('Bad token params: %s' % self.format_list(bad_field_values))
            if unknown_fields:
                self.info('Unknown token params fields: %s' % self.format_list(unknown_fields))

        # Проверяем вайтлисты приложений во всех окружениях
        for self.current_env_name in ENV_NAMES:
            if not isinstance(self.client_lists.get('whitelist_for_scope'), dict):
                self.error('Bad whitelisted clients config: `whitelist_for_scope` must be dict')
            else:
                for _, client_ids in self.client_lists['whitelist_for_scope'].items():
                    for client_id in client_ids:
                        if not isinstance(client_id, str):
                            self.error('Bad client_id: %r (must be str)' % client_id)

        # Ура! Всё проверили.
        self.print_result()


def validate(base_dir, verbose, colorless):
    validator = ConfigValidator(
        config_dir=base_dir,
        verbose_output=verbose,
        colorless_output=colorless,
    )
    try:
        validator.validate()
    except ValidationError:
        sys.exit(1)


def configure_validate_command(commander):
    commander.add_command(
        'validate',
        validate,
    ).add_argument(
        '-d', '--dir',
        dest='base_dir',
        type=str,
        default='../',
    ).add_argument(
        '-v', '--verbose',
        action='store_true',
        dest='verbose',
    ).add_argument(
        '--colorless',
        action='store_true',
        dest='colorless',
    )
