# -*- coding: utf-8 -*-
import logging

from django.conf import settings
from django.http import (
    HttpResponse,
    JsonResponse,
)
from django.utils.functional import cached_property
from django.views.generic.base import View
from passport.backend.core.builders.abc import ABCTemporaryError
from passport.backend.core.builders.blackbox import (
    BaseBlackboxError,
    BLACKBOX_OAUTH_VALID_STATUS,
    BLACKBOX_SESSIONID_NEED_RESET_STATUS,
    BLACKBOX_SESSIONID_VALID_STATUS,
    BLACKBOX_SESSIONID_WRONG_GUARD_STATUS,
    BlackboxInvalidParamsError as BlackboxInvalidParams,
    get_attribute,
)
from passport.backend.core.builders.money_api import BaseMoneyApiError
from passport.backend.core.logging_utils.helpers import mask_sessionid
from passport.backend.core.tvm import get_tvm_credentials_manager
from passport.backend.oauth.core.api.errors import (
    AuthorizationHeaderError,
    BackendFailedError,
    BaseApiError,
    BlackboxInvalidParamsError,
    ExceptionUnhandledError,
    GrantsMissingError,
    InvalidFormError,
    MethodNotAllowedError,
    OAuthTokenValidationError,
    SessguardInvalidError,
    SessionidEmptyError,
    SessionidInvalidError,
    TVMTicketInvalidError,
    UidNotInSessionError,
    UserNotFoundError,
    UserTicketEmptyError,
    UserTicketValidationError,
)
from passport.backend.oauth.core.api.forms import ConsumerForm
from passport.backend.oauth.core.common.avatars import AvatarsMdsApiTemporaryError
from passport.backend.oauth.core.common.blackbox import get_blackbox
from passport.backend.oauth.core.common.error_logs import (
    log_error,
    log_warning,
)
from passport.backend.oauth.core.db.eav import BaseDBError
from passport.backend.oauth.core.db.limits import (
    check_grants,
    GrantsMissingError as GrantsMissing,
    TVMTicketInvalidError as TVMTicketInvalid,
)
from passport.backend.oauth.core.db.scope import Scope
from passport.backend.oauth.core.logs.common import mask_sensitive_fields
from passport.backend.oauth.core.logs.statbox import to_statbox
from ticket_parser2.exceptions import TicketParsingException


log = logging.getLogger('api')


OAUTH_HEADER_PREFIX = 'oauth '


def log_request(request):
    get_values = mask_sensitive_fields(dict(request.GET))
    post_values = mask_sensitive_fields(dict(request.POST))
    log.debug(
        'Got request: %s %s (GET: %s, POST: %s)' % (
            request.method,
            request.path,
            get_values,
            post_values,
        ),
    )


def log_response(response_values, status=200):
    if isinstance(response_values, dict):
        response_values = mask_sensitive_fields(response_values)
    log.debug('Sent response: %d %s' % (status, response_values))


class BaseApiView(View):
    base_form = None
    allowed_methods = ['GET', 'POST']
    required_grants = None
    ensure_ascii_response = True
    temporary_exceptions_list = [
        BaseDBError,
        BaseBlackboxError,
        AvatarsMdsApiTemporaryError,
        ABCTemporaryError,
        BaseMoneyApiError,
    ]

    def __init__(self):
        super(BaseApiView, self).__init__()
        self.consumer = None
        self.form_values = {}
        self.response_values = {}
        self.file_response = {}

    @property
    def uid(self):
        return self.form_values.get('uid')

    @cached_property
    def oauth_token(self):
        auth_header = self.request.env.authorization
        if not auth_header or not auth_header.lower().startswith(OAUTH_HEADER_PREFIX):
            raise AuthorizationHeaderError()
        return auth_header[len(OAUTH_HEADER_PREFIX):].strip()

    def process_form(self, form_class, form_args=None, data=None, files=None):
        form = form_class(data=data or {}, files=files or {}, **(form_args or {}))
        if form.is_valid():
            return form.cleaned_data
        else:
            raise InvalidFormError(errors=form.errors)

    def process_root_form(self, data):
        values = self.process_form(ConsumerForm, data=data)
        self.consumer = values['consumer']

    def process_base_form(self, data, files=None):
        self.form_values = self.process_form(self.base_form, data=data, files=files)

    def make_file_response(self, data, filename, content_type=None):
        response = HttpResponse(
            content=data,
            content_type=content_type or 'text/plain',
        )
        response['Content-Disposition'] = 'attachment; filename="%s"' % filename
        return response

    def make_json_response(self, data):
        return JsonResponse(
            data,
            json_dumps_params={'ensure_ascii': self.ensure_ascii_response},
        )

    def respond_error(self, e):
        return self.make_json_response(dict(self.response_values, status='error', errors=e.errors))

    def respond_success(self):
        if self.file_response:
            return self.make_file_response(
                data=self.file_response['data'],
                filename=self.file_response['filename'],
                content_type=self.file_response.get('content_type'),
            )
        else:
            return self.make_json_response(dict(self.response_values, status='ok'))

    def process_request(self, request):
        raise NotImplementedError()  # pragma: no cover

    def get_user_by_uid(self, uid):
        bb_response = get_blackbox().userinfo(
            uid=uid,
            ip=self.request.env.user_ip,
            dbfields=settings.BLACKBOX_DBFIELDS,
            attributes=settings.BLACKBOX_ATTRIBUTES,
            need_display_name=True,
            need_public_name=False,
            need_aliases=True,
        )
        if not bb_response['uid']:
            raise UserNotFoundError()

        return UserDict.from_userinfo_response(bb_response)

    def get_user_by_login(self, login, need_aliases=True, need_public_name=False, need_display_name=True):
        bb_response = get_blackbox().userinfo(
            login=login,
            ip=self.request.env.user_ip,
            dbfields=settings.BLACKBOX_DBFIELDS,
            attributes=settings.BLACKBOX_ATTRIBUTES,
            need_display_name=need_aliases,
            need_public_name=need_public_name,
            need_aliases=need_aliases,
        )
        if not bb_response['uid']:
            raise UserNotFoundError()

        return UserDict.from_userinfo_response(bb_response)

    def get_user_from_user_ticket(self, required_scope=None):
        if self.request.env.user_ticket is None:
            raise UserTicketEmptyError()
        try:
            user_ticket = get_tvm_credentials_manager().get_user_context().check(self.request.env.user_ticket)
        except TicketParsingException as e:
            log.debug('Invalid TVM user ticket: %s, %s, %s', e.status, e.message, e.debug_info)
            raise UserTicketValidationError('invalid')

        if required_scope is None:
            required_scope = [settings.SESSIONID_SCOPE]
        if required_scope:
            if not isinstance(required_scope, (list, tuple)):
                required_scope = [required_scope]
            if not any(user_ticket.has_scope(scope) for scope in required_scope):
                log.debug(
                    'Invalid TVM user ticket scope: expected one of %s, got %s',
                    ','.join(required_scope),
                    ','.join(user_ticket.scopes),
                )
                raise UserTicketValidationError('scope missing')

        return self.get_user_by_uid(uid=user_ticket.default_uid)

    def get_user_from_session(self, uid=None):
        sessionid = self.request.env.cookies.get('Session_id')
        sslsessionid = self.request.env.cookies.get('sessionid2')
        sessguard = self.request.env.cookies.get('sessguard')
        if not sessionid:
            raise SessionidEmptyError()

        try:
            to_statbox(
                mode='check_cookies',
                host=self.request.env.host,
                consumer=self.consumer,
                have_sessguard=sessguard is not None,
                sessionid=mask_sessionid(sessionid),
            )
            bb_response = get_blackbox().sessionid(
                sessionid=sessionid,
                sslsessionid=sslsessionid,
                sessguard=sessguard,
                host=self.request.env.host,
                ip=self.request.env.user_ip,
                dbfields=settings.BLACKBOX_DBFIELDS,
                attributes=settings.BLACKBOX_ATTRIBUTES,
                need_display_name=True,
                need_public_name=False,
                multisession=True,
                need_aliases=True,
                get_login_id=True,
                request_id=self.request.env.request_id,
            )
        except BlackboxInvalidParams as e:
            raise BlackboxInvalidParamsError(e)

        # Сначала проверим статус всей куки
        cookie_status = bb_response['cookie_status']
        if cookie_status == BLACKBOX_SESSIONID_WRONG_GUARD_STATUS:
            raise SessguardInvalidError('Sessguard is missing or invalid')
        elif cookie_status not in (
            BLACKBOX_SESSIONID_VALID_STATUS,
            BLACKBOX_SESSIONID_NEED_RESET_STATUS,
        ):
            raise SessionidInvalidError('Auth cookie is invalid')

        # Затем статус требуемого пользователя
        users = bb_response['users']
        uid = uid or bb_response['default_uid']
        try:
            user_session = users[uid]
        except KeyError:
            raise UidNotInSessionError('Known uids: %s' % users.keys())

        if user_session['status'] not in (
            BLACKBOX_SESSIONID_VALID_STATUS,
            BLACKBOX_SESSIONID_NEED_RESET_STATUS,
        ):
            raise SessionidInvalidError('User session is invalid')

        return UserDict.from_sessionid_response(bb_response, uid)

    def get_user_from_token(self, require_x_token=True, required_scope=None):
        if not self.oauth_token:
            raise AuthorizationHeaderError()

        bb_response = get_blackbox().oauth(
            oauth_token=self.oauth_token,
            ip=self.request.env.user_ip,
            dbfields=settings.BLACKBOX_DBFIELDS,
            attributes=settings.BLACKBOX_ATTRIBUTES,
            need_display_name=True,
            need_public_name=False,
            need_aliases=True,
            get_login_id=True,
        )
        if bb_response['status'] != BLACKBOX_OAUTH_VALID_STATUS:
            raise OAuthTokenValidationError('token invalid')

        if require_x_token and not any(
            Scope.by_keyword(keyword).has_xtoken_grant
            for keyword in bb_response['oauth']['scope']
        ):
            raise OAuthTokenValidationError('not x-token')

        if required_scope and required_scope not in bb_response['oauth']['scope']:
            raise OAuthTokenValidationError('scope missing')

        return UserDict.from_token_response(bb_response)

    def dispatch_request(self, request, http_method):
        try:
            if http_method not in self.allowed_methods:
                log_warning('Method not allowed (%s): %s' % (request.method, request.path))
                return self.respond_error(MethodNotAllowedError())

            self.process_root_form(request.REQUEST)
            check_grants(
                grants=self.required_grants,
                consumer=self.consumer,
                ip=request.env.consumer_ip,
                service_ticket=request.env.service_ticket,
            )

            if self.base_form is not None:
                self.process_base_form(request.REQUEST, files=request.FILES)

            self.process_request(request)
        except TVMTicketInvalid as e:
            return self.respond_error(TVMTicketInvalidError(e))
        except GrantsMissing as e:
            return self.respond_error(GrantsMissingError(e))
        except BaseApiError as e:
            return self.respond_error(e)
        except tuple(self.temporary_exceptions_list) as e:
            return self.respond_error(BackendFailedError(e))
        except Exception as e:
            log_error(request)
            return self.respond_error(ExceptionUnhandledError(e))

        return self.respond_success()

    def get(self, request):
        return self.dispatch_request(request, http_method='GET')

    def post(self, request):
        return self.dispatch_request(request, http_method='POST')


class UserDict(dict):
    def __init__(self):
        super(UserDict, self).__init__()

        for attr in [
            'aliases',
            'attributes',
            'bb_response',
            'login_id',
            'password_verification_age',
            'uid',
        ]:
            self[attr] = None

    @classmethod
    def from_userinfo_response(cls, userinfo):
        self = cls()
        self.update(cls._build_base(userinfo))
        self.update({
            'password_verification_age': -1,  # взять эту инфу неоткуда
            'bb_response': userinfo,
            'login_id': None,  # взять эту инфу неоткуда
        })
        return self

    @classmethod
    def from_sessionid_response(cls, session_info, uid):
        self = cls()
        userinfo = session_info['users'][uid]
        self.update(cls._build_base(userinfo))
        self.update({
            'password_verification_age': userinfo['auth'].get('password_verification_age', -1),
            'bb_response': userinfo,
            'login_id': session_info.get('login_id'),
        })
        return self

    @classmethod
    def from_token_response(cls, token_info):
        self = cls()
        self.update(cls._build_base(token_info))
        self.update({
            'password_verification_age': -1,  # в случае токена взять эту инфу неоткуда
            'bb_response': token_info,
            'login_id': token_info.get('login_id'),
        })
        return self

    @staticmethod
    def _build_base(userinfo):
        base = {
            'aliases': userinfo['aliases'],
            'attributes': userinfo.get('attributes', {}),
            'uid': userinfo['uid'],
        }
        base['have_password'] = get_attribute(base, settings.BB_ATTR_ACCOUNT_HAVE_PASSWORD)
        return base
