# -*- coding: utf-8 -*-
from passport.backend.oauth.core.common.errors import BaseError


class BaseApiError(BaseError):
    errors = None


class TVMTicketInvalidError(BaseApiError):
    errors = ['tvm_ticket.invalid']


class GrantsMissingError(BaseApiError):
    errors = ['grants.missing']


class ExceptionUnhandledError(BaseApiError):
    """Непредвиденная ошибка бэкенда"""
    errors = ['exception.unhandled']


class BackendFailedError(BaseApiError):
    """Предвиденная, но необработанная ошибка бэкенда"""
    errors = ['backend.failed']


class MethodNotAllowedError(BaseApiError):
    """HTTP-метод не разрешен"""
    errors = ['method.not_allowed']


class InvalidFormError(BaseApiError):
    """Ошибка валидации входных данных"""
    def __init__(self, errors, *args, **kwargs):
        self.errors = []
        super(InvalidFormError, self).__init__(self, *args, **kwargs)
        for field, error_list in errors.items():
            self.errors += ['%s.%s' % (field, error) for error in set(error_list)]


class BlackboxInvalidParamsError(BaseApiError):
    """Переданы недопустимые значения ip, куки или прочих параметров"""
    errors = ['blackbox.invalid_params']


class UserNotFoundError(BaseApiError):
    """Пользователь не найден"""
    errors = ['user.not_found']


class AuthorizationHeaderError(BaseApiError):
    """Невалидный заголовок авторизации"""
    errors = ['authorization.invalid']


class OAuthTokenValidationError(BaseApiError):
    """Невалидный OAuth-токен"""
    errors = ['oauth_token.invalid']


class UserTicketEmptyError(BaseApiError):
    """Юзер-тикет не передан"""
    errors = ['user_ticket.empty']


class UserTicketValidationError(BaseApiError):
    """Невалидный юзер-тикет"""
    errors = ['user_ticket.invalid']


class SessguardInvalidError(BaseApiError):
    """Значение sessguard не прошло проверку в ЧЯ"""
    errors = ['sessguard.invalid']


class SessionidEmptyError(BaseApiError):
    """Session_id не передана"""
    errors = ['sessionid.empty']


class SessionidInvalidError(BaseApiError):
    """Значение Session_id не прошло проверку в ЧЯ"""
    errors = ['sessionid.invalid']


class UidNotInSessionError(BaseApiError):
    """Уид не присутствует в сессии"""
    errors = ['sessionid.no_uid']


class ClientNotFoundError(BaseApiError):
    """Приложение не найдено"""
    errors = ['client.not_found']


__all__ = (
    'AuthorizationHeaderError',
    'BackendFailedError',
    'BlackboxInvalidParamsError',
    'ClientNotFoundError',
    'ExceptionUnhandledError',
    'GrantsMissingError',
    'InvalidFormError',
    'MethodNotAllowedError',
    'OAuthTokenValidationError',
    'SessguardInvalidError',
    'SessionidEmptyError',
    'SessionidInvalidError',
    'TVMTicketInvalidError',
    'UidNotInSessionError',
    'UserNotFoundError',
    'UserTicketEmptyError',
    'UserTicketValidationError',
)
