# -*- coding: utf-8 -*-
from django import forms
from django.core.exceptions import ValidationError
from passport.backend.utils.time import unixtime_to_datetime


MAX_UID = 2 ** 63 - 2  # Все, что больше, ЧЯ не считает валидным UID-ом
MAX_TIMESTAMP_VALUE = 2 ** 31 - 1


# маппинг из ключей ошибок Django в ошибки АПИ
error_messages = {
    'required': 'missing',
    'invalid': 'invalid',
    'max_value': 'invalid',
    'min_value': 'invalid',
    'max_length': 'too_long',
    'min_length': 'too_short',
    'invalid_link': 'invalid',
    'invalid_choice': 'invalid',
    'invalid_list': 'invalid',
}


class ConsumerForm(forms.Form):
    consumer = forms.fields.CharField(required=True, error_messages=error_messages)


class StrippedCharField(forms.fields.CharField):
    def __init__(self, max_length=100, **kwargs):
        super(StrippedCharField, self).__init__(
            max_length=max_length,
            strip=True,
            **kwargs
        )


class TimestampField(forms.fields.IntegerField):
    def __init__(self, **kwargs):
        super(TimestampField, self).__init__(
            min_value=0,
            max_value=MAX_TIMESTAMP_VALUE,
            **kwargs
        )

    def clean(self, value):
        timestamp = super(TimestampField, self).clean(value)
        return unixtime_to_datetime(timestamp or 0)


class BooleanField(forms.fields.BooleanField):
    def __init__(self, default=False, **kwargs):
        super(BooleanField, self).__init__(
            required=False,
            widget=forms.widgets.TextInput(),
            **kwargs
        )
        self._default = default

    def clean(self, value):
        if value is None:
            return self._default
        if isinstance(value, str):
            if value.lower() in ('true', 'yes', '1'):
                return True
            elif value.lower() in ('false', 'no', '0'):
                return False
        return bool(value)


class IntegerField(forms.fields.IntegerField):
    def __init__(self, default=None, **kwargs):
        super(IntegerField, self).__init__(**kwargs)
        self._default = default

    def clean(self, value):
        value = super(IntegerField, self).clean(value)
        if value is None:
            return self._default
        return value


class MappingField(forms.fields.ChoiceField):
    def __init__(self, mapping, required=True, default=None, **kwargs):
        super(MappingField, self).__init__(
            required=required,
            widget=forms.Select(),
            choices=zip(mapping.keys(), mapping.keys()),
            **kwargs
        )
        self._default = default
        self._mapping = mapping.copy()

    def clean(self, value):
        raw_value = super(MappingField, self).clean(value)
        return self._mapping.get(raw_value, self._mapping.get(self._default, raw_value))


class CollectionField(forms.fields.Field):
    def __init__(self, validator, max_count, required=False, **kwargs):
        super(CollectionField, self).__init__(
            required=required,
            widget=forms.widgets.SelectMultiple,
            **kwargs
        )
        self._validator = validator
        self._max_count = max_count

    def clean(self, value):
        raw_items = super(CollectionField, self).clean(value)
        if not raw_items:
            return []

        if len(raw_items) > self._max_count:
            raise ValidationError('too_many')

        cleaned_items = []
        error_codes = []
        for item_id, raw_item in enumerate(raw_items):
            try:
                cleaned_item = self._validator.clean(raw_item)
                cleaned_items.append(cleaned_item)
            except ValidationError as e:
                error_codes.append('%s.%s' % (item_id, e.messages[0]))
        if error_codes:
            raise ValidationError(error_codes)

        return cleaned_items


class UidField(forms.fields.IntegerField):
    def __init__(self, required=True, **kwargs):
        super(UidField, self).__init__(
            required=required,
            min_value=1,
            max_value=MAX_UID,
            **kwargs
        )


class ExtendedValidatorsMixin(forms.Form):
    def _filter_missing(self, cleaned_data, fields):
        return [
            field
            for field in fields
            if not cleaned_data.get(field)
        ]

    def _add_errors_for_missing(self, missing_fields):
        # Выдаём ошибки для всех полей, для которых нет других ошибок валидации
        for field in missing_fields:
            if not self.has_error(field):
                self.add_error(field, 'required')

    def require_if_has_value(self, cleaned_data, collection_field, value, required_fields):
        """Если collection_field содержит значение value, требует все поля из required_fields"""
        if value in cleaned_data.get(collection_field, []):
            missing_fields = self._filter_missing(cleaned_data, required_fields)
            self._add_errors_for_missing(missing_fields)


class UidMixin(forms.Form):
    uid = UidField(error_messages=error_messages)


class UidOptionalMixin(forms.Form):
    uid = UidField(required=False, error_messages=error_messages)
