# -*- coding: utf-8 -*-
import json

from django.conf import settings
from django.http import (
    HttpResponse,
    HttpResponseForbidden,
    HttpResponseRedirect,
    QueryDict,
)
from django.urls import (
    NoReverseMatch,
    reverse,
)
from django.utils.deprecation import MiddlewareMixin
from passport.backend.oauth.core.api.base import (
    log_request,
    log_response,
)
from passport.backend.oauth.core.common.globals import set_request
from passport.backend.oauth.core.env import Environment


class RequestDataMiddleware(MiddlewareMixin):
    """
    Упрощает парсинг данных, пришедших в запросе
    """
    def process_request(self, request):
        if request.content_type not in ('multipart/form-data', 'application/x-www-form-urlencoded'):
            # Django<1.5 позволяла не указывать Content-Type, и некоторые потребители этим пользовались
            request.POST = QueryDict(request.body, encoding=request.encoding)

        # request.REQUEST задеприкейтился и был удалён, но нам он нужен
        all_values = request.GET.copy()
        for key, value in request.POST.lists():
            all_values.setlist(key, value)
        request.REQUEST = all_values


class HTTPSMiddleware(MiddlewareMixin):
    def __init__(self, *args, **kwargs):
        super(HTTPSMiddleware, self).__init__(*args, **kwargs)
        # Да, мне не нравится это. Но работает + малой кровью
        # Также в любой момент легко поменять :-)
        self.optional_https = set()
        try:
            self.optional_https = set([
                reverse('ping').lower(),
            ])
        except NoReverseMatch:  # for using HTTPSMiddleware in oauth_admin
            pass
        self.trusted_hosts = settings.NO_HTTPS_FOR_HOSTS

    def process_request(self, request):
        if settings.REQUIRE_HTTPS:
            if request.path.lower() in self.optional_https:
                return
            if request.get_host() in self.trusted_hosts:
                return
            if not request.is_secure():
                if request.method in ['GET', 'HEAD']:
                    return HttpResponseRedirect('https://%s%s' % (request.get_host(), request.get_full_path()))
                elif request.method in ['POST']:
                    return HttpResponseForbidden('Use HTTPS')
                else:
                    return HttpResponse(status=405)


class StoreEnvironmentMiddleware(MiddlewareMixin):
    """
    Выставляет на request поле env, содержащее подробную информацию о запросе
    """
    def process_request(self, request):
        request.env = Environment.from_request(request)


class StoreRequestMiddleware(MiddlewareMixin):
    """Сохраняет реквест в глобальную переменную"""
    def process_request(self, request):
        set_request(request)

    def process_response(self, request, response):
        set_request(None)
        return response


class LoggingMiddleware(MiddlewareMixin):
    """Логирует запрос и ответ"""
    def process_request(self, request):
        log_request(request)

    def process_response(self, request, response):
        if response.get('Content-Disposition', 'inline').startswith('attachment'):
            response_values = '<%s> (%d bytes)' % (response['Content-Disposition'], len(response.content))
        elif response['Content-Type'] == 'application/json':
            response_values = json.loads(response.content)
        else:
            response_values = response.content.decode().strip()

        log_response(response_values, status=response.status_code)
        response._has_been_logged = True  # чтобы Django не логировал ответ повторно
        return response
