# -*- coding: utf-8 -*-
import logging
import os

from django.conf import settings
from django.http import (
    HttpResponse,
    JsonResponse,
)
from passport.backend.core.builders.blackbox.blackbox import BaseBlackboxError
from passport.backend.oauth.core.common.blackbox import get_blackbox
from passport.backend.oauth.core.db.eav import DBTemporaryError
from passport.backend.oauth.core.db.eav.dbmanager import get_dbm


log = logging.getLogger('ping')


def error_response(error, status=400, description=None, **kwargs):
    response_values = {'error': error}
    if description:
        response_values['error_description'] = description
    response_values.update(kwargs)
    return JsonResponse(response_values, status=status)


def ping(request):
    objects_to_check = [
        value.strip().lower()
        for value in request.GET.get('check', '').split(',')
    ]

    # Проверяем доступность файла. Если его нет или он недоступен - значит, мы должны закрыться от балансера
    if not os.access(settings.PING_TEST_FILE, os.R_OK):
        return error_response(error='service.shut_down', status=521)

    if 'db' in objects_to_check:
        # Пингуем все базы (слейвы)
        for db_name in [settings.CENTRAL_DB_NAME] + list(settings.SHARD_NAMES):
            dbm = get_dbm(db_name)
            try:
                dbm.ping(retries=settings.DB_AVAILABILITY_TEST_RETRIES)
            except DBTemporaryError as e:
                log.warning('Ping db: %s', e)
                return error_response(error='db.unavailable', description=str(e), status=521)

    if 'blackbox' in objects_to_check:
        try:
            bb_response = get_blackbox().userinfo(
                login=settings.BLACKBOX_AVAILABILITY_TEST_LOGIN,
                ip='127.0.0.1',
                dbfields=[],
                attributes=[],
                need_display_name=False,
            )
        except BaseBlackboxError as e:
            log.warning('Ping blackbox: %s', e)
            return error_response(error='blackbox.unavailable', description=str(e), status=521)

        if not bb_response['uid']:
            log.error('Ping blackbox: test user not found')

    return HttpResponse('Pong\n')
