# -*- coding: utf-8 -*-
import logging

import django
from django.conf import settings as oauth_settings
from passport.backend.core.conf import settings as passport_settings
from passport.backend.core.geobase import get_geobase
from passport.backend.core.tvm.tvm_credentials_manager import get_tvm_credentials_manager
from passport.backend.core.useragent.sync import get_useragent
from passport.backend.oauth.core.common.portallib import get_ipreg
from passport.backend.oauth.core.db.config import (
    get_client_lists,
    get_scopes,
    get_token_params,
)
from passport.backend.oauth.core.db.config.device_names_mapping import get_device_names_mapping
from passport.backend.oauth.core.db.config.grants_config import get_grants
from passport.backend.oauth.core.db.config.login_to_uid_mapping import get_login_to_uid_mapping
from passport.backend.oauth.core.db.config.translations import (
    get_scope_short_translations,
    get_scope_translations,
    get_service_translations,
)
from passport.backend.oauth.core.db.startup import (
    configure_mysql,
    prepare_mysql_connections,
)
from passport.backend.utils.warnings import enable_strict_bytes_mode


log = logging.getLogger('startup')


def configure_settings():
    if not passport_settings.configured:
        passport_settings.configure(oauth_settings)


def configure_db_connections():
    # Не создаём соединения с БД в prepare_environment,
    # т.к. prepare_environment выполняется _до_ форка gunicorn,
    # и всё созданное там потом переиспользуется в воркерах _после_ форка.
    # Поэтому явно вызываем эту функцию в хуке гуникорна post_worker_init
    # уже _после_ форка, в каждом воркере, отдельно.
    log.debug('Configuring DB...')
    configure_mysql()
    prepare_mysql_connections()


def prepare_environment(skip_db=False, skip_geodata=False, skip_configs=False):
    log.debug('Preparing common environment...')

    django.setup()
    enable_strict_bytes_mode()
    configure_settings()

    if not skip_db:
        configure_db_connections()

    log.debug('Preloading singletons...')
    get_useragent()

    if not skip_configs:
        get_grants()
        get_tvm_credentials_manager()
        get_client_lists()
        get_device_names_mapping()
        get_login_to_uid_mapping()
        get_scopes()
        get_scope_translations()
        get_scope_short_translations()
        get_service_translations()
        get_token_params()

    if not skip_geodata:
        log.debug('Preloading geodata...')
        get_geobase()
        get_ipreg()

    log.debug('Common environment prepared.')
