# -*- coding: utf-8 -*-
import os

from cryptography.exceptions import (
    InternalError,
    InvalidTag,
)
from cryptography.hazmat.backends import default_backend
from cryptography.hazmat.primitives.ciphers import (
    algorithms,
    Cipher,
    modes,
)
from django.utils.encoding import smart_bytes
from passport.backend.oauth.core.common.errors import BaseError
from passport.backend.oauth.core.common.utils import (
    from_base64_url,
    to_base64_url,
)


"""
Утилиты для шифрования методом AES-GCM
https://cryptography.io/en/latest/hazmat/primitives/symmetric-encryption/#cryptography.hazmat.primitives.ciphers.modes.GCM
"""


SEPARATOR = b':'


class BaseCryptoError(BaseError):
    pass


class UnsupportedVersionError(BaseCryptoError):
    """Зашифрованная строка не содержит версии или версия не поддерживается"""


class DecryptFailedError(BaseCryptoError):
    """Не удалось расшифровать сообщение"""


# Низкоуровневая часть - реализация конкретных алгоритмов


def encrypt_aes_gcm(key, plain_text):
    key = smart_bytes(key)
    plain_text = smart_bytes(plain_text)
    initialization_vector = os.urandom(12)  # 96 бит

    encryptor = Cipher(
        algorithms.AES(key),
        modes.GCM(initialization_vector),
        backend=default_backend(),
    ).encryptor()

    cipher_text = encryptor.update(plain_text) + encryptor.finalize()

    return initialization_vector, cipher_text, encryptor.tag


def decrypt_aes_gcm(key, initialization_vector, cipher_text, tag):
    key = smart_bytes(key)
    cipher_text = smart_bytes(cipher_text)
    decryptor = Cipher(
        algorithms.AES(key),
        modes.GCM(initialization_vector, tag),
        backend=default_backend(),
    ).decryptor()

    return decryptor.update(cipher_text) + decryptor.finalize()


# Высокоуровневая часть - версионированные шифрование и дешифрование строк


def encrypt(keys, message, version=1):
    """
    Зашифровать сообщение шифром указанной версии.
    keys - словарь: отображение из версии в используемый ключ
    """
    iv, cipher_text, tag = encrypt_aes_gcm(keys[version], smart_bytes(message))
    return SEPARATOR.join(
        [
            smart_bytes(version),
        ] + list(map(
            to_base64_url,
            [iv, cipher_text, tag],
        )),
    )


def decrypt(keys, encrypted_message):
    """
    Расшифровать сообщение. keys - словарь ключей, использованный при шифровании.
    Возвращает пару (версия, расшифрованное_сообщение)
    """
    encrypted_message = smart_bytes(encrypted_message)
    if SEPARATOR not in encrypted_message:
        raise UnsupportedVersionError('Version is missing')

    version, remainder = encrypted_message.split(SEPARATOR, 1)
    try:
        version = int(version)
        key = keys[version]
    except (ValueError, KeyError):
        raise UnsupportedVersionError(version)

    try:
        iv, cipher_text, tag = map(
            from_base64_url,
            remainder.split(SEPARATOR),
        )
        return version, decrypt_aes_gcm(key, iv, cipher_text, tag)
    except (TypeError, UnicodeEncodeError, ValueError, InvalidTag, InternalError) as e:
        raise DecryptFailedError('%s: %s' % (e.__class__.__name__, e))
