# -*- coding: utf-8 -*-
from copy import deepcopy
from functools import wraps
import time


def retry(exception_to_catch, exception_to_raise=None, retries=3, timeout=0,
          logger=None, name_for_logger=None):
    """
    Выполняет декорируемую функцию, ретраясь в случае возникновения исключения.
    :param exception_to_catch: исключение (или tuple исключений), при возникновении которого ретраимся.
    :param exception_to_raise: исключение, которое будет выброшено при достижении лимита ретраев.
      Если None, то выбросим пойманное.
    :param retries: лимит ретраев.
    :param timeout: таймаут между ретраями.
    :param logger: логгер, в который будет записано сообщение о случившемся ретрае.
    :param name_for_logger: имя выполняемого метода (для записи в лог).
    :return: результат декорируемой функции.
    """
    def decorator(func):
        @wraps(func)
        def wrapper(*args, **kwargs):
            retries_left = retries
            while retries_left > 0:
                try:
                    return func(*deepcopy(args), **deepcopy(kwargs))
                except exception_to_catch as e:
                    retries_left -= 1
                    if not retries_left:
                        raise exception_to_raise(e) if exception_to_raise is not None else e
                    if logger:
                        logger.debug(
                            '%s failed (%s), retrying (attempt %d/%d)' % (
                                name_for_logger or func.__name__,
                                e.__class__.__name__,
                                retries - retries_left + 1,
                                retries,
                            ),
                        )
                    time.sleep(timeout)
        return wrapper
    return decorator
