# -*- coding: utf-8 -*-
import base64
import hashlib
from itertools import groupby
import random
from urllib.parse import (
    parse_qsl,
    urlencode,
    urlparse,
    urlunparse,
)
import uuid

from django.conf import settings
from django.utils.encoding import (
    smart_bytes,
    smart_text,
)


ESCAPE_MAP = (
    (b'\\', b'\\\\'),
    (b'\t', b'\\t'),
    (b'\n', b'\\n'),
    (b'\r', b'\\r'),
    (b'\0', b'\\0'),
)


class classproperty(object):
    def __init__(self, method):
        self.method = method

    def __get__(self, obj, owner):
        return self.method(owner)


def first_or_none(collection):
    if not collection:
        return None
    return collection[0]


def int_or_default(str_value, default=None):
    try:
        return int(str_value)
    except (ValueError, TypeError):
        return default


def split_by_predicate(sequence, predicate, item_getter=None):
    positive, negative = [], []
    for item in sequence:
        value = item_getter(item) if item_getter else item
        if predicate(item):
            positive.append(value)
        else:
            negative.append(value)
    return positive, negative


def smart_any(function, sequence):
    for x in sequence:
        if function(x):
            return True
    return False


def escape(text):
    text = smart_bytes(text)
    for char, replacement in ESCAPE_MAP:
        text = text.replace(char, replacement)
    return smart_text(text, errors='replace')


def mask_string(string, show_first_n):
    """Заменяет все символы строки, кроме нескольких первых, звёздочками"""
    if not string:
        return string
    return string[:show_first_n] + '*' * (len(string) - show_first_n)


def random_stuff():
    return uuid.uuid4().hex


def make_token_alias():
    return ''.join(
        random.choice(settings.TOKEN_ALIAS_ALPHABET)
        for _ in range(settings.TOKEN_ALIAS_LENGTH)
    )


def make_hash(value, version=1):
    assert version == 1, f'Version {version} not supported yet'
    return b':'.join([
        str(version).encode(),
        base64.b64encode(hashlib.sha256(smart_bytes(value)).digest()).rstrip(b'='),
    ])


def sorted_uniqs(items):
    """
    Возвращает список уникальных элементов, отсортированный по частоте (uniq -c | sort -rn)
    """
    sorted_items = sorted(items, key=items.count, reverse=True)
    return [item for item, _ in groupby(sorted_items)]


def to_base64_url(text):
    return base64.urlsafe_b64encode(smart_bytes(text)).strip(b'=')


def from_base64_url(text):
    if len(text) % 4:
        text += b'=' * (4 - len(text) % 4)
    return base64.urlsafe_b64decode(smart_bytes(text))


def int_to_bytes(number, byte_count):
    number %= 256 ** byte_count
    return number.to_bytes(byte_count, byteorder='big', signed=False)


def bytes_to_int(bytes_):
    return int.from_bytes(bytes_, byteorder='big', signed=False)


def hex_to_bytes(hex):
    return bytes.fromhex(hex)


def update_url_params(url, **params):
    url_parts = urlparse(url)
    query = dict(parse_qsl(url_parts.query))
    query.update(params)
    url_parts = url_parts._replace(query=urlencode(query))
    return urlunparse(url_parts)


def normalize_url(url, drop_path_for_schemes=(), drop_params=False, drop_query=False, drop_fragment=False):
    """
    Приводит урл к каноническому представлению: приводит хост и схему к нижнему регистру, заменяет некоторые кастомные
    схемы, кодирует кириллические хосты в IDNA. При strict=True дополнительно отбрасывает params, query и fragment.

    Используется для сравнения урлов; для редиректа нужно использовать ненормализованную форму.
    """
    url_parts = urlparse(url)

    scheme = url_parts.scheme.lower()

    hostname = smart_text(url_parts.hostname or '').encode('idna')
    port = url_parts.port
    netloc = '%s:%s' % (smart_text(hostname), port) if port else hostname

    path = smart_text(url_parts.path or '')
    if scheme in drop_path_for_schemes:
        path = ''
    elif path and path.endswith('/'):
        path = path.rstrip('/')

    params = '' if drop_params else url_parts.params
    query = '' if drop_query else url_parts.query
    fragment = '' if drop_fragment else url_parts.fragment

    return urlunparse(map(smart_text, [scheme, netloc, path, params, query, fragment]))
