# -*- coding: utf-8 -*-
from collections import namedtuple
from urllib.parse import urlparse

from django.conf import settings
from django.utils.encoding import smart_bytes
from passport.backend.oauth.core.common.avatars import get_avatars_mds_api
from passport.backend.oauth.core.common.portallib import is_yandex_ip
from passport.backend.oauth.core.common.utils import (
    first_or_none,
    normalize_url,
    now,
    random_stuff,
    smart_any,
)
from passport.backend.oauth.core.db.config import get_client_lists
from passport.backend.oauth.core.db.eav import (
    EavAttr,
    EavModel,
    Index,
)
from passport.backend.oauth.core.db.eav.types import Event
from passport.backend.oauth.core.db.mixins.deletion_mixin import DeletionMixin
from passport.backend.oauth.core.db.mixins.ownership_mixin import OwnershipMixin
from passport.backend.oauth.core.db.mixins.scopes_mixin import ScopesMixin
from passport.backend.oauth.core.db.schemas import (
    client_attributes_table,
    client_by_display_id_table,
    client_by_owner_table,
    client_by_params_table,
    client_by_uid_table,
)
from passport.backend.oauth.core.db.scope import Scope
from passport.backend.oauth.core.db.turboapp import (
    make_turboapp_redirect_uri,
    TURBOAPP_URL_SCHEME,
)
from passport.backend.utils.time import unixtime_to_datetime


ApprovalStatus = namedtuple(
    'ApprovalStatus',
    ['NotRequired', 'Pending', 'Approved', 'Rejected'],
)._make(range(4))


EventType = namedtuple(
    'EventType',
    [
        'Truncated',
        'Blocked',
        'Unblocked',
        'ApprovalGranted',
        'ApprovalRejected',
        'ApprovalRequired',
        'ApprovalRequiredAgain',
        'ApprovalNotRequired',
        'CreatorChanged',
        'Glogouted',
    ]
)._make(range(1, 11))

EVENT_NAMES = {
    1: '...',
    2: 'Blocked',
    3: 'Unblocked',
    4: 'Approval granted',
    5: 'Approval rejected',
    6: 'Approval required',
    7: 'Approval required again',
    8: 'Approval not required any more',
    9: 'Owner changed',
    10: 'Glogouted'
}


APPROVAL_STATUS_NAMES = {
    0: 'Not required',
    1: 'Pending',
    2: 'Approved',
    3: 'Rejected',
}


PLATFORM_ANDROID = 'android'
PLATFORM_IOS = 'ios'
PLATFORM_TURBOAPP = 'turboapp'
PLATFORM_WEB = 'web'


def make_icon_key(client_id):
    """
    icon_key имеет вид "<client_id>-<random_hash>. (У старых приложений вместо random_hash был автоинкрементный id)
    """
    return '%s-%s' % (client_id, random_stuff())


def make_icon_id(group_id, key):
    return '%s/%s' % (group_id, key)


class Client(EavModel, ScopesMixin, OwnershipMixin, DeletionMixin):
    _table = client_attributes_table
    _indexes = {
        'uid': Index(client_by_uid_table),
        'params': Index(
            client_by_params_table,
            distinct_matched_key_fields=['uid', 'display_id', 'approval_status', 'is_yandex'],
            collection_fields=['services'],
        ),
        'owner': Index(
            client_by_owner_table,
            distinct_matched_key_fields=['owner_groups', 'owner_uids'],
            collection_fields=['uids'],
            nullable_fields=['owner_groups', 'owner_uids'],
            synthetic_fields=['uids'],
            extra_fields_generator=OwnershipMixin.make_values_for_owners_index,
        ),
        'display_id': Index(client_by_display_id_table),
    }

    uid = EavAttr()
    secret = EavAttr()
    old_secret = EavAttr()
    approval_status = EavAttr()
    is_blocked = EavAttr()
    is_yandex = EavAttr()
    default_title = EavAttr()
    title_ru = EavAttr()
    title_uk = EavAttr()
    title_en = EavAttr()
    title_tr = EavAttr()
    glogouted = EavAttr()
    created = EavAttr()
    modified = EavAttr()
    display_id = EavAttr()
    default_description = EavAttr()
    description_ru = EavAttr()
    description_uk = EavAttr()
    description_en = EavAttr()
    description_tr = EavAttr()
    icon = EavAttr()  # TODO: удалить после полного перехода на icon_id
    icon_id = EavAttr()  # строка вида "group_id/key" (в терминах аватарницы)
    homepage = EavAttr()
    services = EavAttr()
    telegram_bot_name = EavAttr()
    allow_nonpublic_granttypes = EavAttr()
    ios_default_app_id = EavAttr()
    ios_extra_app_ids = EavAttr()
    ios_appstore_url = EavAttr()
    android_default_package_name = EavAttr()
    android_extra_package_names = EavAttr()
    android_cert_fingerprints = EavAttr()
    android_appstore_url = EavAttr()
    turboapp_base_url = EavAttr()
    contact_email = EavAttr()

    _callback = EavAttr('callback')
    _redirect_uris = EavAttr('redirect_uris')
    _events = EavAttr('events')
    _extra_visible_scope_ids = EavAttr('extra_visible_scope_ids')

    @classmethod
    def create(cls, uid, scopes, default_title, redirect_uris=None, default_description=None,
               icon=None, icon_id=None, homepage=None):
        time_now = now()
        client = cls(
            uid=uid,
            secret=random_stuff(),
            old_secret=None,
            redirect_uris=redirect_uris or (),
            is_blocked=False,
            default_title=default_title,
            created=time_now,
            modified=time_now,
            display_id=random_stuff(),
            default_description=default_description,
            icon=icon,
            icon_id=icon_id,
            homepage=homepage,
        )
        client.set_scopes(scopes)
        return client

    @classmethod
    def _parse(cls, oauth_block):
        return oauth_block['client_attributes']

    def is_created_by(self, uid):
        return self.uid == uid

    def can_be_edited(self, uid, user_ip):
        if not self.is_created_by(uid) and not self.is_owned_by(uid):
            # Редактировать могут только создатель или владельцы
            return False
        if self.is_yandex and not is_yandex_ip(user_ip):
            # Яндексовые приложения разрешаем редактировать только из интранета
            return False
        return True

    def get_title(self, language='ru'):
        return getattr(self, 'title_%s' % language, None) or self.default_title

    def get_description(self, language='ru'):
        return getattr(self, 'description_%s' % language, None) or self.default_description

    def get_icon_url(self, size=None):
        if self.icon_id:
            group_id, key = self.icon_id.split('/')
            return get_avatars_mds_api().get_read_url(
                group_id=group_id,
                key=key,
                size=size or settings.AVATAR_DEFAULT_SIZE,
            )

    def set_scopes(self, scopes):
        require_approval = will_require_approval(new_scopes=scopes, client=self)

        super(Client, self).set_scopes(scopes)
        self.services = list(set(scope.service_name for scope in self.scopes))

        if require_approval:
            # модерация потребуется
            if self.approval_status == ApprovalStatus.NotRequired:
                self.add_event(EventType.ApprovalRequired)
            elif self.approval_status in [ApprovalStatus.Approved, ApprovalStatus.Rejected]:
                self.add_event(EventType.ApprovalRequiredAgain)
            self.approval_status = ApprovalStatus.Pending

        elif not smart_any(lambda scope: scope.requires_approval, self.scopes):
            # опасных прав нет - модерация не нужна
            if self.approval_status in [ApprovalStatus.Pending, ApprovalStatus.Approved, ApprovalStatus.Rejected]:
                self.add_event(EventType.ApprovalNotRequired)
            self.approval_status = ApprovalStatus.NotRequired

    @classmethod
    def by_display_id(cls, display_id, allow_deleted=False):
        return first_or_none(
            cls.by_index(
                'display_id',
                display_id=smart_bytes(display_id),
                limit=1,
                allow_deleted=allow_deleted,
            ),
        )

    @property
    def events(self):
        return [
            '%s %s' % (
                unixtime_to_datetime(event.timestamp),
                EVENT_NAMES.get(event.type, '?'),
            )
            for event in self._events
        ]

    @property
    def redirect_uris(self):
        result = []
        if self._callback:
            result.append(self._callback)
        result.extend(self._redirect_uris)
        return result

    @property
    def ios_app_ids(self):
        result = []
        if self.ios_default_app_id:
            result.append(self.ios_default_app_id)
        result.extend(self.ios_extra_app_ids)
        return result

    @property
    def android_package_names(self):
        result = []
        if self.android_default_package_name:
            result.append(self.android_default_package_name)
        result.extend(self.android_extra_package_names)
        return result

    @property
    def platforms(self):
        result = set()
        if self.ios_app_ids:
            result.add(PLATFORM_IOS)
        if self.android_package_names and self.android_cert_fingerprints:
            result.add(PLATFORM_ANDROID)
        if self.turboapp_base_url:
            result.add(PLATFORM_TURBOAPP)
        if self.redirect_uris:
            result.add(PLATFORM_WEB)
        return result

    @property
    def platform_specific_urlscheme(self):
        return 'yx%s' % self.display_id

    @property
    def universal_link_domains(self):
        result = []
        for tld in settings.OAUTH_TLDS:
            host = settings.OAUTH_PUBLIC_HOST_TEMPLATE % {'tld': tld}
            result.append('https://yx%s.%s' % (self.display_id, host))
        return result

    @property
    def platform_specific_redirect_uris(self):
        result = []
        for platform in (PLATFORM_IOS, PLATFORM_ANDROID):
            if platform in self.platforms:
                for domain in self.universal_link_domains:
                    result.append(
                        '%s/auth/finish?platform=%s' % (
                            domain,
                            platform,
                        ),
                    )
                result.append(
                    '%s:///auth/finish?platform=%s' % (
                        self.platform_specific_urlscheme,
                        platform,
                    ),
                )
        if PLATFORM_TURBOAPP in self.platforms:
            result.append(
                make_turboapp_redirect_uri(self.turboapp_base_url),
            )
        return result

    @property
    def default_redirect_uri(self):
        return first_or_none(self.redirect_uris + self.platform_specific_redirect_uris)

    @property
    def requires_approval(self):
        return self.approval_status == ApprovalStatus.Pending

    def set_extra_visible_scopes(self, visible_scopes):
        self._extra_visible_scope_ids = [scope.id for scope in visible_scopes]

    @property
    def extra_visible_scopes(self):
        """Скоупы, открытые специально этому приложению"""
        scopes = set([Scope.by_id(scope_id) for scope_id in self._extra_visible_scope_ids])

        if settings.DEFAULT_PHONE_SCOPE_KEYWORD is not None:
            default_phone_scope = Scope.by_keyword(settings.DEFAULT_PHONE_SCOPE_KEYWORD)
            # Турбоаппы и некоторые другие приложения могут видеть телефонный скоуп.
            if (
                (settings.CAN_TURBOAPPS_SEE_DEFAULT_PHONE_SCOPE and PLATFORM_TURBOAPP in self.platforms) or
                self.display_id in get_client_lists().whitelist_for_scope(settings.DEFAULT_PHONE_SCOPE_KEYWORD)
            ):
                scopes.add(default_phone_scope)

        # Также если у приложения есть какой-то скоуп, в настоящее время не видимый создателю приложения - продолжаем
        # показывать его в списке доступных (чтобы при любом редактировании приложения не отозвались все его токены)
        scopes.update(
            scope
            for scope in self.scopes
            if scope not in self.scopes_visible_for_creator
        )
        return scopes

    @property
    def scopes_visible_for_creator(self):
        """Скоупы, которые владелец приложения видит на экране создания приложения"""
        return set(Scope.list(
            show_hidden=False,
            uid=self.uid,
        ))

    @property
    def visible_scopes(self):
        """Скоупы, которые владелец приложения видит на экране редактирования приложения"""
        visible_scopes = self.scopes_visible_for_creator
        visible_scopes.update(self.extra_visible_scopes)
        return visible_scopes

    @property
    def scopes_invisible_for_creator(self):
        """Скоупы, которые уже есть у приложения, но владелец приложения их не видит"""
        return self.scopes - self.scopes_visible_for_creator

    def add_event(self, event_type, timestamp=None):
        if not self._events:
            self._events = []
        self._events.append(Event(type=event_type, timestamp=timestamp))
        if len(self._events) > settings.CLIENT_MAX_EVENTS_COUNT:
            self._events.pop(0)
            self._events[0].type = EventType.Truncated

    def make_new_secret(self):
        self.old_secret = self.secret
        self.secret = random_stuff()

    def restore_old_secret(self):
        if self.old_secret:
            self.secret = self.old_secret
            del self.old_secret

    def block(self):
        if not self.is_blocked:
            self.is_blocked = True
            self.add_event(EventType.Blocked)

    def unblock(self):
        if self.is_blocked:
            self.is_blocked = False
            self.add_event(EventType.Unblocked)

    def glogout(self):
        self.glogouted = now()
        self.add_event(EventType.Glogouted)

    def approve(self):
        if self.requires_approval:
            self.approval_status = ApprovalStatus.Approved
            self.add_event(EventType.ApprovalGranted)

    def reject_approval(self):
        if self.requires_approval:
            self.approval_status = ApprovalStatus.Rejected
            self.add_event(EventType.ApprovalRejected)

    def change_creator(self, uid):
        if self.uid != uid:
            self.uid = uid
            self.add_event(EventType.CreatorChanged)


def list_clients_by_creator(uid):
    return Client.by_index('uid', uid=uid)


def is_redirect_uri_insecure(redirect_uri):
    # Считаем небезопасными все http-урлы, кроме локальных
    parsed = urlparse(redirect_uri)
    return parsed.scheme == 'http' and parsed.hostname not in ('localhost', '127.0.0.1', '::1')


def is_redirect_uri_allowed(redirect_uri, allowed_redirect_uris):
    normalized_redirect_uri = normalize_url(
        redirect_uri,
        drop_path_for_schemes={TURBOAPP_URL_SCHEME},
        drop_params=True,
        drop_query=True,
        drop_fragment=True,
    )
    for allowed_uri in allowed_redirect_uris:
        normalized_allowed_uri = normalize_url(
            allowed_uri,
            drop_path_for_schemes={TURBOAPP_URL_SCHEME},
            drop_params=True,
            drop_query=True,
            drop_fragment=True,
        )
        if normalized_redirect_uri == normalized_allowed_uri:
            return True

    return False


def will_require_approval(new_scopes, client=None):
    """
    Потребуется ли одобрение модератора после создания приложения или изменения его скоупов
    """
    if client is not None and client.is_yandex:
        # защищаем яндексовые приложения от временной потери возможности выдавать новые токены
        return False

    old_scopes = client.scopes if client is not None else []
    scopes_added = [scope for scope in new_scopes if scope not in old_scopes]
    if smart_any(lambda scope: scope.requires_approval, scopes_added):
        # добавятся новые опасные права - модерация потребуется
        return True
    elif not smart_any(lambda scope: scope.requires_approval, new_scopes):
        # опасных прав не будет - модерация не нужна
        return False
    elif client is not None:
        # ничего не меняется по сравнению с текущим состоянием
        return client.approval_status == ApprovalStatus.Pending
    else:
        # приложение только регистрируется
        return False
