# -*- coding: utf-8 -*-
from django.conf import settings
from passport.backend.core.lazy_loader import (
    lazy_loadable,
    LazyLoader,
)
from passport.backend.oauth.core.db.config.base import BaseJsonConfig
from passport.backend.oauth.core.db.config.login_to_uid_mapping import get_uid_by_tagged_login_or_uid


@lazy_loadable('ACL')
class AclConfig(BaseJsonConfig):
    lazy_loadable_requirements = (
        'LoginToUidMapping',
    )

    def __init__(self):
        super(AclConfig, self).__init__(filename=settings.ACL_CONFIG)

    def postprocess(self):
        # резолвим все логины в уиды
        config = {}
        for tagged_login, acl_list in self.config.items():
            uid = get_uid_by_tagged_login_or_uid(tagged_login)
            config[uid] = acl_list
        self.config = config

    def acl(self, uid):
        all_permissions = [s.split(':', 1) for s in self.data.get(uid, [])]
        result = {}
        for group in ('action', 'service'):
            result[group] = [permission[1] for permission in all_permissions if permission[0] == group]
        return result

    def admin_for_services(self, uid, all_services):
        services_available = set(all_services)
        services_enabled = set(self.acl(uid)['service'])
        if '*' in services_enabled:
            return services_available
        return services_enabled.intersection(services_available)

    def is_action_enabled(self, uid, action):
        actions_enabled = self.acl(uid)['action']
        return action in actions_enabled or '*' in actions_enabled


def get_acl():
    return LazyLoader.get_instance('ACL')
