# -*- coding: utf-8 -*-
from django.conf import settings
from passport.backend.core.grants.grants_config import GrantsConfig as BaseGrantsConfig
from passport.backend.core.lazy_loader import (
    lazy_loadable,
    LazyLoader,
)
from passport.backend.core.tvm import get_tvm_credentials_manager
from ticket_parser2.exceptions import TicketParsingException


class BaseGrantsError(Exception):
    """Не хватает грантов"""


class TVMTicketInvalidError(BaseGrantsError):
    """Передан невалидный тикет"""
    def __init__(self, ticket_parsing_exception):
        super(TVMTicketInvalidError, self).__init__(
            'Got invalid TVM ticket: %s (%s)' % (
                ticket_parsing_exception.message.lower(),
                ticket_parsing_exception.debug_info,
            ),
        )


class GrantsMissingError(BaseGrantsError):
    """Не хватает грантов для использования ручки"""
    def __init__(self, grant, consumer, ip, client_id=None):
        self.grant = grant
        self.consumer = consumer
        self.ip = ip
        self.client_id = client_id
        # вызываю конструктор предка, чтобы правильно заполнились поля message и args
        consumer_info = '%s, client_id=%s' % (ip, client_id) if client_id else '%s, wo ticket' % ip
        super(GrantsMissingError, self).__init__(
            'Consumer "%s" (%s) doesn\'t have grant "%s"' % (
                consumer,
                consumer_info,
                grant,
            ),
        )


class AccessDeniedError(BaseGrantsError):
    """Не хватает грантов для выдачи токена"""
    def __init__(self, scope, limiter_type, forbidden_value):
        self.scope = scope
        self.limiter_type = limiter_type
        self.forbidden_value = forbidden_value
        super(AccessDeniedError, self).__init__(
            'Cannot use scope "%s": %s="%s" is forbidden' % (
                self.scope,
                self.limiter_type,
                self.forbidden_value,
            ),
        )


@lazy_loadable('Grants')
class GrantsConfig(BaseGrantsConfig):
    def __init__(self):
        super(GrantsConfig, self).__init__(
            grants_paths=[settings.GRANTS_CONFIG],
        )

    def _is_consumer_unknown(self, consumer):
        return consumer not in self.per_consumer_rtree

    def _is_grant_type_allowed(self, consumer, ip, grant_type):
        if self._is_consumer_unknown(consumer):
            return grant_type in settings.DEFAULT_GRANT_TYPES
        return self.has_grant(consumer, ip, 'grant_type.%s' % grant_type, service_ticket=None)

    def _is_client_allowed(self, consumer, ip, client_id):
        if self._is_consumer_unknown(consumer):
            return True
        return self.has_grant(consumer, ip, 'client.%s' % client_id, service_ticket=None)

    def _is_ip_allowed(self, consumer, ip, allow_unknown_consumers=False):
        if self._is_consumer_unknown(consumer):
            return allow_unknown_consumers
        return self.is_valid_ip(ip, consumer)

    def _get_ticket_src(self, service_ticket):
        if service_ticket is None:
            return
        try:
            parsed_ticket = get_tvm_credentials_manager().service_context.check(service_ticket)
            return parsed_ticket.src
        except TicketParsingException as ex:
            raise TVMTicketInvalidError(ex)

    def has_grant(self, consumer, ip, grant, service_ticket):
        if ':' in grant:
            # боремся с наследием старой схемы грантов
            raise ValueError('Deprecated grant notation ("%s"): colon is prohibited' % grant)
        permission = self.is_permitted(
            required_grants=[grant],
            optional_grants=[],
            ip=ip,
            consumer=consumer,
            tvm_client_id=self._get_ticket_src(service_ticket),
        )
        return permission.is_allowed

    def check_grant(self, grant, consumer, ip, service_ticket):
        if not self.has_grant(grant=grant, consumer=consumer, ip=ip, service_ticket=service_ticket):
            client_id = self._get_ticket_src(service_ticket)
            raise GrantsMissingError(grant, consumer, ip, client_id=client_id)

    def check_scope_grants(self, consumer, grant_type, client_id, ip):
        forbidden = None
        if not self._is_ip_allowed(consumer, ip, allow_unknown_consumers=True):
            forbidden = ('ip', ip)
        elif not self._is_grant_type_allowed(consumer, ip, grant_type):
            forbidden = ('grant_type', grant_type)
        elif not self._is_client_allowed(consumer, ip, client_id):
            forbidden = ('client_id', client_id)

        if forbidden:
            limiter_type, forbidden_value = forbidden
            raise AccessDeniedError(consumer, limiter_type, forbidden_value)

    def check_scope_specific_grants(self, scope, grant_type, client_id, ip):
        consumer = '%s/%s' % (scope, grant_type)
        if self._is_consumer_unknown(consumer):
            return

        allowed_by_ip = self._is_ip_allowed(
            consumer,
            ip,
            allow_unknown_consumers=True,
        )
        if not allowed_by_ip:
            raise AccessDeniedError(consumer, 'ip', ip)

        client_allowed = self._is_client_allowed(consumer, ip, client_id)
        if not client_allowed:
            raise AccessDeniedError(consumer, 'client_id', client_id)


def get_grants():
    grants = LazyLoader.get_instance('Grants')
    grants.load()
    return grants
