# -*- coding: utf-8 -*-
from django.conf import settings
from passport.backend.core.lazy_loader import (
    lazy_loadable,
    LazyLoader,
)
from passport.backend.oauth.core.db.config.base import BaseJsonConfig


class BaseTranslationsConfig(BaseJsonConfig):
    def translate(self, key, language, default):
        if language not in self.data:
            raise ValueError('Unknown language: %s' % language)
        return self.data[language].get(key) or default


@lazy_loadable('Scope translations')
class ScopeTranslationsConfig(BaseTranslationsConfig):
    def __init__(self):
        super(ScopeTranslationsConfig, self).__init__(filename=settings.SCOPE_TRANSLATIONS_CONFIG)


@lazy_loadable('Scope short translations')
class ScopeShortTranslationsConfig(BaseTranslationsConfig):
    def __init__(self):
        super(ScopeShortTranslationsConfig, self).__init__(filename=settings.SCOPE_SHORT_TRANSLATIONS_CONFIG)


@lazy_loadable('Service translations')
class ServiceTranslationsConfig(BaseTranslationsConfig):
    def __init__(self):
        super(ServiceTranslationsConfig, self).__init__(filename=settings.SERVICE_TRANSLATIONS_CONFIG)


def get_scope_translations():
    return LazyLoader.get_instance('Scope translations')


def get_scope_short_translations():
    return LazyLoader.get_instance('Scope short translations')


def get_service_translations():
    return LazyLoader.get_instance('Service translations')


def localize_scope(key, language, default, short=False):
    translations = get_scope_short_translations() if short else get_scope_translations()
    return translations.translate(key, language, default)


def localize_service(key, language):
    return get_service_translations().translate(key, language, key)
