# -*- coding: utf-8 -*-
from collections import namedtuple
from typing import Dict

from django.conf import settings
from passport.backend.oauth.core.db.config import get_device_name


AppPlatform = namedtuple(
    'AppPlatform',
    ['Unknown', 'Android', 'iOS'],
)._make(range(3))


def parse_app_platform(app_platform_raw):
    if app_platform_raw:
        if isinstance(app_platform_raw, int):
            # защита от двойного парсинга
            return app_platform_raw
        app_platform_raw = app_platform_raw.strip().lower()
        if app_platform_raw.startswith('android'):
            return AppPlatform.Android
        elif app_platform_raw in ('iphone', 'ipad', 'ipod', 'apple', 'ios'):
            return AppPlatform.iOS
    return AppPlatform.Unknown


def app_platform_to_name(app_platform):
    """Для записи в логи и выдачи наружу"""
    return {
        AppPlatform.Unknown: 'unknown',
        AppPlatform.Android: 'android',
        AppPlatform.iOS: 'ios',
    }[app_platform]


def get_device_info(request_values) -> Dict[str, str]:
    """
    Почётный зоопарк костылей для интеграции с АМ.

    Собирает из request.REQUEST словарь параметров, описывающих мобильное устройство:
      - параметры , передаваемые АМом: uuid, deviceid, app_id, app_platform, manufacturer,
          model, app_version, am_version, ifv, device_name
      - device_id, передаваемый извне или вычисляемый
      - model_name, вычисляемый по model

    Все эти параметры необязательны и могут отсутствовать.
    """
    device_info_fields = (
        'uuid',  # uuid приложения
        'deviceid',  # deviceid (0, если приложение не смогло его получить)
        'app_id',  # id приложения (package name для android, bundle id для айфона, что-то-там - для WP)
        'app_platform',  # платформа
        'manufacturer',  # вендор устройства
        'model',  # модель устройства
        'app_version',  # версия приложения
        'am_version',  # версия account manager
        'ifv',  # id for vendor, только для iOS
        'device_name',  # имя, заданное владельцем (только для iOS)
        'device_id',  # универсальный идентификатор устройства
    )
    alias_to_field_mapping = {
        'app_version_name': 'app_version',
        'am_version_name': 'am_version',
    }
    device_info = {field: value for field, value in request_values.items() if field in device_info_fields}
    for alias, field in alias_to_field_mapping.items():
        if request_values.get(alias) and not device_info.get(field):
            device_info[field] = request_values[alias]

    if not device_info.get('device_id'):
        device_info.pop('device_id', None)  # на случай пустой строки
        # Если device_id не передан, попытаемся его собрать из того, чем богаты
        FIELDS_PRIORITY_LIST = (
            'deviceid',
            'ifv',
        )
        for field in FIELDS_PRIORITY_LIST:
            if device_info.get(field) and device_info[field] not in settings.INVALID_DEVICE_IDS:
                device_info['device_id'] = device_info[field]
                break

    model = device_info.get('model')
    manufacturer = device_info.get('manufacturer')
    if model or manufacturer:
        device_info['model_name'] = get_device_name(model, manufacturer)

    device_name = device_info.get('device_name')
    if device_name:
        device_name = device_info['device_name'] = device_name.strip()

    # Старые версии АМ на android передают device_name, равный model - считаем
    # такой device_name отсутствующим
    if not device_name or device_name == model:
        device_info.pop('device_name', None)

    return device_info
