# -*- coding: utf-8 -*-
from django.utils.encoding import smart_bytes
from passport.backend.oauth.core.db.eav.errors import (
    AttributeNotFoundError,
    EAVAttributeTypesAlreadyRegisteredError,
)
from passport.backend.oauth.core.db.eav.value_serializers import (
    bool_deserializer,
    bool_serializer,
    datetime_deserializer,
    datetime_serializer,
    encrypted_string_deserializer,
    encrypted_string_serializer,
    hash_serializer,
    int_deserializer,
    make_collection_deserializer,
    make_collection_serializer,
    string_deserializer,
    string_serializer,
    trivial,
)


class Attribute(object):
    def __init__(self, name, attr_serializer, attr_deserializer, index_serializer, default_value=None):
        """
        * name - имя атрибута. Должно содержаться в ATTRIBUTE_TYPES для нужной сущности.
        * attr_serializer - функция, превращающая значение атрибута в строку,
        пригодную для записи в БД (в таблицу атрибутов).
        * attr_deserializer - функция, превращающая значение, прочитанное из БД, в значение атрибута.
        * index_serializer - функция, превращающая значение атрибута в строку,
        пригодную для записи в БД (в индексную таблицу).
        * default_value - значение, которое будет выставлено на модель, если атрибут отсутствует в БД.
        И наоборот, атрибуты, имеющие такое значение, не будут писаться в БД.
        """
        self.name = name
        self.attr_serializer = attr_serializer
        self.attr_deserializer = attr_deserializer
        self.index_serializer = index_serializer
        self.default_value = default_value


class StringAttribute(Attribute):
    def __init__(self, name):
        super(StringAttribute, self).__init__(
            name=name,
            attr_serializer=string_serializer,
            attr_deserializer=string_deserializer,
            index_serializer=string_serializer,
            default_value='',
        )


class HashedStringAttribute(Attribute):
    def __init__(self, name):
        super(HashedStringAttribute, self).__init__(
            name=name,
            attr_serializer=string_serializer,
            attr_deserializer=string_deserializer,
            index_serializer=hash_serializer,
            default_value='',
        )


class EncryptedStringAttribute(Attribute):
    def __init__(self, name):
        super(EncryptedStringAttribute, self).__init__(
            name=name,
            attr_serializer=encrypted_string_serializer,
            attr_deserializer=encrypted_string_deserializer,
            index_serializer=None,  # пока не будем разрешать строить индекс по шифрованному атрибуту
            default_value='',
        )


class BooleanAttribute(Attribute):
    def __init__(self, name):
        super(BooleanAttribute, self).__init__(
            name=name,
            attr_serializer=bool_serializer,
            attr_deserializer=bool_deserializer,
            index_serializer=bool_serializer,
            default_value=False,
        )


class IntegerAttribute(Attribute):
    def __init__(self, name, default_value=None):
        super(IntegerAttribute, self).__init__(
            name=name,
            attr_serializer=string_serializer,
            attr_deserializer=int_deserializer,
            index_serializer=trivial,
            default_value=default_value,
        )


class DatetimeAttribute(Attribute):
    def __init__(self, name):
        super(DatetimeAttribute, self).__init__(
            name=name,
            attr_serializer=datetime_serializer,
            attr_deserializer=datetime_deserializer,
            index_serializer=trivial,
        )


class CollectionAttribute(Attribute):
    def __init__(self, name, element_type, preserve_order=False):
        super(CollectionAttribute, self).__init__(
            name=name,
            attr_serializer=make_collection_serializer(preserve_order=preserve_order),
            attr_deserializer=make_collection_deserializer(element_type),
            index_serializer=make_collection_serializer(preserve_order=preserve_order),
            default_value=(),
        )


ATTRIBUTE_TYPES = {}  # заполняется через register_attribute_types

VIRTUAL_ATTR_ENTITY_ID = 0


ATTRIBUTES_BY_NAME = dict(
    (
        entity,
        {
            attribute.name: (attr_type, attribute)
            for attr_type, attribute in ATTRIBUTE_TYPES[entity].items()
        },
    )
    for entity in ATTRIBUTE_TYPES
)


def register_attribute_types(name, attributes):
    if name in ATTRIBUTE_TYPES:
        raise EAVAttributeTypesAlreadyRegisteredError()  # noqa
    ATTRIBUTE_TYPES[name] = attributes
    ATTRIBUTES_BY_NAME[name] = {
        attribute.name: (attr_type, attribute)
        for attr_type, attribute in attributes.items()
    }


def attr_by_name(entity, attr_name):
    try:
        return ATTRIBUTES_BY_NAME[entity][attr_name]
    except KeyError:
        raise AttributeNotFoundError('Entity "%s" doesn\'t have attribute "%s"' % (entity, attr_name))


def attr_by_type(entity, attr_type):
    try:
        return ATTRIBUTE_TYPES[entity][attr_type]
    except KeyError:
        raise AttributeNotFoundError('Unknown entity ("%s") or attr_type ("%s")' % (entity, attr_type))


def serialize_attribute(entity, entity_id, attr_name, value):
    attr_type, attribute = attr_by_name(entity, attr_name)
    serialized_value = attribute.attr_serializer(value, parent_name=entity, parent_id=entity_id)
    return attr_type, smart_bytes(serialized_value)


def deserialize_attribute(entity, entity_id, attr_type, serialized_value):
    attribute = attr_by_type(entity, attr_type)
    value = attribute.attr_deserializer(
        serialized_value,
        parent_name=entity,
        parent_id=entity_id,
        attr_name=attribute.name,
    )
    return attribute.name, value


def serialize_attribute_for_index(entity, entity_id, attr_name, value):
    attr_type, attribute = attr_by_name(entity, attr_name)
    serialized_value = attribute.index_serializer(value, parent_name=entity, parent_id=entity_id)
    return attribute.name, serialized_value
