# -*- coding: utf-8 -*-
from passport.backend.oauth.core.db.eav.attributes import serialize_attribute_for_index
from sqlalchemy import (
    and_,
    or_,
)


def as_list(value):
    if isinstance(value, (list, tuple, set)):
        return list(value)
    return [value]


class Index(object):
    def __init__(self, index_table, distinct_matched_key_fields=None,
                 substr_matched_key_fields=None, collection_fields=None,
                 nullable_fields=None, synthetic_fields=None, extra_fields_generator=None):
        """
        Индекс над таблицей, позволяющий выполнять запросы на выборку.
        index_table: таблица индекса
        distinct_matched_key_fields: поля, по которым будет производиться поиск по точному совпадению (по
          умолчанию - все поля, кроме id)
        substr_matched_key_fields: строковые поля, по которым будет производиться поиск по подстроке
        collection_fields: поля, содержащие множества элементов (сериализованные как
          "|element1|element2|element3|". Поиск будет производиться по непустоте пересечения множеств.
        nullable_fields: если все поля из этого списка пусты, то запись в индекс производиться не будет
          (при этом ошибки не возникнет)
        synthetic_fields - вычисляемые поля. Мы можем по ним искать, но на модели их нет.
        extra_fields_generator - функция, принимающая словарь полей объекта и
          возвращающая словарь дополнительных полей (уже сериализованных!) для записи в индексную таблицу
        """
        self._table = index_table
        self._distinct_matched_key_fields = distinct_matched_key_fields or [
            column.name for column in index_table.columns if column.name != 'id'
        ]
        self.synthetic_fields = synthetic_fields or []
        self._substr_matched_key_fields = substr_matched_key_fields or []
        self._collection_fields = collection_fields or []
        self._nullable_fields = nullable_fields or []
        self._all_key_fields = (
            self._distinct_matched_key_fields +
            self._substr_matched_key_fields +
            self._collection_fields
        )
        if len(set(self._all_key_fields)) < len(self._all_key_fields):
            raise ValueError('Field sets must not overlap: %s' % self._all_key_fields)
        self.extra_fields_generator = extra_fields_generator

    @property
    def table(self):
        return self._table

    @property
    def key_fields(self):
        return set(self._all_key_fields) - set(self.synthetic_fields)

    def make_key_values_for_search(self, entity_name, entity_id, all_values):
        """Отдаёт набор полей, по которым объект можно найти в индексе"""
        if (
            self._nullable_fields and
            not any(all_values.get(field_name) for field_name in self._nullable_fields)
        ):
            # все nullable-поля пусты - значит, этого объекта в этом индексе нет
            return
        return dict(
            serialize_attribute_for_index(entity_name, entity_id, field_name, all_values.get(field_name))
            for field_name in self.key_fields
        )

    def make_all_key_values(self, entity_name, entity_id, all_values):
        """Отдаёт набор полей для записи в индекс"""
        result = self.make_key_values_for_search(entity_name, entity_id, all_values)
        if result is None:
            # все nullable-поля пусты - значит, этот объект в этот индекс записывать не нужно
            return
        if self.extra_fields_generator is not None:
            result.update(self.extra_fields_generator(all_values))
        return result

    def make_clause(self, values):
        """
        Формирует условие запроса по индексу.
        values: словарь field_name -> value_list
        """
        distinct_matched_comparisons = and_(
            self._table.c[field].in_(as_list(values[field]))
            for field in self._distinct_matched_key_fields
            if values.get(field) is not None
        )
        substr_matched_comparisons = and_(
            or_(
                self._table.c[field].contains(element) for element in as_list(values[field])
            )
            for field in self._substr_matched_key_fields
            if values.get(field) is not None
        )
        collection_comparisons = and_(
            or_(
                self._table.c[field].contains('|%s|' % element) for element in as_list(values[field])
            )
            for field in self._collection_fields
            if values.get(field) is not None
        )

        return and_(
            distinct_matched_comparisons,
            substr_matched_comparisons,
            collection_comparisons,
        )
