# -*- coding: utf-8 -*-
from copy import deepcopy
import logging

from passport.backend.oauth.core.common.utils import classproperty
from passport.backend.oauth.core.db.eav.attributes import attr_by_name


log = logging.getLogger('db.eav')


class EavAttr(object):
    """Eav-атрибут модели"""
    def __init__(self, attr_name=None):
        self._name = attr_name

    def _encode_attr_value(self, value, instance):
        """Препроцессинг перед записью в БД"""
        return value

    def _decode_attr_value(self, value, instance):
        """Постпроцессинг после записи в БД"""
        return value

    def __get__(self, instance, owner):
        _, attr = attr_by_name(instance.entity_name, self._name)
        raw_value = instance._attributes.get(attr.name)
        return self._decode_attr_value(raw_value, instance) if raw_value is not None else attr.default_value

    def __set__(self, instance, value):
        _, attr = attr_by_name(instance.entity_name, self._name)
        if value is not None and value != attr.default_value:
            instance._attributes[attr.name] = self._encode_attr_value(value, instance)
        else:
            if attr.name in instance._attributes:
                del instance._attributes[attr.name]
        return value

    def __delete__(self, instance):
        _, attr = attr_by_name(instance.entity_name, self._name)
        if attr.name in instance._attributes:
            del instance._attributes[attr.name]


class EavModelMeta(type):
    def __new__(cls, name, bases, attrs):
        for attr_name in attrs.keys():
            attr = attrs[attr_name]
            if isinstance(attr, EavAttr) and attr._name is None:
                attr._name = attr_name

        return type.__new__(cls, name, bases, attrs)


class BaseEavModel(object, metaclass=EavModelMeta):
    """Базовый класс eav-модели. Содержит методы работы с атрибутами модели.
    Методы сериализации и десериализации должны быть определены в наследнике."""
    _entity_name = None

    def __init__(self, entity_id=None, **kwargs):
        self._id = entity_id
        self._attributes = {}
        for attr, value in kwargs.items():
            EavAttr(attr).__set__(self, value)

    def __repr__(self):
        return '<%s id=%d>: %s' % (self.__class__.__name__, self.id, self._attributes)

    def __eq__(self, other):
        return (
            self.id == other.id and
            self.entity_name == other.entity_name and
            self._attributes == other._attributes
        )

    def __ne__(self, other):
        return not self == other

    def __hash__(self):
        return self.id

    @classproperty
    def entity_name(cls):
        return cls._entity_name or cls.__name__.lower()

    @property
    def id(self):
        return self._id

    def snapshot(self):
        return deepcopy(self)

    @classmethod
    def by_id(cls, entity_id):
        raise NotImplementedError()  # pragma: no cover

    @classmethod
    def by_index(cls, index_name, limit=None, offset=None, **kwargs):
        raise NotImplementedError()  # pragma: no cover

    def serialize(self, old):
        raise NotImplementedError()  # pragma: no cover

    def delete(self):
        raise NotImplementedError()  # pragma: no cover
