# -*- coding: utf-8 -*-
from django.conf import settings
from passport.backend.oauth.core.db.eav.dbmanager import get_dbm
from passport.backend.oauth.core.db.eav.sharder import get_db_name


class BaseExecutable(object):
    """Базовый класс для всего, что можно исполнить в БД: запросов и транзакций"""
    def __init__(self):
        self.timeout = None
        self.retries = None
        self.force_master = False

    @property
    def dbm(self):
        """DBManager, на котором будет исполняться запрос/транзакция"""
        raise NotImplementedError()  # pragma: no cover

    def _execute(self):
        raise NotImplementedError()  # pragma: no cover

    def execute(self, retries=None, force_master=False):
        """
        Выполняет запрос/транзакцию, возвращает результаты исполнения.
        retries - если указано, то подменяет глобальную настройки числа ретраев.
        force_master - выполнить запрос в мастере, даже если его можно исполнить на слейве
        """
        self.retries = retries or self.retries
        self.force_master = force_master or self.force_master
        return self._execute()

    @property
    def is_selectable(self):
        """Является ли запрос read-only"""
        raise NotImplementedError()  # pragma: no cover


class BaseQuery(BaseExecutable):
    """Базовый класс запроса"""
    def __init__(self, table):
        super(BaseQuery, self).__init__()
        self._table = table
        self._values = {}

    def to_sql(self):
        """Возвращает SQLалхимийный запрос"""
        raise NotImplementedError()  # pragma: no cover

    @property
    def is_selectable(self):
        return self.to_sql().is_selectable

    @property
    def values(self):
        return self._values

    def _execute(self, timeout=None, retries=None):
        # Use transactions, Luke
        return self.dbm.execute(self)

    def __repr__(self):
        return '%s  <=  %s' % (self.to_sql(), self.values)


class Query(BaseQuery):
    """Базовый класс запроса в указанную БД"""
    def __init__(self, table, db_name):
        super(Query, self).__init__(table=table)
        self._db_name = db_name

    @property
    def dbm(self):
        return get_dbm(self._db_name)


class CentralQuery(Query):
    """Базовый класс запроса в центральную БД"""
    def __init__(self, table):
        super(CentralQuery, self).__init__(table=table, db_name=settings.CENTRAL_DB_NAME)


class AutoShardedQuery(BaseQuery):
    """
    Базовый класс запроса для работы с eav-сущностями.
    Имя базы автоматически определяется по id сущности
    """
    def __init__(self, table, entity_name, entity_id):
        super(AutoShardedQuery, self).__init__(table=table)
        self._entity_name = entity_name
        self._entity_id = entity_id

    @property
    def dbm(self):
        return get_dbm(get_db_name(self._entity_name, self._entity_id))
