# -*- coding: utf-8 -*-
from bisect import bisect
from collections import defaultdict

from django.conf import settings


def build_range_shard_function(ranges):
    """ ranges: [(shard, begin_interval), ...] """
    def range_shard_function(key):
        position = bisect(beginning_intervals, key)
        if position == 0:
            raise ValueError('Invalid key value: %s' % key)
        return shards[position - 1]

    ranges = sorted(ranges, key=lambda r: r[1])
    shards = [r[0] for r in ranges]
    beginning_intervals = [r[1] for r in ranges]
    return range_shard_function


class _Sharder(object):
    def configure(self, buckets_to_db_names, shard_function):
        self.shard_function = shard_function
        self.buckets_to_db_names = buckets_to_db_names

    def key_to_db_name(self, key):
        bucket_number = self.shard_function(key)
        return self.buckets_to_db_names[bucket_number]


_sharders = defaultdict(_Sharder)


def get_sharder(entity_name):
    return _sharders[entity_name]


def is_sharded(entity_name):
    return entity_name in _sharders.keys()


def get_db_name(entity_name, key):
    if is_sharded(entity_name):
        return get_sharder(entity_name).key_to_db_name(key)
    else:
        return settings.CENTRAL_DB_NAME
