# -*- coding: utf-8 -*-
import logging

from django.conf import settings
from django.utils.encoding import (
    smart_bytes,
    smart_text,
)
from passport.backend.oauth.core.common.crypto import (
    decrypt,
    encrypt,
    UnsupportedVersionError,
)
from passport.backend.oauth.core.common.utils import (
    int_to_bytes,
    make_hash,
)
from passport.backend.utils.time import (
    datetime_to_integer_unixtime,
    unixtime_to_datetime,
)


log = logging.getLogger('db.eav')


ID_MAX_BYTE_COUNT = 8


def trivial(value, **kwargs):
    return value


def string_serializer(value, **kwargs):
    return smart_bytes(value)


def string_deserializer(serialized_value, **kwargs):
    return serialized_value.decode()


def hash_serializer(value, **kwargs):
    return make_hash(value)


def int_deserializer(serialized_value, **kwargs):
    return int(serialized_value)


def make_collection_serializer(preserve_order=False):
    def collection_serializer(items, **kwargs):
        items = items or []
        return b'|%s|' % b'|'.join(map(
            smart_bytes,
            items if preserve_order else sorted(items),
        ))
    return collection_serializer


def make_collection_deserializer(element_type):
    def collection_deserializer(serialized_items, **kwargs):
        if serialized_items == b'||':
            return []
        return list(map(
            element_type,
            serialized_items[1:-1].split(b'|'),
        ))
    return collection_deserializer


def bool_serializer(value, **kwargs):
    return 1 if value else 0


def bool_deserializer(serialized_value, **kwargs):
    return bool(int(serialized_value))


def datetime_serializer(value, **kwargs):
    return datetime_to_integer_unixtime(value)


def datetime_deserializer(value, **kwargs):
    return unixtime_to_datetime(float(value))  # раньше в БД писался float


def encrypted_string_serializer(value, parent_id=None, **kwargs):
    value = smart_bytes(value) + int_to_bytes(parent_id, ID_MAX_BYTE_COUNT)
    return encrypt(settings.ATTRIBUTE_CIPHER_KEYS, value, version=2)


def encrypted_string_deserializer(serialized_value, parent_id=None, parent_name=None, attr_name=None, **kwargs):
    try:
        version, decoded_value = decrypt(settings.ATTRIBUTE_CIPHER_KEYS, serialized_value)
        if version == 1:
            return decoded_value.decode()
        else:
            if int_to_bytes(parent_id, ID_MAX_BYTE_COUNT) != decoded_value[-ID_MAX_BYTE_COUNT:]:
                log.error('Corrupted attr %s.%s (entity_id=%s)' % (
                    parent_name,
                    attr_name,
                    parent_id,
                ))
            return decoded_value[:-ID_MAX_BYTE_COUNT].decode()
    except UnsupportedVersionError:
        log.warning('Unable to decrypt %s.%s (entity_id=%s). Using it as unencrypted' % (
            parent_name,
            attr_name,
            parent_id,
        ))
        return smart_text(serialized_value)
