# -*- coding: utf-8 -*-
import logging
from typing import Iterable

from django.conf import settings
from passport.backend.core.builders.blackbox.utils import get_attribute
from passport.backend.core.portallib.portallib import (
    is_yandex_ip,
    is_yandex_server_ip,
)
from passport.backend.oauth.core.db.client import Client
from passport.backend.oauth.core.db.config.grants_config import (
    AccessDeniedError,
    get_grants,
    GrantsMissingError,
    TVMTicketInvalidError,
)
from passport.backend.oauth.core.db.scope import Scope
from passport.backend.oauth.core.logs.statbox import (
    StatboxLogger,
    to_statbox,
)


log = logging.getLogger('limits')


def assert_is_not_spammer(bb_response, grant_type: str, client: Client, ip: str):
    karma = bb_response.get('karma')
    is_shared = get_attribute(bb_response, settings.BB_ATTR_IS_SHARED)
    if is_shared:
        return
    if client.is_yandex:
        return
    # https://wiki.yandex-team.ru/passport/karma/
    # Ограничиваем аккаунты с постфиксом 100, не имеющие "хороших" префиксов
    if karma is not None and karma // 1000 not in [2, 4, 6] and karma % 1000 == 100:
        to_statbox(
            mode='issue_token',
            status='error',
            reason='limited_by.karma',
            grant_type=grant_type,
            client_id=client.display_id,
            user_ip=ip,
            uid=bb_response['uid'],
        )
        raise AccessDeniedError('', 'karma', karma)


def assert_is_not_child(bb_response, grant_type: str, client: Client, ip: str):
    is_child = get_attribute(bb_response, settings.BB_ATTR_IS_CHILD)
    if not is_child:
        return
    if client.is_yandex:
        return
    to_statbox(
        mode='issue_token',
        status='error',
        reason='limited_by.account_type',
        grant_type=grant_type,
        client_id=client.display_id,
        user_ip=ip,
        uid=bb_response['uid'],
    )
    raise AccessDeniedError('', 'account_type', 'child')


def assert_is_allowed(scope_list: Iterable[Scope], grant_type: str, client_id: str, ip: str, uid: int):
    """Проверка грантов при выдаче токена"""

    try:
        # ни при каких условиях не даём выдавать токены с пустым скоупом
        if not scope_list:
            raise AccessDeniedError(scope='', limiter_type='scopes', forbidden_value='')
        # проверяем ограничения на скоупы
        for scope in scope_list:
            get_grants().check_scope_grants(scope.keyword, grant_type, client_id, ip)
            # Поддержка грантов на отдельные grant_type для скоупа
            get_grants().check_scope_specific_grants(scope.keyword, grant_type, client_id, ip)

        # проверяем ограничения на grant_type
        get_grants().check_scope_grants('grant_type:%s' % grant_type, grant_type, client_id, ip)
    except AccessDeniedError as e:
        to_statbox(
            mode='issue_token',
            status='error',
            reason='limited_by.%s' % e.limiter_type,
            grant_type=grant_type,
            client_id=client_id,
            user_ip=ip,
            uid=uid,
        )
        log.debug(e)
        raise


def has_grant(grant, consumer, ip, service_ticket):
    return get_grants().has_grant(grant=grant, consumer=consumer, ip=ip, service_ticket=service_ticket)


def check_grant(grant, consumer, ip, service_ticket):
    try:
        get_grants().check_grant(grant=grant, consumer=consumer, ip=ip, service_ticket=service_ticket)
    except TVMTicketInvalidError as e:
        log.debug(e)
        raise
    except GrantsMissingError as e:
        to_statbox(
            mode='check_grants',
            status='error',
            grant=e.grant,
            consumer=e.consumer,
            user_ip=e.ip,
        )
        log.debug(e)
        raise


def check_grants(grants, consumer, ip, service_ticket):
    for grant in grants:
        check_grant(grant=grant, consumer=consumer, ip=ip, service_ticket=service_ticket)


def restrict_non_yandex_clients(grant_type, client, user_ip):
    """
    Почётный временно-постоянный костыль для закрытия от внешнего мира
    непубличной части апи по выдаче токенов
    """
    # TODO: оторвать лишние логи и настройки после полного включения ограничений
    _is_yandex_ip = is_yandex_ip(user_ip) or is_yandex_server_ip(user_ip)
    statbox = StatboxLogger(
        mode='issue_token_by_nonpublic_grant_type',
        grant_type=grant_type,
        client_id=client.display_id,
        user_ip=user_ip,
        is_yandex_ip=_is_yandex_ip,
    )

    if (
        grant_type not in settings.NONPUBLIC_GRANT_TYPES or
        not settings.FORBID_NONPUBLIC_GRANT_TYPES
    ):
        # нам сюда не нужно, даже логировать не будем
        return
    elif _is_yandex_ip and client.display_id in settings.ALLOW_NONPUBLIC_GRANT_TYPES_FOR_CLIENTS_FROM_YANDEX_IPS.get(
        grant_type, [],
    ):
        allowed = True
        reason = 'whitelisted'
    elif not _is_yandex_ip and client.display_id in settings.ALLOW_NONPUBLIC_GRANT_TYPES_FOR_CLIENTS_FROM_INTERNET.get(
        grant_type, [],
    ):
        allowed = True
        reason = 'whitelisted'
    elif client.is_yandex:
        if client.allow_nonpublic_granttypes:
            allowed = True
            reason = 'is_yandex_and_allowed'
        else:
            allowed = False
            reason = 'is_yandex_but_not_allowed'
    else:
        allowed = False
        reason = 'other'

    statbox.log(
        action='allowed' if allowed else 'forbidden',
        reason=reason,
    )
    if not allowed:
        error = AccessDeniedError(
            scope=client.scopes,
            limiter_type='nonpublic_grant_type',
            forbidden_value=grant_type,
        )
        log.debug(error)
        raise error


def check_substitute_user_ip_grant(ip):
    """
    Функция определяет по ip источника запроса, есть ли у данного источника возможность передавать
    пользовательский ip в специальном заголовке
    """
    # TODO: перейти на аналогичную функцию из core после обновления core
    # (нужно будет отказаться от переопределения конструктора GrantsConfig)
    return get_grants().is_valid_ip(ip, settings.SUBSTITUTE_USER_IP_CONSUMER)
