# -*- coding: utf-8 -*-
import hashlib
import hmac

from cryptography.hazmat.backends import default_backend
from cryptography.hazmat.primitives.ciphers import (
    algorithms,
    Cipher,
    modes,
)
from django.conf import settings
from django.utils.encoding import smart_bytes
from passport.backend.oauth.core.common.utils import (
    bytes_to_int,
    from_base64_url,
    hex_to_bytes,
    int_to_bytes,
    to_base64_url,
)
from passport.backend.oauth.core.db.client import Client
from passport.backend.oauth.core.db.eav import EntityNotFoundError
from passport.backend.oauth.core.db.errors import PsuidInvalidError


def make_psuid(uid, client, seed=0, version=None):
    """
    PSUID - publisher-specific user id
    """
    version = version or settings.PSUID_DEFAULT_VERSION
    encryption_key = smart_bytes(settings.PSUID_ENCRYPTION_KEYS[version])
    signature_key = smart_bytes(settings.PSUID_SIGNATURE_KEYS[version])

    data_to_encrypt = int_to_bytes(uid, 8) + int_to_bytes(seed, 8)
    iv = hex_to_bytes(client.display_id)  # 16 байт - именно такого размера должен быть iv
    cipher = Cipher(algorithms.AES(encryption_key), modes.CBC(iv), backend=default_backend())
    encryptor = cipher.encryptor()
    encrypted_data = encryptor.update(data_to_encrypt) + encryptor.finalize()

    # тут и ниже используем client.id вместо client.display_id, чтобы сэкономить 12 байт
    data_to_sign = int_to_bytes(client.id, 4) + data_to_encrypt
    signature = hmac.new(key=signature_key, msg=data_to_sign, digestmod=hashlib.md5).digest()

    psuid = b'.'.join([
        str(version).encode(),
        to_base64_url(int_to_bytes(client.id, 4)),
        to_base64_url(encrypted_data),
        to_base64_url(signature),
    ])
    return psuid.decode()


def parse_psuid(psuid, allow_deleted_client=False):
    parts = smart_bytes(psuid).split(b'.')
    if len(parts) != 4:
        raise PsuidInvalidError('Not enough parts')

    try:
        version = int(parts[0])
        client_id = bytes_to_int(from_base64_url(parts[1]))
        encrypted_data = from_base64_url(parts[2])
        signature = from_base64_url(parts[3])
    except (ValueError, TypeError) as e:
        raise PsuidInvalidError('Failed to parse: %s' % e)

    if version not in settings.PSUID_ENCRYPTION_KEYS or version not in settings.PSUID_SIGNATURE_KEYS:
        raise PsuidInvalidError('Unknown version: %r' % version)
    encryption_key = smart_bytes(settings.PSUID_ENCRYPTION_KEYS[version])
    signature_key = smart_bytes(settings.PSUID_SIGNATURE_KEYS[version])

    try:
        client = Client.by_id(client_id, allow_deleted=allow_deleted_client)
    except EntityNotFoundError:
        raise PsuidInvalidError('Client %s not found' % client_id)

    iv = hex_to_bytes(client.display_id)
    cipher = Cipher(algorithms.AES(encryption_key), modes.CBC(iv), backend=default_backend())
    decryptor = cipher.decryptor()
    try:
        decrypted_data = decryptor.update(encrypted_data) + decryptor.finalize()
    except ValueError as e:
        raise PsuidInvalidError('Failed to decrypt: %s' % e)

    uid = bytes_to_int(decrypted_data[:8])
    seed = bytes_to_int(decrypted_data[8:])

    data_to_sign = int_to_bytes(client.id, 4) + decrypted_data
    expected_signature = hmac.new(key=signature_key, msg=data_to_sign, digestmod=hashlib.md5).digest()
    if not hmac.compare_digest(signature, expected_signature):
        raise PsuidInvalidError('Signature mismatch')

    return uid, seed, client
