# -*- coding: utf-8 -*-
import base64
from collections import namedtuple
from datetime import timedelta
import logging
import os
from random import randint

from django.conf import settings
from django.utils.encoding import smart_bytes
from passport.backend.core.utils.crc import (
    bytes_to_long,
    crc5_epc,
)
from passport.backend.core.utils.pkce import (
    CODE_CHALLENGE_METHOD_PLAIN,
    CODE_CHALLENGE_METHOD_SHA256,
    is_pkce_valid,
)
from passport.backend.oauth.core.common.decorators import retry
from passport.backend.oauth.core.common.utils import (
    first_or_none,
    now,
    random_stuff,
)
from passport.backend.oauth.core.db.eav import (
    BaseDBError,
    CREATE,
    EavAttr,
    EavModel,
    Index,
    UPDATE,
)
from passport.backend.oauth.core.db.errors import (
    ExpiredRequestError,
    PaymentAuthNotPassedError,
    VerificationCodeCollisionError,
    WrongRequestUserError,
)
from passport.backend.oauth.core.db.mixins.client_mixin import ClientMixin
from passport.backend.oauth.core.db.mixins.scopes_mixin import ScopesMixin
from passport.backend.oauth.core.db.mixins.ttl_mixin import TtlMixin
from passport.backend.oauth.core.db.schemas import (
    request_attributes_table,
    request_by_code_table,
    request_by_display_id_table,
    request_by_unique_code_table,
)
from passport.backend.oauth.core.db.scope import is_payment_auth_required


log = logging.getLogger('db.request')


REQUEST_CODE_GENERATION_RETRIES = 3  # TODO: попробовать вернуть в сеттинги

CRC_LENGTH = 2  # # crc5 вернёт число от 0 до 2 ** 5 - 1, оно заведомо поместится в две десятичные цифры


ActivationStatus = namedtuple(
    'ActivationStatus',
    ['NotRequired', 'Pending', 'Activated'],
)._make(range(3))


CodeAlphabet = namedtuple(
    'CodeAlphabet',
    ['Digits', 'Base32'],
)._make(range(2))


CodeStrength = namedtuple(
    'CodeStrength',
    ['Basic', 'Medium', 'Long', 'BelowMedium', 'BelowMediumWithCRC', 'MediumWithCRC'],
)._make(range(6))


CodeType = namedtuple(
    'CodeType',
    ['ClientBound', 'Unique'],
)._make(range(2))


CodeChallengeMethod = namedtuple(
    'CodeChallengeMethod',
    ['NotRequired', 'Plain', 'S256'],
)._make(range(3))


def code_strength_to_name(code_strength):
    """Для записи в логи"""
    return {
        0: 'basic',
        1: 'medium',
        2: 'long',
        3: 'below_medium',
        4: 'below_medium_with_crc',
        5: 'medium_with_crc',
    }[code_strength]


def code_challenge_method_to_name(code_challenge_method):
    """Для записи в логи"""
    return {
        0: None,
        1: 'plain',
        2: 'S256',
    }[code_challenge_method]


def _make_crc(message, alphabet):
    message = message.encode()
    crc_int = crc5_epc(bytes_to_long(message))
    if alphabet == CodeAlphabet.Digits:
        return '%02d' % crc_int
    elif alphabet == CodeAlphabet.Base32:
        # немного магии: выравниваем наши 5 бит так, чтобы они попали в первый символ base32
        padded = smart_bytes(chr(crc_int << 3))
        return base64.b32encode(padded)[:1].decode()
    else:
        raise NotImplementedError('Unknown alphabet: %s' % alphabet)  # pragma: no cover


def _generate_code(alphabet, length, with_crc=False):
    assert alphabet in {CodeAlphabet.Digits, CodeAlphabet.Base32}

    if alphabet == CodeAlphabet.Digits:
        random_part = str(randint(10 ** (length - 1), 10 ** length))
    elif alphabet == CodeAlphabet.Base32:
        random_part = base64.b32encode(os.urandom(length)).lower()[:length].decode()
    else:
        raise NotImplementedError('Unknown alphabet: %s' % alphabet)  # pragma: no cover

    if with_crc:
        return random_part + _make_crc(random_part, alphabet)
    else:
        return random_part


def generate_code(strength):
    if strength == CodeStrength.BelowMedium:
        return _generate_code(CodeAlphabet.Digits, settings.REQUEST_CODE_BELOW_MEDIUM_LENGTH)
    elif strength == CodeStrength.BelowMediumWithCRC:
        return _generate_code(CodeAlphabet.Digits, settings.REQUEST_CODE_BELOW_MEDIUM_LENGTH, with_crc=True)
    elif strength == CodeStrength.Medium:
        return _generate_code(CodeAlphabet.Base32, settings.REQUEST_CODE_MEDIUM_LENGTH)
    elif strength == CodeStrength.MediumWithCRC:
        return _generate_code(CodeAlphabet.Base32, settings.REQUEST_CODE_MEDIUM_LENGTH, with_crc=True)
    elif strength == CodeStrength.Long:
        return _generate_code(CodeAlphabet.Base32, settings.REQUEST_CODE_HIGH_LENGTH)
    else:
        return _generate_code(CodeAlphabet.Digits, settings.REQUEST_CODE_LOW_LENGTH)


def does_code_look_valid(code):
    known_lengths = {
        settings.REQUEST_CODE_LOW_LENGTH,
        settings.REQUEST_CODE_BELOW_MEDIUM_LENGTH,
        settings.REQUEST_CODE_HIGH_LENGTH,
    }

    if len(code) in known_lengths:
        return True

    for crc_len, crc_alphabet in (
        (1, CodeAlphabet.Base32),
        (2, CodeAlphabet.Digits),
    ):
        if len(code) - crc_len in known_lengths:
            random_part = code[:-crc_len]
            crc = code[-crc_len:]
            if crc == _make_crc(random_part, crc_alphabet):
                return True

    return False


class Request(EavModel, ClientMixin, ScopesMixin, TtlMixin):
    _table = request_attributes_table
    _indexes = {
        'display_id': Index(request_by_display_id_table),
        'client_bound_code': Index(request_by_code_table, nullable_fields=['code']),
        'unique_code': Index(request_by_unique_code_table, nullable_fields=['unique_code']),
    }

    uid = EavAttr()
    is_accepted = EavAttr()
    _client_bound_code = EavAttr('code')
    _unique_code = EavAttr('unique_code')
    redirect_uri = EavAttr()
    is_token_response = EavAttr()
    state = EavAttr()
    display_id = EavAttr()
    device_id = EavAttr()
    device_name = EavAttr()
    created = EavAttr()
    code_strength = EavAttr()
    code_type = EavAttr()
    activation_status = EavAttr()
    code_challenge = EavAttr()
    code_challenge_method = EavAttr()
    payment_auth_context_id = EavAttr()
    payment_auth_scope_addendum = EavAttr()
    payment_auth_scheme = EavAttr()
    login_id = EavAttr()

    @classmethod
    def create(cls, uid, client, scopes=None, redirect_uri=None, is_token_response=False, state=None,
               ttl=None, device_id=None, device_name=None,
               code_strength=CodeStrength.Basic, code_type=CodeType.ClientBound,
               code_challenge=None, code_challenge_method=None, login_id=None):
        return cls(
            uid=uid,
            client_id=client.id,
            scope_ids=[scope.id for scope in scopes] if scopes else client.scope_ids,
            is_accepted=False,
            redirect_uri=redirect_uri or client.default_redirect_uri,
            is_token_response=is_token_response,
            state=state,
            display_id=random_stuff(),
            created=now(),
            expires=now() + timedelta(seconds=ttl or settings.REQUEST_UNACCEPTED_TTL),
            device_id=device_id,
            device_name=device_name,
            code_strength=code_strength,
            code_type=code_type,
            code_challenge=code_challenge,
            code_challenge_method=code_challenge_method,
            login_id=login_id,
        )

    @classmethod
    def by_display_id(cls, display_id):
        return first_or_none(cls.by_index('display_id', display_id=smart_bytes(display_id)))

    @classmethod
    def by_verification_code(cls, client_id, code):
        if client_id:
            return first_or_none(cls.by_index('client_bound_code', client_id=client_id, code=smart_bytes(code)))
        else:
            return first_or_none(cls.by_index('unique_code', unique_code=smart_bytes(code)))

    @property
    def code(self):
        if self.code_type == CodeType.ClientBound:
            return self._client_bound_code
        elif self.code_type == CodeType.Unique:
            return self._unique_code

    def make_code(self):
        code = generate_code(self.code_strength)
        if self.code_type == CodeType.ClientBound:
            self._client_bound_code = code
        elif self.code_type == CodeType.Unique:
            assert self.code_strength in {CodeStrength.Medium, CodeStrength.MediumWithCRC, CodeStrength.Long}
            self._unique_code = code

    def is_invalidated_by_user_logout(self, revoke_time):
        return bool(revoke_time) and self.created < revoke_time

    @property
    def needs_activation(self):
        return self.activation_status == ActivationStatus.Pending

    def check_code_verifier(self, code_verifier):
        code_challenge_method = {
            CodeChallengeMethod.Plain: CODE_CHALLENGE_METHOD_PLAIN,
            CodeChallengeMethod.S256: CODE_CHALLENGE_METHOD_SHA256,
        }.get(self.code_challenge_method)

        return is_pkce_valid(
            challenge=self.code_challenge,
            challenge_method=code_challenge_method,
            verifier=code_verifier,
        )


def get_request(display_id, uid):
    request = Request.by_display_id(display_id)
    if not request or request.is_accepted or request.is_expired:
        raise ExpiredRequestError()
    if request.uid != uid:
        raise WrongRequestUserError()
    return request


retry_request_modification = retry(
    exception_to_catch=BaseDBError,  # ретраимся и при ошибке сети, и при коллизии кодов в БД (чтобы создать другой код)
    exception_to_raise=VerificationCodeCollisionError,
    retries=REQUEST_CODE_GENERATION_RETRIES,
    logger=log,
    name_for_logger='Database',
)


@retry_request_modification
def create_request(is_accepted=False, uid=None, make_code=False, require_activation=False, **kwargs):
    """Создать реквест"""
    with CREATE(Request.create(uid=uid, **kwargs), retries=1) as request:
        if is_accepted and not (request.uid and make_code):
            raise ValueError('Unable to accept request without both uid and code')
        request.is_accepted = is_accepted
        if make_code:
            request.make_code()
        if require_activation:
            request.activation_status = ActivationStatus.Pending

    return request


@retry_request_modification
def accept_request(request, uid=None, ttl=None, scopes=None, payment_auth_scope_addendum=None):
    """Принять реквест. После этого у него должны быть заполнены и уид, и код подтверждения"""
    with UPDATE(request, retries=1):
        request.uid = request.uid or uid
        if not request.uid:
            raise ValueError('Unable to accept request without uid')
        request.is_accepted = True
        if not request.code:
            request.make_code()
        request.expires = now() + timedelta(seconds=ttl or settings.REQUEST_ACCEPTED_TTL)
        if scopes:
            if not set(scopes).issubset(request.scopes):
                raise ValueError('Scope expansion is prohibited')  # недостижимый в реальности случай, но перестрахуемся
            request.scope_ids = [scope.id for scope in scopes]

        if is_payment_auth_required(request.scopes):
            if not (request.payment_auth_context_id and payment_auth_scope_addendum):
                # Сюда IRL мы попадать не должны, валидация должна случаться раньше. Тут проверка лишь для надёжности.
                raise PaymentAuthNotPassedError()
            request.payment_auth_scope_addendum = payment_auth_scope_addendum
    return request
