# -*- coding: utf-8 -*-
from django.utils.encoding import smart_text
from passport.backend.oauth.core.db.eav.attributes import (
    BooleanAttribute,
    CollectionAttribute,
    DatetimeAttribute,
    HashedStringAttribute,
    IntegerAttribute,
    register_attribute_types,
    StringAttribute,
)
from passport.backend.oauth.core.db.eav.schemas import (
    auto_id_columns,
    central_metadata,
    eav_columns,
    register_auto_id,
    shard_metadata,
)
from passport.backend.oauth.core.db.eav.types import Event
from sqlalchemy import TIMESTAMP
from sqlalchemy.schema import (
    Column,
    Table,
)
from sqlalchemy.sql.functions import current_timestamp
from sqlalchemy.types import (
    BigInteger,
    Boolean,
    Integer,
    String,
    VARBINARY,
)


# Типы атрибутов для OAuth-моделей

register_attribute_types(
    'client',
    {
        1: IntegerAttribute('uid'),
        2: CollectionAttribute('scope_ids', int),
        3: StringAttribute('secret'),
        4: StringAttribute('old_secret'),
        5: StringAttribute('callback'),  # устарел, производится миграция на redirect_uris
        6: IntegerAttribute('approval_status'),
        7: BooleanAttribute('is_blocked'),
        8: StringAttribute('default_title'),
        9: StringAttribute('title_ru'),  # TODO: танкер
        10: StringAttribute('title_en'),
        11: StringAttribute('title_tr'),
        12: DatetimeAttribute('glogouted'),
        13: DatetimeAttribute('created'),
        14: DatetimeAttribute('modified'),
        15: StringAttribute('display_id'),
        16: CollectionAttribute('services', smart_text),
        17: CollectionAttribute('events', Event.parse),
        18: StringAttribute('default_description'),
        19: StringAttribute('icon'),
        20: StringAttribute('homepage'),
        21: StringAttribute('title_uk'),
        22: StringAttribute('description_ru'),
        23: StringAttribute('description_en'),
        24: StringAttribute('description_tr'),
        25: StringAttribute('description_uk'),
        26: BooleanAttribute('is_yandex'),
        27: StringAttribute('icon_id'),
        28: StringAttribute('telegram_bot_name'),
        29: DatetimeAttribute('deleted'),
        30: BooleanAttribute('allow_nonpublic_granttypes'),
        31: CollectionAttribute('redirect_uris', smart_text, preserve_order=True),
        32: StringAttribute('ios_default_app_id'),
        33: StringAttribute('android_default_package_name'),
        34: CollectionAttribute('android_cert_fingerprints', smart_text),
        35: StringAttribute('ios_appstore_url'),
        36: StringAttribute('android_appstore_url'),
        37: CollectionAttribute('owner_uids', int),
        38: CollectionAttribute('owner_groups', smart_text),
        39: CollectionAttribute('owner_group_uids', int),
        40: CollectionAttribute('ios_extra_app_ids', smart_text),
        41: CollectionAttribute('android_extra_package_names', smart_text),
        42: CollectionAttribute('extra_visible_scope_ids', int),
        43: StringAttribute('turboapp_base_url'),
        44: StringAttribute('contact_email'),
    },
)
register_attribute_types(
    'request',
    {
        1: IntegerAttribute('uid'),
        2: IntegerAttribute('client_id'),
        3: CollectionAttribute('scope_ids', int),
        4: BooleanAttribute('is_accepted'),
        5: StringAttribute('code'),
        6: DatetimeAttribute('expires'),
        7: StringAttribute('redirect_uri'),
        8: BooleanAttribute('is_token_response'),
        9: StringAttribute('state'),
        10: StringAttribute('display_id'),
        11: StringAttribute('device_id'),
        12: StringAttribute('device_name'),
        13: DatetimeAttribute('created'),
        14: IntegerAttribute('code_strength', default_value=0),
        15: IntegerAttribute('activation_status', default_value=0),
        16: StringAttribute('code_challenge'),
        17: IntegerAttribute('code_challenge_method', default_value=0),
        18: StringAttribute('unique_code'),
        19: IntegerAttribute('code_type', default_value=0),
        20: StringAttribute('payment_auth_context_id'),
        21: StringAttribute('payment_auth_scope_addendum'),
        22: StringAttribute('payment_auth_scheme'),
        23: StringAttribute('login_id'),
    },
)
register_attribute_types(
    'token',
    {
        1: HashedStringAttribute('access_token'),
        2: CollectionAttribute('scope_ids', int),
        3: IntegerAttribute('uid'),
        4: IntegerAttribute('client_id'),
        5: StringAttribute('device_id'),
        6: StringAttribute('device_name'),
        7: DatetimeAttribute('expires'),
        8: BooleanAttribute('is_refreshable'),
        9: DatetimeAttribute('created'),
        10: DatetimeAttribute('issued'),
        11: StringAttribute('meta'),
        12: HashedStringAttribute('alias'),
        13: StringAttribute('comment'),  # не используется
        14: IntegerAttribute('x_token_id'),
        15: StringAttribute('payment_auth_context_id'),
        16: StringAttribute('payment_auth_scope_addendum'),
        17: StringAttribute('login_id'),
        18: IntegerAttribute('app_platform', default_value=0),
        19: BooleanAttribute('is_xtoken_trusted'),
    },
)

# TODO: когда появятся ещё виртуальные атрибуты, подумать об их типизации (аналогично обычным атрибутам)
TOKEN_VIRTUAL_ATTR_IS_STATELESS = 1001


# ID для сущностей OAuth

register_auto_id('client', Table('auto_id_client', central_metadata, *auto_id_columns()))
register_auto_id('request', Table('auto_id_request', central_metadata, *auto_id_columns()))
register_auto_id('token', Table('auto_id_token', central_metadata, *auto_id_columns()))


# Токены

token_attributes_table = Table('token_attributes', shard_metadata, *eav_columns())


# индекс по access_token - для проверки токена
token_by_access_token_table = Table(
    'token_by_access_token',
    shard_metadata,
    Column('access_token', VARBINARY(100), primary_key=True, nullable=False),
    Column('id', BigInteger, nullable=False),
)

# индекс по uid и alias - для проверки токена по алиасу
token_by_alias_table = Table(
    'token_by_alias',
    shard_metadata,
    Column('uid', BigInteger, primary_key=True, autoincrement=False, nullable=False),
    Column('alias', VARBINARY(50), primary_key=True, nullable=False),
    Column('id', BigInteger, nullable=False),
)

# два в одном: индексы по уиду (для списка токенов, инвалидации при глогауте)
# и по параметрам (для проверки наличия токена для реюзания оного)
token_by_params_table = Table(
    'token_by_params',
    shard_metadata,
    Column('uid', BigInteger, primary_key=True, autoincrement=False, index=True, nullable=False),
    Column('client_id', Integer, primary_key=True, autoincrement=False, nullable=False),
    Column('scope_ids', String(600), primary_key=True, nullable=False),  # намеренно не VARBINARY - для LIKE-запросов
    Column('device_id', VARBINARY(50), primary_key=True, nullable=False),
    Column('id', BigInteger, nullable=False),
)


# Приложения

client_attributes_table = Table('client_attributes', central_metadata, *eav_columns(id_type=Integer))

# индекс по display_id (аналог access_token) - для получения приложения
client_by_display_id_table = Table(
    'client_by_display_id',
    central_metadata,
    Column('display_id', VARBINARY(50), primary_key=True, nullable=False),
    Column('id', Integer, nullable=False),
)

# индекс по уиду - для списка зареганных приложений
client_by_uid_table = Table(
    'client_by_uid',
    central_metadata,
    Column('uid', BigInteger, index=True, nullable=False),
    Column('id', Integer, unique=True, nullable=False),
)

# индекс по параметрам - для админки
client_by_params_table = Table(
    'client_by_params',
    central_metadata,
    Column('uid', BigInteger, nullable=False),
    Column('display_id', VARBINARY(50), index=True, nullable=False),
    Column('approval_status', Integer, index=True, nullable=False),
    Column('services', String(700), nullable=False),  # намеренно не VARBINARY - для LIKE-запросов
    Column('is_yandex', Boolean, index=True, nullable=False),
    Column('id', Integer, unique=True, nullable=False),
)

# индекс по владельцам - для списка приложений, к которым есть доступ
client_by_owner_table = Table(
    'client_by_owner',
    central_metadata,
    Column('owner_groups', VARBINARY(2000), nullable=True, index=True),
    Column('owner_uids', VARBINARY(2000), nullable=True, index=True),
    Column('uids', String(60000), nullable=True, index=True),  # намеренно не VARBINARY - для LIKE-запросов
    Column('modified_at', TIMESTAMP(), server_default=current_timestamp(), nullable=False),
    Column('id', Integer, unique=True, nullable=False),
)

# Реквесты

request_attributes_table = Table('request_attributes', central_metadata, *eav_columns())

# индекс по display_id (аналог access_token) - для получения реквеста
request_by_display_id_table = Table(
    'request_by_display_id',
    central_metadata,
    Column('display_id', VARBINARY(100), primary_key=True, nullable=False),
    Column('id', BigInteger, unique=True, nullable=False),
)

# индекс по коду, привязанному к приложению - для выдачи токена по коду
request_by_code_table = Table(
    'request_by_code',
    central_metadata,
    Column('code', VARBINARY(50), primary_key=True, autoincrement=False, nullable=False),
    Column('client_id', BigInteger, primary_key=True, autoincrement=False, nullable=False),
    Column('id', BigInteger, unique=True, nullable=False),
)


# индекс по уникальному коду - для выдачи токена по коду
request_by_unique_code_table = Table(
    'request_by_unique_code',
    central_metadata,
    Column('unique_code', VARBINARY(50), primary_key=True, autoincrement=False, nullable=False),
    Column('id', BigInteger, unique=True, nullable=False),
)
