# -*- coding: utf-8 -*-
from django.conf import settings
from django.utils.encoding import smart_text
from passport.backend.oauth.core.db.config import (
    get_scopes,
    localize_scope,
    localize_service,
)


DELETED_SCOPES_PREFIX = 'deleted:'


class Scope(object):
    @classmethod
    def by_id(cls, id):
        scope_data = get_scopes().data.get(str(id))
        if not scope_data:
            raise ValueError('Scope with id=%s not found' % id)
        return cls(
            id=int(id),
            keyword=scope_data['keyword'],
            default_title=scope_data.get('default_title', scope_data['keyword']),
            ttl=scope_data.get('ttl'),
            is_ttl_refreshable=bool(scope_data.get('is_ttl_refreshable')),
            requires_approval=bool(scope_data.get('requires_approval')),
            is_hidden=bool(scope_data.get('is_hidden')),
            visible_for_uids=scope_data.get('visible_for_uids') or [],
            visible_for_consumers=scope_data.get('visible_for_consumers') or [],
            has_xtoken_grant=bool(scope_data.get('has_xtoken_grant')),
            allowed_for_turboapps=bool(scope_data.get('allowed_for_turboapps', False)),
            tags=scope_data.get('tags') or [],
        )

    @classmethod
    def by_keyword(cls, keyword):
        if keyword.startswith(DELETED_SCOPES_PREFIX):
            raise ValueError('Unable to get deleted scope: %s' % keyword)
        scope_ids = [id for id, data in get_scopes().data.items() if data['keyword'] == keyword]
        if not scope_ids:
            raise ValueError('Scope with keyword=%s not found' % keyword)
        return cls.by_id(scope_ids[0])

    @classmethod
    def list(cls, show_hidden=True, uid=None):
        result = []
        for id in get_scopes().data:
            scope = cls.by_id(id)
            if scope.is_deleted:
                continue
            if scope.is_hidden:
                is_shown_to_uid = uid and uid in scope.visible_for_uids
                if not show_hidden and not is_shown_to_uid:
                    continue
            result.append(scope)
        return result

    def __init__(self, id, keyword, default_title, ttl=None, is_ttl_refreshable=False,
                 requires_approval=False, is_hidden=False, visible_for_uids=None, visible_for_consumers=None,
                 has_xtoken_grant=False, allowed_for_turboapps=False, tags=None):
        self.id = id
        self.keyword = keyword
        self.default_title = default_title
        self.ttl = ttl or None
        self.is_ttl_refreshable = is_ttl_refreshable
        self.requires_approval = requires_approval
        self.is_hidden = is_hidden
        self.visible_for_uids = set(visible_for_uids or [])
        self.visible_for_consumers = set(visible_for_consumers or [])
        self.has_xtoken_grant = has_xtoken_grant
        self.allowed_for_turboapps = allowed_for_turboapps
        self.service_name = keyword.split(':', 1)[0]
        self.tags = tags or []

    def get_title(self, language='ru', with_slug=False, short=False):
        title = localize_scope(self.keyword, language, default=self.default_title, short=short)
        if with_slug:
            title = '%s (%s)' % (title, self.keyword)
        return title

    def get_service_title(self, language='ru', with_slug=False):
        title = localize_service(self.service_name, language)
        if with_slug:
            title = '%s (%s)' % (title, self.service_name)
        return title

    @property
    def is_deleted(self):
        return self.keyword.startswith(DELETED_SCOPES_PREFIX)

    def __eq__(self, other):
        return isinstance(other, Scope) and self.id == other.id

    def __ne__(self, other):
        return not (self == other)

    def __str__(self):
        return smart_text(self.keyword)

    def __repr__(self):
        return '%r' % self.keyword

    def __hash__(self):
        return self.id


def list_services_with_titles(language='ru'):
    return {
        scope.service_name: scope.get_service_title(language)
        for scope in Scope.list(show_hidden=True)
    }


def _get_payment_auth_service(scopes):
    # У токена не может быть платёжных скоупов, принадлежащих разным сервисам. Поэтому отдаём первый же найденный сервис
    for s in scopes:
        if s.service_name in settings.SERVICES_REQUIRING_PAYMENT_AUTH:
            return s.service_name


def is_payment_auth_required(scopes):
    service = _get_payment_auth_service(scopes)
    return bool(service)


def get_payment_auth_app_ids(scopes):
    service = _get_payment_auth_service(scopes)
    return settings.PAYMENT_AUTH_SERVICE_TO_APP_IDS.get(service, [])


def has_x_token_scope(scopes):
    return any(scope.has_xtoken_grant for scope in scopes)
